"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SesPinpoint = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ses-pinpoint](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonpinpointemailservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SesPinpoint extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ses-pinpoint](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonpinpointemailservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ses-pinpoint';
        this.accessLevelList = {
            "Write": [
                "CreateConfigurationSet",
                "CreateConfigurationSetEventDestination",
                "CreateDedicatedIpPool",
                "CreateDeliverabilityTestReport",
                "CreateEmailIdentity",
                "DeleteConfigurationSet",
                "DeleteConfigurationSetEventDestination",
                "DeleteDedicatedIpPool",
                "DeleteEmailIdentity",
                "PutAccountDedicatedIpWarmupAttributes",
                "PutAccountSendingAttributes",
                "PutConfigurationSetDeliveryOptions",
                "PutConfigurationSetReputationOptions",
                "PutConfigurationSetSendingOptions",
                "PutConfigurationSetTrackingOptions",
                "PutDedicatedIpInPool",
                "PutDedicatedIpWarmupAttributes",
                "PutDeliverabilityDashboardOption",
                "PutEmailIdentityDkimAttributes",
                "PutEmailIdentityFeedbackAttributes",
                "PutEmailIdentityMailFromAttributes",
                "SendEmail",
                "UpdateConfigurationSetEventDestination"
            ],
            "Read": [
                "GetAccount",
                "GetBlacklistReports",
                "GetConfigurationSet",
                "GetConfigurationSetEventDestinations",
                "GetDedicatedIp",
                "GetDedicatedIps",
                "GetDeliverabilityDashboardOptions",
                "GetDeliverabilityTestReport",
                "GetDomainStatisticsReport",
                "GetEmailIdentity",
                "ListTagsForResource"
            ],
            "List": [
                "ListConfigurationSets",
                "ListDedicatedIpPools",
                "ListDeliverabilityTestReports",
                "ListEmailIdentities"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Create a configuration set. Configuration sets are groups of rules that you can apply to the emails you send using Amazon Pinpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateConfigurationSet.html
     */
    toCreateConfigurationSet() {
        this.to('ses-pinpoint:CreateConfigurationSet');
        return this;
    }
    /**
     * Create an event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateConfigurationSetEventDestination.html
     */
    toCreateConfigurationSetEventDestination() {
        this.to('ses-pinpoint:CreateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Create a new pool of dedicated IP addresses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateDedicatedIpPool.html
     */
    toCreateDedicatedIpPool() {
        this.to('ses-pinpoint:CreateDedicatedIpPool');
        return this;
    }
    /**
     * Create a new predictive inbox placement test.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateDeliverabilityTestReport.html
     */
    toCreateDeliverabilityTestReport() {
        this.to('ses-pinpoint:CreateDeliverabilityTestReport');
        return this;
    }
    /**
     * Verifies an email identity for use with Amazon Pinpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateEmailIdentity.html
     */
    toCreateEmailIdentity() {
        this.to('ses-pinpoint:CreateEmailIdentity');
        return this;
    }
    /**
     * Delete an existing configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteConfigurationSet.html
     */
    toDeleteConfigurationSet() {
        this.to('ses-pinpoint:DeleteConfigurationSet');
        return this;
    }
    /**
     * Delete an event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteConfigurationSetEventDestination.html
     */
    toDeleteConfigurationSetEventDestination() {
        this.to('ses-pinpoint:DeleteConfigurationSetEventDestination');
        return this;
    }
    /**
     * Delete a dedicated IP pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteDedicatedIpPool.html
     */
    toDeleteDedicatedIpPool() {
        this.to('ses-pinpoint:DeleteDedicatedIpPool');
        return this;
    }
    /**
     * Deletes an email identity that you previously verified for use with Amazon Pinpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteEmailIdentity.html
     */
    toDeleteEmailIdentity() {
        this.to('ses-pinpoint:DeleteEmailIdentity');
        return this;
    }
    /**
     * Obtain information about the email-sending status and capabilities
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetAccount.html
     */
    toGetAccount() {
        this.to('ses-pinpoint:GetAccount');
        return this;
    }
    /**
     * Retrieve a list of the blacklists that your dedicated IP addresses appear on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetBlacklistReports.html
     */
    toGetBlacklistReports() {
        this.to('ses-pinpoint:GetBlacklistReports');
        return this;
    }
    /**
     * Get information about an existing configuration set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetConfigurationSet.html
     */
    toGetConfigurationSet() {
        this.to('ses-pinpoint:GetConfigurationSet');
        return this;
    }
    /**
     * Retrieve a list of event destinations that are associated with a configuration set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetConfigurationSetEventDestinations.html
     */
    toGetConfigurationSetEventDestinations() {
        this.to('ses-pinpoint:GetConfigurationSetEventDestinations');
        return this;
    }
    /**
     * Get information about a dedicated IP address
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDedicatedIp.html
     */
    toGetDedicatedIp() {
        this.to('ses-pinpoint:GetDedicatedIp');
        return this;
    }
    /**
     * List the dedicated IP addresses that are associated with your Amazon Pinpoint account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDedicatedIps.html
     */
    toGetDedicatedIps() {
        this.to('ses-pinpoint:GetDedicatedIps');
        return this;
    }
    /**
     * Show the status of the Deliverability dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDeliverabilityDashboardOptions.html
     */
    toGetDeliverabilityDashboardOptions() {
        this.to('ses-pinpoint:GetDeliverabilityDashboardOptions');
        return this;
    }
    /**
     * Retrieve the results of a predictive inbox placement test
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDeliverabilityTestReport.html
     */
    toGetDeliverabilityTestReport() {
        this.to('ses-pinpoint:GetDeliverabilityTestReport');
        return this;
    }
    /**
     * Retrieve inbox placement and engagement rates for the domains that you use to send email
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDomainStatisticsReport.html
     */
    toGetDomainStatisticsReport() {
        this.to('ses-pinpoint:GetDomainStatisticsReport');
        return this;
    }
    /**
     * Provides information about a specific identity associated with your Amazon Pinpoint account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetEmailIdentity.html
     */
    toGetEmailIdentity() {
        this.to('ses-pinpoint:GetEmailIdentity');
        return this;
    }
    /**
     * List all of the configuration sets associated with your Amazon Pinpoint account in the current region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListConfigurationSets.html
     */
    toListConfigurationSets() {
        this.to('ses-pinpoint:ListConfigurationSets');
        return this;
    }
    /**
     * List all of the dedicated IP pools that exist in your Amazon Pinpoint account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListDedicatedIpPools.html
     */
    toListDedicatedIpPools() {
        this.to('ses-pinpoint:ListDedicatedIpPools');
        return this;
    }
    /**
     * Show a list of the predictive inbox placement tests that you've performed, regardless of their statuses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListDeliverabilityTestReports.html
     */
    toListDeliverabilityTestReports() {
        this.to('ses-pinpoint:ListDeliverabilityTestReports');
        return this;
    }
    /**
     * Returns a list of all of the email identities that are associated with your Amazon Pinpoint account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListEmailIdentities.html
     */
    toListEmailIdentities() {
        this.to('ses-pinpoint:ListEmailIdentities');
        return this;
    }
    /**
     * Retrieve a list of the tags (keys and values) that are associated with a specific resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('ses-pinpoint:ListTagsForResource');
        return this;
    }
    /**
     * Enable or disable the automatic warm-up feature for dedicated IP addresses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutAccountDedicatedIpWarmupAttributes.html
     */
    toPutAccountDedicatedIpWarmupAttributes() {
        this.to('ses-pinpoint:PutAccountDedicatedIpWarmupAttributes');
        return this;
    }
    /**
     * Enable or disable the ability of your account to send email
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutAccountSendingAttributes.html
     */
    toPutAccountSendingAttributes() {
        this.to('ses-pinpoint:PutAccountSendingAttributes');
        return this;
    }
    /**
     * Associate a configuration set with a dedicated IP pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetDeliveryOptions.html
     */
    toPutConfigurationSetDeliveryOptions() {
        this.to('ses-pinpoint:PutConfigurationSetDeliveryOptions');
        return this;
    }
    /**
     * Enable or disable collection of reputation metrics for emails that you send using a particular configuration set in a specific AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetReputationOptions.html
     */
    toPutConfigurationSetReputationOptions() {
        this.to('ses-pinpoint:PutConfigurationSetReputationOptions');
        return this;
    }
    /**
     * Enable or disable email sending for messages that use a particular configuration set in a specific AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetSendingOptions.html
     */
    toPutConfigurationSetSendingOptions() {
        this.to('ses-pinpoint:PutConfigurationSetSendingOptions');
        return this;
    }
    /**
     * Specify a custom domain to use for open and click tracking elements in email that you send using Amazon Pinpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetTrackingOptions.html
     */
    toPutConfigurationSetTrackingOptions() {
        this.to('ses-pinpoint:PutConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Move a dedicated IP address to an existing dedicated IP pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDedicatedIpInPool.html
     */
    toPutDedicatedIpInPool() {
        this.to('ses-pinpoint:PutDedicatedIpInPool');
        return this;
    }
    /**
     * Put Dedicated IP warm up attributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDedicatedIpWarmupAttributes.html
     */
    toPutDedicatedIpWarmupAttributes() {
        this.to('ses-pinpoint:PutDedicatedIpWarmupAttributes');
        return this;
    }
    /**
     * Enable or disable the Deliverability dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDeliverabilityDashboardOption.html
     */
    toPutDeliverabilityDashboardOption() {
        this.to('ses-pinpoint:PutDeliverabilityDashboardOption');
        return this;
    }
    /**
     * Used to enable or disable DKIM authentication for an email identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityDkimAttributes.html
     */
    toPutEmailIdentityDkimAttributes() {
        this.to('ses-pinpoint:PutEmailIdentityDkimAttributes');
        return this;
    }
    /**
     * Used to enable or disable feedback forwarding for an identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityFeedbackAttributes.html
     */
    toPutEmailIdentityFeedbackAttributes() {
        this.to('ses-pinpoint:PutEmailIdentityFeedbackAttributes');
        return this;
    }
    /**
     * Used to enable or disable the custom Mail-From domain configuration for an email identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityMailFromAttributes.html
     */
    toPutEmailIdentityMailFromAttributes() {
        this.to('ses-pinpoint:PutEmailIdentityMailFromAttributes');
        return this;
    }
    /**
     * Sends an email message
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_SendEmail.html
     */
    toSendEmail() {
        this.to('ses-pinpoint:SendEmail');
        return this;
    }
    /**
     * Add one or more tags (keys and values) to a specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('ses-pinpoint:TagResource');
        return this;
    }
    /**
     * Remove one or more tags (keys and values) from a specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('ses-pinpoint:UntagResource');
        return this;
    }
    /**
     * Update the configuration of an event destination for a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_UpdateConfigurationSetEventDestination.html
     */
    toUpdateConfigurationSetEventDestination() {
        this.to('ses-pinpoint:UpdateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Adds a resource of type configuration-set to the statement
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationSet(configurationSetName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dedicated-ip-pool to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DedicatedIp.html
     *
     * @param customVerificationEmailTemplateName - Identifier for the customVerificationEmailTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDedicatedIpPool(customVerificationEmailTemplateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:dedicated-ip-pool/${CustomVerificationEmailTemplateName}';
        arn = arn.replace('${CustomVerificationEmailTemplateName}', customVerificationEmailTemplateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deliverability-test-report to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeliverabilityTestReport.html
     *
     * @param customVerificationEmailTemplateName - Identifier for the customVerificationEmailTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeliverabilityTestReport(customVerificationEmailTemplateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:deliverability-test-report/${CustomVerificationEmailTemplateName}';
        arn = arn.replace('${CustomVerificationEmailTemplateName}', customVerificationEmailTemplateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-destination to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_EventDestination.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param eventDestinationName - Identifier for the eventDestinationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventDestination(configurationSetName, eventDestinationName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}:event-destination/${EventDestinationName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${EventDestinationName}', eventDestinationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_IdentityInfo.html
     *
     * @param identityName - Identifier for the identityName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentity(identityName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:identity/${IdentityName}';
        arn = arn.replace('${IdentityName}', identityName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The "Return-Path" address, which specifies where bounces and complaints are sent by email feedback forwarding.
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeedbackAddress(value, operator) {
        return this.if(`ses:FeedbackAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address of a message.
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromAddress(value, operator) {
        return this.if(`ses:FromAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address that is used as the display name of a message.
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromDisplayName(value, operator) {
        return this.if(`ses:FromDisplayName`, value, operator || 'StringLike');
    }
    /**
     * The recipient addresses of a message, which include the "To", "CC", and "BCC" addresses.
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipients(value, operator) {
        return this.if(`ses:Recipients`, value, operator || 'StringLike');
    }
}
exports.SesPinpoint = SesPinpoint;
//# sourceMappingURL=data:application/json;base64,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