"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Qldb = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [qldb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqldb.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Qldb extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [qldb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqldb.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'qldb';
        this.accessLevelList = {
            "Write": [
                "CancelJournalKinesisStream",
                "CreateLedger",
                "DeleteLedger",
                "ExecuteStatement",
                "ExportJournalToS3",
                "InsertSampleData",
                "SendCommand",
                "ShowCatalog",
                "StreamJournalToKinesis",
                "UpdateLedger"
            ],
            "Read": [
                "DescribeJournalKinesisStream",
                "DescribeJournalS3Export",
                "DescribeLedger",
                "GetBlock",
                "GetDigest",
                "GetRevision",
                "ListTagsForResource"
            ],
            "List": [
                "ListJournalKinesisStreamsForLedger",
                "ListJournalS3Exports",
                "ListJournalS3ExportsForLedger",
                "ListLedgers"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to cancel a journal kinesis stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_CancelJournalKinesisStream.html
     */
    toCancelJournalKinesisStream() {
        this.to('qldb:CancelJournalKinesisStream');
        return this;
    }
    /**
     * Grants permission to create a ledger
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_CreateLedger.html
     */
    toCreateLedger() {
        this.to('qldb:CreateLedger');
        return this;
    }
    /**
     * Grants permission to delete a ledger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DeleteLedger.html
     */
    toDeleteLedger() {
        this.to('qldb:DeleteLedger');
        return this;
    }
    /**
     * Grants permission to describe information about a journal kinesis stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeJournalKinesisStream.html
     */
    toDescribeJournalKinesisStream() {
        this.to('qldb:DescribeJournalKinesisStream');
        return this;
    }
    /**
     * Grants permission to describe information about a journal export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeJournalS3Export.html
     */
    toDescribeJournalS3Export() {
        this.to('qldb:DescribeJournalS3Export');
        return this;
    }
    /**
     * Grants permission to describe a ledger
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeLedger.html
     */
    toDescribeLedger() {
        this.to('qldb:DescribeLedger');
        return this;
    }
    /**
     * Grants permission to send commands to a ledger via the console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html
     */
    toExecuteStatement() {
        this.to('qldb:ExecuteStatement');
        return this;
    }
    /**
     * Grants permission to export journal contents to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ExportJournalToS3.html
     */
    toExportJournalToS3() {
        this.to('qldb:ExportJournalToS3');
        return this;
    }
    /**
     * Grants permission to retrieve a block from a ledger for a given BlockAddress
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetBlock.html
     */
    toGetBlock() {
        this.to('qldb:GetBlock');
        return this;
    }
    /**
     * Grants permission to retrieve a digest from a ledger for a given BlockAddress
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetDigest.html
     */
    toGetDigest() {
        this.to('qldb:GetDigest');
        return this;
    }
    /**
     * Grants permission to retrieve a revision for a given document ID and a given BlockAddress
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetRevision.html
     */
    toGetRevision() {
        this.to('qldb:GetRevision');
        return this;
    }
    /**
     * Grants permission to insert sample application data via the console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html
     */
    toInsertSampleData() {
        this.to('qldb:InsertSampleData');
        return this;
    }
    /**
     * Grants permission to list journal kinesis streams for a specified ledger
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalKinesisStreamsForLedger.html
     */
    toListJournalKinesisStreamsForLedger() {
        this.to('qldb:ListJournalKinesisStreamsForLedger');
        return this;
    }
    /**
     * Grants permission to list journal export jobs for all ledgers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalS3Exports.html
     */
    toListJournalS3Exports() {
        this.to('qldb:ListJournalS3Exports');
        return this;
    }
    /**
     * Grants permission to list journal export jobs for a specified ledger
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalS3ExportsForLedger.html
     */
    toListJournalS3ExportsForLedger() {
        this.to('qldb:ListJournalS3ExportsForLedger');
        return this;
    }
    /**
     * Grants permission to list existing ledgers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListLedgers.html
     */
    toListLedgers() {
        this.to('qldb:ListLedgers');
        return this;
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('qldb:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to send commands to a ledger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/Using.API.html
     */
    toSendCommand() {
        this.to('qldb:SendCommand');
        return this;
    }
    /**
     * Grants permission to view a ledger's catalog via the console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html
     */
    toShowCatalog() {
        this.to('qldb:ShowCatalog');
        return this;
    }
    /**
     * Grants permission to stream journal contents to a Kinesis Data Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_StreamJournalToKinesis.html
     */
    toStreamJournalToKinesis() {
        this.to('qldb:StreamJournalToKinesis');
        return this;
    }
    /**
     * Grants permission to add one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_TagResource.html
     */
    toTagResource() {
        this.to('qldb:TagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_UntagResource.html
     */
    toUntagResource() {
        this.to('qldb:UntagResource');
        return this;
    }
    /**
     * Grants permission to update properties on a ledger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_UpdateLedger.html
     */
    toUpdateLedger() {
        this.to('qldb:UpdateLedger');
        return this;
    }
    /**
     * Adds a resource of type ledger to the statement
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/what-is.html
     *
     * @param ledgerName - Identifier for the ledgerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLedger(ledgerName, account, region, partition) {
        var arn = 'arn:${Partition}:qldb:${Region}:${Account}:ledger/${LedgerName}';
        arn = arn.replace('${LedgerName}', ledgerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/what-is.html
     *
     * @param ledgerName - Identifier for the ledgerName.
     * @param streamId - Identifier for the streamId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStream(ledgerName, streamId, account, region, partition) {
        var arn = 'arn:${Partition}:qldb:${Region}:${Account}:stream/${LedgerName}/${StreamId}';
        arn = arn.replace('${LedgerName}', ledgerName);
        arn = arn.replace('${StreamId}', streamId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Qldb = Qldb;
//# sourceMappingURL=data:application/json;base64,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