"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [route53](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Route53 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [route53](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'route53';
        this.accessLevelList = {
            "Write": [
                "AssociateVPCWithHostedZone",
                "ChangeResourceRecordSets",
                "CreateHealthCheck",
                "CreateHostedZone",
                "CreateQueryLoggingConfig",
                "CreateReusableDelegationSet",
                "CreateTrafficPolicy",
                "CreateTrafficPolicyInstance",
                "CreateTrafficPolicyVersion",
                "CreateVPCAssociationAuthorization",
                "DeleteHealthCheck",
                "DeleteHostedZone",
                "DeleteQueryLoggingConfig",
                "DeleteReusableDelegationSet",
                "DeleteTrafficPolicy",
                "DeleteTrafficPolicyInstance",
                "DeleteVPCAssociationAuthorization",
                "DisassociateVPCFromHostedZone",
                "UpdateHealthCheck",
                "UpdateHostedZoneComment",
                "UpdateTrafficPolicyComment",
                "UpdateTrafficPolicyInstance"
            ],
            "Tagging": [
                "ChangeTagsForResource"
            ],
            "Read": [
                "GetAccountLimit",
                "GetHealthCheck",
                "GetHostedZoneLimit",
                "GetQueryLoggingConfig",
                "GetReusableDelegationSetLimit",
                "GetTrafficPolicy",
                "GetTrafficPolicyInstance",
                "GetTrafficPolicyInstanceCount",
                "TestDNSAnswer"
            ],
            "List": [
                "GetChange",
                "GetCheckerIpRanges",
                "GetGeoLocation",
                "GetHealthCheckCount",
                "GetHealthCheckLastFailureReason",
                "GetHealthCheckStatus",
                "GetHostedZone",
                "GetHostedZoneCount",
                "GetReusableDelegationSet",
                "ListGeoLocations",
                "ListHealthChecks",
                "ListHostedZones",
                "ListHostedZonesByName",
                "ListHostedZonesByVPC",
                "ListQueryLoggingConfigs",
                "ListResourceRecordSets",
                "ListReusableDelegationSets",
                "ListTagsForResource",
                "ListTagsForResources",
                "ListTrafficPolicies",
                "ListTrafficPolicyInstances",
                "ListTrafficPolicyInstancesByHostedZone",
                "ListTrafficPolicyInstancesByPolicy",
                "ListTrafficPolicyVersions",
                "ListVPCAssociationAuthorizations"
            ]
        };
    }
    /**
     * Grants permission to associate an additional Amazon VPC with a private hosted zone
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_AssociateVPCWithHostedZone.html
     */
    toAssociateVPCWithHostedZone() {
        this.to('route53:AssociateVPCWithHostedZone');
        return this;
    }
    /**
     * Grants permission to create, update, or delete a record, which contains authoritative DNS information for a specified domain or subdomain name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeResourceRecordSets.html
     */
    toChangeResourceRecordSets() {
        this.to('route53:ChangeResourceRecordSets');
        return this;
    }
    /**
     * Grants permission to add, edit, or delete tags for a health check or a hosted zone
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeTagsForResource.html
     */
    toChangeTagsForResource() {
        this.to('route53:ChangeTagsForResource');
        return this;
    }
    /**
     * Grants permission to create a new health check, which monitors the health and performance of your web applications, web servers, and other resources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHealthCheck.html
     */
    toCreateHealthCheck() {
        this.to('route53:CreateHealthCheck');
        return this;
    }
    /**
     * Grants permission to create a public hosted zone, which you use to specify how the Domain Name System (DNS) routes traffic on the Internet for a domain, such as example.com, and its subdomains
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHostedZone.html
     */
    toCreateHostedZone() {
        this.to('route53:CreateHostedZone');
        return this;
    }
    /**
     * Grants permission to create a configuration for DNS query logging
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateQueryLoggingConfig.html
     */
    toCreateQueryLoggingConfig() {
        this.to('route53:CreateQueryLoggingConfig');
        return this;
    }
    /**
     * Grants permission to create a delegation set (a group of four name servers) that can be reused by multiple hosted zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateReusableDelegationSet.html
     */
    toCreateReusableDelegationSet() {
        this.to('route53:CreateReusableDelegationSet');
        return this;
    }
    /**
     * Grants permission to create a traffic policy, which you use to create multiple DNS records for one domain name (such as example.com) or one subdomain name (such as www.example.com)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicy.html
     */
    toCreateTrafficPolicy() {
        this.to('route53:CreateTrafficPolicy');
        return this;
    }
    /**
     * Grants permission to create records in a specified hosted zone based on the settings in a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyInstance.html
     */
    toCreateTrafficPolicyInstance() {
        this.to('route53:CreateTrafficPolicyInstance');
        return this;
    }
    /**
     * Grants permission to create a new version of an existing traffic policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyVersion.html
     */
    toCreateTrafficPolicyVersion() {
        this.to('route53:CreateTrafficPolicyVersion');
        return this;
    }
    /**
     * Grants permission to authorize the AWS account that created a specified VPC to submit an AssociateVPCWithHostedZone request, which associates the VPC with a specified hosted zone that was created by a different account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateVPCAssociationAuthorization.html
     */
    toCreateVPCAssociationAuthorization() {
        this.to('route53:CreateVPCAssociationAuthorization');
        return this;
    }
    /**
     * Grants permission to delete a health check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHealthCheck.html
     */
    toDeleteHealthCheck() {
        this.to('route53:DeleteHealthCheck');
        return this;
    }
    /**
     * Grants permission to delete a hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHostedZone.html
     */
    toDeleteHostedZone() {
        this.to('route53:DeleteHostedZone');
        return this;
    }
    /**
     * Grants permission to delete a configuration for DNS query logging
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteQueryLoggingConfig.html
     */
    toDeleteQueryLoggingConfig() {
        this.to('route53:DeleteQueryLoggingConfig');
        return this;
    }
    /**
     * Grants permission to delete a reusable delegation set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteReusableDelegationSet.html
     */
    toDeleteReusableDelegationSet() {
        this.to('route53:DeleteReusableDelegationSet');
        return this;
    }
    /**
     * Grants permission to delete a traffic policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicy.html
     */
    toDeleteTrafficPolicy() {
        this.to('route53:DeleteTrafficPolicy');
        return this;
    }
    /**
     * Grants permission to delete a traffic policy instance and all the records that Route 53 created when you created the instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicyInstance.html
     */
    toDeleteTrafficPolicyInstance() {
        this.to('route53:DeleteTrafficPolicyInstance');
        return this;
    }
    /**
     * Grants permission to remove authorization for associating an Amazon Virtual Private Cloud with a Route 53 private hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteVPCAssociationAuthorization.html
     */
    toDeleteVPCAssociationAuthorization() {
        this.to('route53:DeleteVPCAssociationAuthorization');
        return this;
    }
    /**
     * Grants permission to disassociate an Amazon Virtual Private Cloud from a Route 53 private hosted zone
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DisassociateVPCFromHostedZone.html
     */
    toDisassociateVPCFromHostedZone() {
        this.to('route53:DisassociateVPCFromHostedZone');
        return this;
    }
    /**
     * Grants permission to get the specified limit for the current account, for example, the maximum number of health checks that you can create using the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetAccountLimit.html
     */
    toGetAccountLimit() {
        this.to('route53:GetAccountLimit');
        return this;
    }
    /**
     * Grants permission to get the current status of a request to create, update, or delete one or more records
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetChange.html
     */
    toGetChange() {
        this.to('route53:GetChange');
        return this;
    }
    /**
     * Grants permission to get a list of the IP ranges that are used by Route 53 health checkers to check the health of your resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetCheckerIpRanges.html
     */
    toGetCheckerIpRanges() {
        this.to('route53:GetCheckerIpRanges');
        return this;
    }
    /**
     * Grants permission to get information about whether a specified geographic location is supported for Route 53 geolocation records
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetGeoLocation.html
     */
    toGetGeoLocation() {
        this.to('route53:GetGeoLocation');
        return this;
    }
    /**
     * Grants permission to get information about a specified health check
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheck.html
     */
    toGetHealthCheck() {
        this.to('route53:GetHealthCheck');
        return this;
    }
    /**
     * Grants permission to get the number of health checks that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckCount.html
     */
    toGetHealthCheckCount() {
        this.to('route53:GetHealthCheckCount');
        return this;
    }
    /**
     * Grants permission to get the reason that a specified health check failed most recently
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckLastFailureReason.html
     */
    toGetHealthCheckLastFailureReason() {
        this.to('route53:GetHealthCheckLastFailureReason');
        return this;
    }
    /**
     * Grants permission to get the status of a specified health check
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckStatus.html
     */
    toGetHealthCheckStatus() {
        this.to('route53:GetHealthCheckStatus');
        return this;
    }
    /**
     * Grants permission to get information about a specified hosted zone including the four name servers that Route 53 assigned to the hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZone.html
     */
    toGetHostedZone() {
        this.to('route53:GetHostedZone');
        return this;
    }
    /**
     * Grants permission to get the number of hosted zones that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneCount.html
     */
    toGetHostedZoneCount() {
        this.to('route53:GetHostedZoneCount');
        return this;
    }
    /**
     * Grants permission to get the specified limit for a specified hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneLimit.html
     */
    toGetHostedZoneLimit() {
        this.to('route53:GetHostedZoneLimit');
        return this;
    }
    /**
     * Grants permission to get information about a specified configuration for DNS query logging
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetQueryLoggingConfig.html
     */
    toGetQueryLoggingConfig() {
        this.to('route53:GetQueryLoggingConfig');
        return this;
    }
    /**
     * Grants permission to get information about a specified reusable delegation set, including the four name servers that are assigned to the delegation set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSet.html
     */
    toGetReusableDelegationSet() {
        this.to('route53:GetReusableDelegationSet');
        return this;
    }
    /**
     * Grants permission to get the maximum number of hosted zones that you can associate with the specified reusable delegation set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSetLimit.html
     */
    toGetReusableDelegationSetLimit() {
        this.to('route53:GetReusableDelegationSetLimit');
        return this;
    }
    /**
     * Grants permission to get information about a specified traffic policy version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicy.html
     */
    toGetTrafficPolicy() {
        this.to('route53:GetTrafficPolicy');
        return this;
    }
    /**
     * Grants permission to get information about a specified traffic policy instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstance.html
     */
    toGetTrafficPolicyInstance() {
        this.to('route53:GetTrafficPolicyInstance');
        return this;
    }
    /**
     * Grants permission to get the number of traffic policy instances that are associated with the current AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstanceCount.html
     */
    toGetTrafficPolicyInstanceCount() {
        this.to('route53:GetTrafficPolicyInstanceCount');
        return this;
    }
    /**
     * Grants permission to get a list of geographic locations that Route 53 supports for geolocation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListGeoLocations.html
     */
    toListGeoLocations() {
        this.to('route53:ListGeoLocations');
        return this;
    }
    /**
     * Grants permission to get a list of the health checks that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHealthChecks.html
     */
    toListHealthChecks() {
        this.to('route53:ListHealthChecks');
        return this;
    }
    /**
     * Grants permission to get a list of the public and private hosted zones that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZones.html
     */
    toListHostedZones() {
        this.to('route53:ListHostedZones');
        return this;
    }
    /**
     * Grants permission to get a list of your hosted zones in lexicographic order. Hosted zones are sorted by name with the labels reversed, for example, com.example.www.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZonesByName.html
     */
    toListHostedZonesByName() {
        this.to('route53:ListHostedZonesByName');
        return this;
    }
    /**
     * Grants permission to get a list of all the private hosted zones that a specified VPC is associated with
     *
     * Access Level: List
     *
     * Dependent actions:
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZonesByVPC.html
     */
    toListHostedZonesByVPC() {
        this.to('route53:ListHostedZonesByVPC');
        return this;
    }
    /**
     * Grants permission to list the configurations for DNS query logging that are associated with the current AWS account or the configuration that is associated with a specified hosted zone.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListQueryLoggingConfigs.html
     */
    toListQueryLoggingConfigs() {
        this.to('route53:ListQueryLoggingConfigs');
        return this;
    }
    /**
     * Grants permission to list the records in a specified hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListResourceRecordSets.html
     */
    toListResourceRecordSets() {
        this.to('route53:ListResourceRecordSets');
        return this;
    }
    /**
     * Grants permission to list the reusable delegation sets that are associated with the current AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListReusableDelegationSets.html
     */
    toListReusableDelegationSets() {
        this.to('route53:ListReusableDelegationSets');
        return this;
    }
    /**
     * Grants permission to list tags for one health check or hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('route53:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list tags for up to 10 health checks or hosted zones
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResources.html
     */
    toListTagsForResources() {
        this.to('route53:ListTagsForResources');
        return this;
    }
    /**
     * Grants permission to get information about the latest version for every traffic policy that is associated with the current AWS account. Policies are listed in the order in which they were created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicies.html
     */
    toListTrafficPolicies() {
        this.to('route53:ListTrafficPolicies');
        return this;
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created by using the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstances.html
     */
    toListTrafficPolicyInstances() {
        this.to('route53:ListTrafficPolicyInstances');
        return this;
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created in a specified hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByHostedZone.html
     */
    toListTrafficPolicyInstancesByHostedZone() {
        this.to('route53:ListTrafficPolicyInstancesByHostedZone');
        return this;
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created using a specified traffic policy version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByPolicy.html
     */
    toListTrafficPolicyInstancesByPolicy() {
        this.to('route53:ListTrafficPolicyInstancesByPolicy');
        return this;
    }
    /**
     * Grants permission to get information about all the versions for a specified traffic policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyVersions.html
     */
    toListTrafficPolicyVersions() {
        this.to('route53:ListTrafficPolicyVersions');
        return this;
    }
    /**
     * Grants permission to get a list of the VPCs that were created by other accounts and that can be associated with a specified hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListVPCAssociationAuthorizations.html
     */
    toListVPCAssociationAuthorizations() {
        this.to('route53:ListVPCAssociationAuthorizations');
        return this;
    }
    /**
     * Grants permission to get the value that Route 53 returns in response to a DNS query for a specified record name and type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_TestDNSAnswer.html
     */
    toTestDNSAnswer() {
        this.to('route53:TestDNSAnswer');
        return this;
    }
    /**
     * Grants permission to update an existing health check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHealthCheck.html
     */
    toUpdateHealthCheck() {
        this.to('route53:UpdateHealthCheck');
        return this;
    }
    /**
     * Grants permission to update the comment for a specified hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHostedZoneComment.html
     */
    toUpdateHostedZoneComment() {
        this.to('route53:UpdateHostedZoneComment');
        return this;
    }
    /**
     * Grants permission to update the comment for a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyComment.html
     */
    toUpdateTrafficPolicyComment() {
        this.to('route53:UpdateTrafficPolicyComment');
        return this;
    }
    /**
     * Grants permission to update the records in a specified hosted zone that were created based on the settings in a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyInstance.html
     */
    toUpdateTrafficPolicyInstance() {
        this.to('route53:UpdateTrafficPolicyInstance');
        return this;
    }
    /**
     * Adds a resource of type change to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_Change.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChange(id, partition) {
        var arn = 'arn:${Partition}:route53:::change/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type delegationset to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-reusable-delegation-set
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDelegationset(id, partition) {
        var arn = 'arn:${Partition}:route53:::delegationset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type healthcheck to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-health-check
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHealthcheck(id, partition) {
        var arn = 'arn:${Partition}:route53:::healthcheck/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hostedzone to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-hosted-zone
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHostedzone(id, partition) {
        var arn = 'arn:${Partition}:route53:::hostedzone/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type trafficpolicy to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policies.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrafficpolicy(id, partition) {
        var arn = 'arn:${Partition}:route53:::trafficpolicy/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type trafficpolicyinstance to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policy-records.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrafficpolicyinstance(id, partition) {
        var arn = 'arn:${Partition}:route53:::trafficpolicyinstance/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type queryloggingconfig to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/query-logs.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onQueryloggingconfig(id, partition) {
        var arn = 'arn:${Partition}:route53:::queryloggingconfig/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html
     *
     * @param vpcId - Identifier for the vpcId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVpc(vpcId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc/${VpcId}';
        arn = arn.replace('${VpcId}', vpcId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Route53 = Route53;
//# sourceMappingURL=data:application/json;base64,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