"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53domains = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [route53domains](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53domains.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Route53domains extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [route53domains](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53domains.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'route53domains';
        this.accessLevelList = {
            "Read": [
                "CheckDomainAvailability",
                "GetContactReachabilityStatus",
                "GetDomainDetail",
                "GetDomainSuggestions",
                "GetOperationDetail",
                "ViewBilling"
            ],
            "Tagging": [
                "DeleteTagsForDomain",
                "UpdateTagsForDomain"
            ],
            "Write": [
                "DisableDomainAutoRenew",
                "DisableDomainTransferLock",
                "EnableDomainAutoRenew",
                "EnableDomainTransferLock",
                "RegisterDomain",
                "RenewDomain",
                "ResendContactReachabilityEmail",
                "RetrieveDomainAuthCode",
                "TransferDomain",
                "UpdateDomainContact",
                "UpdateDomainContactPrivacy",
                "UpdateDomainNameservers"
            ],
            "List": [
                "ListDomains",
                "ListOperations",
                "ListTagsForDomain"
            ]
        };
    }
    /**
     * Grants permission to check the availability of one domain name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_CheckDomainAvailability.html
     */
    toCheckDomainAvailability() {
        this.to('route53domains:CheckDomainAvailability');
        return this;
    }
    /**
     * Grants permission to delete the specified tags for a domain
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_DeleteTagsForDomain.html
     */
    toDeleteTagsForDomain() {
        this.to('route53domains:DeleteTagsForDomain');
        return this;
    }
    /**
     * Grants permission to configure Amazon Route 53 to automatically renew the specified domain before the domain registration expires
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_DisableDomainAutoRenew.html
     */
    toDisableDomainAutoRenew() {
        this.to('route53domains:DisableDomainAutoRenew');
        return this;
    }
    /**
     * Grants permission to remove the transfer lock on the domain (specifically the clientTransferProhibited status) to allow domain transfers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_DisableDomainTransferLock.html
     */
    toDisableDomainTransferLock() {
        this.to('route53domains:DisableDomainTransferLock');
        return this;
    }
    /**
     * Grants permission to configure Amazon Route 53 to automatically renew the specified domain before the domain registration expires
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_DisableDomainAutoRenew.html
     */
    toEnableDomainAutoRenew() {
        this.to('route53domains:EnableDomainAutoRenew');
        return this;
    }
    /**
     * Grants permission to set the transfer lock on the domain (specifically the clientTransferProhibited status) to prevent domain transfers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_EnableDomainTransferLock.html
     */
    toEnableDomainTransferLock() {
        this.to('route53domains:EnableDomainTransferLock');
        return this;
    }
    /**
     * For operations that require confirmation that the email address for the registrant contact is valid, such as registering a new domain, grants permission to get information about whether the registrant contact has responded
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetContactReachabilityStatus.html
     */
    toGetContactReachabilityStatus() {
        this.to('route53domains:GetContactReachabilityStatus');
        return this;
    }
    /**
     * Grants permission to get detailed information about a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetDomainDetail.html
     */
    toGetDomainDetail() {
        this.to('route53domains:GetDomainDetail');
        return this;
    }
    /**
     * Grants permission to get a list of suggested domain names given a string, which can either be a domain name or simply a word or phrase (without spaces)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetDomainSuggestions.html
     */
    toGetDomainSuggestions() {
        this.to('route53domains:GetDomainSuggestions');
        return this;
    }
    /**
     * Grants permission to get the current status of an operation that is not completed
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_GetOperationDetail.html
     */
    toGetOperationDetail() {
        this.to('route53domains:GetOperationDetail');
        return this;
    }
    /**
     * Grants permission to list all the domain names registered with Amazon Route 53 for the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_ListDomains.html
     */
    toListDomains() {
        this.to('route53domains:ListDomains');
        return this;
    }
    /**
     * Grants permission to list the operation IDs of operations that are not yet complete
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_ListOperations.html
     */
    toListOperations() {
        this.to('route53domains:ListOperations');
        return this;
    }
    /**
     * Grants permission to list all the tags that are associated with the specified domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_ListTagsForDomain.html
     */
    toListTagsForDomain() {
        this.to('route53domains:ListTagsForDomain');
        return this;
    }
    /**
     * Grants permission to register domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_RegisterDomain.html
     */
    toRegisterDomain() {
        this.to('route53domains:RegisterDomain');
        return this;
    }
    /**
     * Grants permission to renew domains for the specified number of years
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_RenewDomain.html
     */
    toRenewDomain() {
        this.to('route53domains:RenewDomain');
        return this;
    }
    /**
     * For operations that require confirmation that the email address for the registrant contact is valid, such as registering a new domain, grants permission to resend the confirmation email to the current email address for the registrant contact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_ResendContactReachabilityEmail.html
     */
    toResendContactReachabilityEmail() {
        this.to('route53domains:ResendContactReachabilityEmail');
        return this;
    }
    /**
     * Grants permission to get the AuthCode for the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_RetrieveDomainAuthCode.html
     */
    toRetrieveDomainAuthCode() {
        this.to('route53domains:RetrieveDomainAuthCode');
        return this;
    }
    /**
     * Grants permission to transfer a domain from another registrar to Amazon Route 53
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_TransferDomain.html
     */
    toTransferDomain() {
        this.to('route53domains:TransferDomain');
        return this;
    }
    /**
     * Grants permission to update the contact information for domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_UpdateDomainContact.html
     */
    toUpdateDomainContact() {
        this.to('route53domains:UpdateDomainContact');
        return this;
    }
    /**
     * Grants permission to update the domain contact privacy setting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_UpdateDomainContactPrivacy.html
     */
    toUpdateDomainContactPrivacy() {
        this.to('route53domains:UpdateDomainContactPrivacy');
        return this;
    }
    /**
     * Grants permission to replace the current set of name servers for a domain with the specified set of name servers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_UpdateDomainNameservers.html
     */
    toUpdateDomainNameservers() {
        this.to('route53domains:UpdateDomainNameservers');
        return this;
    }
    /**
     * Grants permission to add or update tags for a specified domain
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_UpdateTagsForDomain.html
     */
    toUpdateTagsForDomain() {
        this.to('route53domains:UpdateTagsForDomain');
        return this;
    }
    /**
     * Grants permission to get all the domain-related billing records for the current AWS account for a specified period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_domains_ViewBilling.html
     */
    toViewBilling() {
        this.to('route53domains:ViewBilling');
        return this;
    }
}
exports.Route53domains = Route53domains;
//# sourceMappingURL=data:application/json;base64,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