"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53resolver = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [route53resolver](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53resolver.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Route53resolver extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [route53resolver](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53resolver.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'route53resolver';
        this.accessLevelList = {
            "Write": [
                "AssociateResolverEndpointIpAddress",
                "AssociateResolverQueryLogConfig",
                "AssociateResolverRule",
                "CreateResolverEndpoint",
                "CreateResolverQueryLogConfig",
                "CreateResolverRule",
                "DeleteResolverEndpoint",
                "DeleteResolverQueryLogConfig",
                "DeleteResolverRule",
                "DisassociateResolverEndpointIpAddress",
                "DisassociateResolverQueryLogConfig",
                "DisassociateResolverRule",
                "PutResolverQueryLogConfigPolicy",
                "PutResolverRulePolicy",
                "UpdateResolverEndpoint",
                "UpdateResolverRule"
            ],
            "Read": [
                "GetResolverEndpoint",
                "GetResolverQueryLogConfig",
                "GetResolverQueryLogConfigAssociation",
                "GetResolverQueryLogConfigPolicy",
                "GetResolverRule",
                "GetResolverRuleAssociation",
                "GetResolverRulePolicy",
                "ListTagsForResource"
            ],
            "List": [
                "ListResolverEndpointIpAddresses",
                "ListResolverEndpoints",
                "ListResolverQueryLogConfigAssociations",
                "ListResolverQueryLogConfigs",
                "ListResolverRuleAssociations",
                "ListResolverRules"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to associate a specified IP address with a Resolver endpoint. This is an IP address that DNS queries pass through on the way to your network (outbound) or your VPCs (inbound)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverEndpointIpAddress.html
     */
    toAssociateResolverEndpointIpAddress() {
        this.to('route53resolver:AssociateResolverEndpointIpAddress');
        return this;
    }
    /**
     * Grants permission to associate an Amazon VPC with a specified query logging configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverQueryLogConfig.html
     */
    toAssociateResolverQueryLogConfig() {
        this.to('route53resolver:AssociateResolverQueryLogConfig');
        return this;
    }
    /**
     * Grants permission to associate a specified Resolver rule with a specified VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverRule.html
     */
    toAssociateResolverRule() {
        this.to('route53resolver:AssociateResolverRule');
        return this;
    }
    /**
     * Grants permission to create a Resolver endpoint. There are two types of Resolver endpoints, inbound and outbound
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html
     */
    toCreateResolverEndpoint() {
        this.to('route53resolver:CreateResolverEndpoint');
        return this;
    }
    /**
     * Grants permission to create a Resolver query logging configuration, which defines where you want Resolver to save DNS query logs that originate in your VPCs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverQueryLogConfig.html
     */
    toCreateResolverQueryLogConfig() {
        this.to('route53resolver:CreateResolverQueryLogConfig');
        return this;
    }
    /**
     * For DNS queries that originate in your VPC, grants permission to define how to route the queries out of the VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverRule.html
     */
    toCreateResolverRule() {
        this.to('route53resolver:CreateResolverRule');
        return this;
    }
    /**
     * Grants permission to delete a Resolver endpoint. The effect of deleting a Resolver endpoint depends on whether it's an inbound or an outbound endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DeleteResolverEndpoint.html
     */
    toDeleteResolverEndpoint() {
        this.to('route53resolver:DeleteResolverEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a Resolver query logging configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DeleteResolverQueryLogConfig.html
     */
    toDeleteResolverQueryLogConfig() {
        this.to('route53resolver:DeleteResolverQueryLogConfig');
        return this;
    }
    /**
     * Grants permission to delete a Resolver rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DeleteResolverRule.html
     */
    toDeleteResolverRule() {
        this.to('route53resolver:DeleteResolverRule');
        return this;
    }
    /**
     * Grants permission to remove a specified IP address from a Resolver endpoint. This is an IP address that DNS queries pass through on the way to your network (outbound) or your VPCs (inbound)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverEndpointIpAddress.html
     */
    toDisassociateResolverEndpointIpAddress() {
        this.to('route53resolver:DisassociateResolverEndpointIpAddress');
        return this;
    }
    /**
     * Grants permission to remove the association between a specified Resolver query logging configuration and a specified VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverQueryLogConfig.html
     */
    toDisassociateResolverQueryLogConfig() {
        this.to('route53resolver:DisassociateResolverQueryLogConfig');
        return this;
    }
    /**
     * Grants permission to remove the association between a specified Resolver rule and a specified VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverRule.html
     */
    toDisassociateResolverRule() {
        this.to('route53resolver:DisassociateResolverRule');
        return this;
    }
    /**
     * Grants permission to get information about a specified Resolver endpoint, such as whether it's an inbound or an outbound endpoint, and the IP addresses in your VPC that DNS queries are forwarded to on the way into or out of your VPC
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverEndpoint.html
     */
    toGetResolverEndpoint() {
        this.to('route53resolver:GetResolverEndpoint');
        return this;
    }
    /**
     * Grants permission to get information about a specified Resolver query logging configuration, such as the number of VPCs that the configuration is logging queries for and the location that logs are sent to
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverQueryLogConfig.html
     */
    toGetResolverQueryLogConfig() {
        this.to('route53resolver:GetResolverQueryLogConfig');
        return this;
    }
    /**
     * Grants permission to get information about a specified association between a Resolver query logging configuration and an Amazon VPC. When you associate a VPC with a query logging configuration, Resolver logs DNS queries that originate in that VPC
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverQueryLogConfigAssociation.html
     */
    toGetResolverQueryLogConfigAssociation() {
        this.to('route53resolver:GetResolverQueryLogConfigAssociation');
        return this;
    }
    /**
     * Grants permission to get information about a specified Resolver query logging policy, which specifies the Resolver query logging operations and resources that you want to allow another AWS account to use
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverQueryLogConfigPolicy.html
     */
    toGetResolverQueryLogConfigPolicy() {
        this.to('route53resolver:GetResolverQueryLogConfigPolicy');
        return this;
    }
    /**
     * Grants permission to get information about a specified Resolver rule, such as the domain name that the rule forwards DNS queries for and the IP address that queries are forwarded to.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverRule.html
     */
    toGetResolverRule() {
        this.to('route53resolver:GetResolverRule');
        return this;
    }
    /**
     * Grants permission to get information about an association between a specified Resolver rule and a VPC
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverRuleAssociation.html
     */
    toGetResolverRuleAssociation() {
        this.to('route53resolver:GetResolverRuleAssociation');
        return this;
    }
    /**
     * Grants permission to get information about a Resolver rule policy, which specifies the Resolver operations and resources that you want to allow another AWS account to use
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverRulePolicy.html
     */
    toGetResolverRulePolicy() {
        this.to('route53resolver:GetResolverRulePolicy');
        return this;
    }
    /**
     * For a specified Resolver endpoint, grants permission to list the IP addresses that DNS queries pass through on the way to your network (outbound) or your VPCs (inbound)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverEndpointIpAddresses.html
     */
    toListResolverEndpointIpAddresses() {
        this.to('route53resolver:ListResolverEndpointIpAddresses');
        return this;
    }
    /**
     * Grants permission to list all the Resolver endpoints that were created using the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverEndpoints.html
     */
    toListResolverEndpoints() {
        this.to('route53resolver:ListResolverEndpoints');
        return this;
    }
    /**
     * Grants permission to list information about associations between Amazon VPCs and query logging configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigAssociations.html
     */
    toListResolverQueryLogConfigAssociations() {
        this.to('route53resolver:ListResolverQueryLogConfigAssociations');
        return this;
    }
    /**
     * Grants permission to list information about the specified query logging configurations, which define where you want Resolver to save DNS query logs and specify the VPCs that you want to log queries for
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigs.html
     */
    toListResolverQueryLogConfigs() {
        this.to('route53resolver:ListResolverQueryLogConfigs');
        return this;
    }
    /**
     * Grants permission to list the associations that were created between Resolver rules and VPCs using the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRuleAssociations.html
     */
    toListResolverRuleAssociations() {
        this.to('route53resolver:ListResolverRuleAssociations');
        return this;
    }
    /**
     * Grants permission to list the Resolver rules that were created using the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRules.html
     */
    toListResolverRules() {
        this.to('route53resolver:ListResolverRules');
        return this;
    }
    /**
     * Grants permission to list the tags that you associated with the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('route53resolver:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to specify an AWS account that you want to share a query logging configuration with, the query logging configuration that you want to share, and the operations that you want the account to be able to perform on the configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_PutResolverQueryLogConfigPolicy.html
     */
    toPutResolverQueryLogConfigPolicy() {
        this.to('route53resolver:PutResolverQueryLogConfigPolicy');
        return this;
    }
    /**
     * Grants permission to specify an AWS account that you want to share rules with, the Resolver rules that you want to share, and the operations that you want the account to be able to perform on those rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_PutResolverRulePolicy.html
     */
    toPutResolverRulePolicy() {
        this.to('route53resolver:PutResolverRulePolicy');
        return this;
    }
    /**
     * Grants permission to add one or more tags to a specified resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_TagResource.html
     */
    toTagResource() {
        this.to('route53resolver:TagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from a specified resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_UntagResource.html
     */
    toUntagResource() {
        this.to('route53resolver:UntagResource');
        return this;
    }
    /**
     * Grants permission to update selected settings for an inbound or an outbound Resolver endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_UpdateResolverEndpoint.html
     */
    toUpdateResolverEndpoint() {
        this.to('route53resolver:UpdateResolverEndpoint');
        return this;
    }
    /**
     * Grants permission to update settings for a specified Resolver rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_UpdateResolverRule.html
     */
    toUpdateResolverRule() {
        this.to('route53resolver:UpdateResolverRule');
        return this;
    }
    /**
     * Adds a resource of type resolver-query-log-config to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/access-control-overview.html#access-control-resources
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResolverQueryLogConfig(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:route53resolver:${Region}:${Account}:resolver-query-log-config/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resolver-rule to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/access-control-overview.html#access-control-resources
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResolverRule(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:route53resolver:${Region}:${Account}:resolver-rule/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resolver-endpoint to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/access-control-overview.html#access-control-resources
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResolverEndpoint(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:route53resolver:${Region}:${Account}:resolver-endpoint/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Route53resolver = Route53resolver;
//# sourceMappingURL=data:application/json;base64,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