"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Serverlessrepo = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [serverlessrepo](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsserverlessapplicationrepository.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Serverlessrepo extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [serverlessrepo](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsserverlessapplicationrepository.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'serverlessrepo';
        this.accessLevelList = {
            "Write": [
                "CreateApplication",
                "CreateApplicationVersion",
                "CreateCloudFormationChangeSet",
                "CreateCloudFormationTemplate",
                "DeleteApplication",
                "PutApplicationPolicy",
                "UnshareApplication",
                "UpdateApplication"
            ],
            "Read": [
                "GetApplication",
                "GetApplicationPolicy",
                "GetCloudFormationTemplate",
                "SearchApplications"
            ],
            "List": [
                "ListApplicationDependencies",
                "ListApplicationVersions",
                "ListApplications"
            ]
        };
    }
    /**
     * Creates an application, optionally including an AWS SAM file to create the first application version in the same call.
     *
     * Access Level: Write
     */
    toCreateApplication() {
        this.to('serverlessrepo:CreateApplication');
        return this;
    }
    /**
     * Creates an application version.
     *
     * Access Level: Write
     */
    toCreateApplicationVersion() {
        this.to('serverlessrepo:CreateApplicationVersion');
        return this;
    }
    /**
     * Creates an AWS CloudFormation ChangeSet for the given application.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationType()
     */
    toCreateCloudFormationChangeSet() {
        this.to('serverlessrepo:CreateCloudFormationChangeSet');
        return this;
    }
    /**
     * Creates an AWS CloudFormation template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationType()
     */
    toCreateCloudFormationTemplate() {
        this.to('serverlessrepo:CreateCloudFormationTemplate');
        return this;
    }
    /**
     * Deletes the specified application
     *
     * Access Level: Write
     */
    toDeleteApplication() {
        this.to('serverlessrepo:DeleteApplication');
        return this;
    }
    /**
     * Gets the specified application.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationType()
     */
    toGetApplication() {
        this.to('serverlessrepo:GetApplication');
        return this;
    }
    /**
     * Gets the policy for the specified application.
     *
     * Access Level: Read
     */
    toGetApplicationPolicy() {
        this.to('serverlessrepo:GetApplicationPolicy');
        return this;
    }
    /**
     * Gets the specified AWS CloudFormation template
     *
     * Access Level: Read
     */
    toGetCloudFormationTemplate() {
        this.to('serverlessrepo:GetCloudFormationTemplate');
        return this;
    }
    /**
     * Retrieves the list of applications nested in the containing application
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationType()
     */
    toListApplicationDependencies() {
        this.to('serverlessrepo:ListApplicationDependencies');
        return this;
    }
    /**
     * Lists versions for the specified application owned by the requester.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationType()
     */
    toListApplicationVersions() {
        this.to('serverlessrepo:ListApplicationVersions');
        return this;
    }
    /**
     * Lists applications owned by the requester.
     *
     * Access Level: List
     */
    toListApplications() {
        this.to('serverlessrepo:ListApplications');
        return this;
    }
    /**
     * Puts the policy for the specified application.
     *
     * Access Level: Write
     */
    toPutApplicationPolicy() {
        this.to('serverlessrepo:PutApplicationPolicy');
        return this;
    }
    /**
     * Gets all applications authorized for this user
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationType()
     */
    toSearchApplications() {
        this.to('serverlessrepo:SearchApplications');
        return this;
    }
    /**
     * Unshares the specified application
     *
     * Access Level: Write
     */
    toUnshareApplication() {
        this.to('serverlessrepo:UnshareApplication');
        return this;
    }
    /**
     * Updates meta-data of the application
     *
     * Access Level: Write
     */
    toUpdateApplication() {
        this.to('serverlessrepo:UpdateApplication');
        return this;
    }
    /**
     * Adds a resource of type applications to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onApplications(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:serverlessrepo:${Region}:${Account}:applications/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Application type
     *
     * Applies to actions:
     * - .toCreateCloudFormationChangeSet()
     * - .toCreateCloudFormationTemplate()
     * - .toGetApplication()
     * - .toListApplicationDependencies()
     * - .toListApplicationVersions()
     * - .toSearchApplications()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApplicationType(value, operator) {
        return this.if(`serverlessrepo:applicationType`, value, operator || 'StringLike');
    }
}
exports.Serverlessrepo = Serverlessrepo;
//# sourceMappingURL=data:application/json;base64,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