"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sms = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsservermigrationservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sms extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsservermigrationservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sms';
        this.accessLevelList = {
            "Write": [
                "CreateApp",
                "CreateReplicationJob",
                "DeleteApp",
                "DeleteAppLaunchConfiguration",
                "DeleteAppReplicationConfiguration",
                "DeleteAppValidationConfiguration",
                "DeleteReplicationJob",
                "DeleteServerCatalog",
                "DisassociateConnector",
                "GenerateChangeSet",
                "GenerateTemplate",
                "ImportAppCatalog",
                "ImportServerCatalog",
                "LaunchApp",
                "NotifyAppValidationOutput",
                "PutAppLaunchConfiguration",
                "PutAppReplicationConfiguration",
                "PutAppValidationConfiguration",
                "SendMessage",
                "StartAppReplication",
                "StartOnDemandAppReplication",
                "StartOnDemandReplicationRun",
                "StopAppReplication",
                "TerminateApp",
                "UpdateApp",
                "UpdateReplicationJob"
            ],
            "Read": [
                "GetApp",
                "GetAppLaunchConfiguration",
                "GetAppReplicationConfiguration",
                "GetAppValidationConfiguration",
                "GetAppValidationOutput",
                "GetConnectors",
                "GetMessages",
                "GetReplicationJobs",
                "GetReplicationRuns",
                "GetServers"
            ],
            "List": [
                "ListApps"
            ]
        };
    }
    /**
     * Grants permission to create an application configuration to migrate on-premise application onto AWS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_CreateApp.html
     */
    toCreateApp() {
        this.to('sms:CreateApp');
        return this;
    }
    /**
     * Grants permission to create a job to migrate on-premise server onto AWS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_CreateReplicationJob.html
     */
    toCreateReplicationJob() {
        this.to('sms:CreateReplicationJob');
        return this;
    }
    /**
     * Grants permission to delete an existing application configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteApp.html
     */
    toDeleteApp() {
        this.to('sms:DeleteApp');
        return this;
    }
    /**
     * Grants permission to delete launch configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteAppLaunchConfiguration.html
     */
    toDeleteAppLaunchConfiguration() {
        this.to('sms:DeleteAppLaunchConfiguration');
        return this;
    }
    /**
     * Grants permission to delete replication configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteAppReplicationConfiguration.html
     */
    toDeleteAppReplicationConfiguration() {
        this.to('sms:DeleteAppReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to delete validation configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteAppValidationConfiguration.html
     */
    toDeleteAppValidationConfiguration() {
        this.to('sms:DeleteAppValidationConfiguration');
        return this;
    }
    /**
     * Grants permission to delete an existing job to migrate on-premise server onto AWS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteReplicationJob.html
     */
    toDeleteReplicationJob() {
        this.to('sms:DeleteReplicationJob');
        return this;
    }
    /**
     * Grants permission to delete the complete list of on-premise servers gathered into AWS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DeleteServerCatalog.html
     */
    toDeleteServerCatalog() {
        this.to('sms:DeleteServerCatalog');
        return this;
    }
    /**
     * Grants permission to disassociate a connector that has been associated
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_DisassociateConnector.html
     */
    toDisassociateConnector() {
        this.to('sms:DisassociateConnector');
        return this;
    }
    /**
     * Grants permission to generate a changeSet for the CloudFormation stack of an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GenerateChangeSet.html
     */
    toGenerateChangeSet() {
        this.to('sms:GenerateChangeSet');
        return this;
    }
    /**
     * Grants permission to generate a CloudFormation template for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GenerateTemplate.html
     */
    toGenerateTemplate() {
        this.to('sms:GenerateTemplate');
        return this;
    }
    /**
     * Grants permission to get the configuration and statuses for an existing application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetApp.html
     */
    toGetApp() {
        this.to('sms:GetApp');
        return this;
    }
    /**
     * Grants permission to get launch configuration for an existing application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppLaunchConfiguration.html
     */
    toGetAppLaunchConfiguration() {
        this.to('sms:GetAppLaunchConfiguration');
        return this;
    }
    /**
     * Grants permission to get replication configuration for an existing application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppReplicationConfiguration.html
     */
    toGetAppReplicationConfiguration() {
        this.to('sms:GetAppReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to get validation configuration for an existing application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppValidationConfiguration.html
     */
    toGetAppValidationConfiguration() {
        this.to('sms:GetAppValidationConfiguration');
        return this;
    }
    /**
     * Grants permission to get notification sent from application validation script.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetAppValidationOutput.html
     */
    toGetAppValidationOutput() {
        this.to('sms:GetAppValidationOutput');
        return this;
    }
    /**
     * Grants permission to get all connectors that have been associated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetConnectors.html
     */
    toGetConnectors() {
        this.to('sms:GetConnectors');
        return this;
    }
    /**
     * Grants permission to gets messages from AWS Server Migration Service to Server Migration Connector
     *
     * Access Level: Read
     */
    toGetMessages() {
        this.to('sms:GetMessages');
        return this;
    }
    /**
     * Grants permission to get all existing jobs to migrate on-premise servers onto AWS
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetReplicationJobs.html
     */
    toGetReplicationJobs() {
        this.to('sms:GetReplicationJobs');
        return this;
    }
    /**
     * Grants permission to get all runs for an existing job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetReplicationRuns.html
     */
    toGetReplicationRuns() {
        this.to('sms:GetReplicationRuns');
        return this;
    }
    /**
     * Grants permission to get all servers that have been imported
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_GetServers.html
     */
    toGetServers() {
        this.to('sms:GetServers');
        return this;
    }
    /**
     * Grants permission to import application catalog from AWS Application Discovery Service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_ImportAppCatalog.html
     */
    toImportAppCatalog() {
        this.to('sms:ImportAppCatalog');
        return this;
    }
    /**
     * Grants permission to gather a complete list of on-premise servers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_ImportServerCatalog.html
     */
    toImportServerCatalog() {
        this.to('sms:ImportServerCatalog');
        return this;
    }
    /**
     * Grants permission to create and launch a CloudFormation stack for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_LaunchApp.html
     */
    toLaunchApp() {
        this.to('sms:LaunchApp');
        return this;
    }
    /**
     * Grants permission to get a list of summaries for existing applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_ListAppss.html
     */
    toListApps() {
        this.to('sms:ListApps');
        return this;
    }
    /**
     * Grants permission to send notification for application validation script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_NotifyAppValidationOutput.html
     */
    toNotifyAppValidationOutput() {
        this.to('sms:NotifyAppValidationOutput');
        return this;
    }
    /**
     * Grants permission to create or update launch configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_PutAppLaunchConfiguration.html
     */
    toPutAppLaunchConfiguration() {
        this.to('sms:PutAppLaunchConfiguration');
        return this;
    }
    /**
     * Grants permission to create or update replication configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_PutAppReplicationConfiguration.html
     */
    toPutAppReplicationConfiguration() {
        this.to('sms:PutAppReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to put validation configuration for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_PutAppValidationConfiguration.html
     */
    toPutAppValidationConfiguration() {
        this.to('sms:PutAppValidationConfiguration');
        return this;
    }
    /**
     * Grants permission to send message from Server Migration Connector to AWS Server Migration Service
     *
     * Access Level: Write
     */
    toSendMessage() {
        this.to('sms:SendMessage');
        return this;
    }
    /**
     * Grants permission to create and start replication jobs for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StartAppReplication.html
     */
    toStartAppReplication() {
        this.to('sms:StartAppReplication');
        return this;
    }
    /**
     * Grants permission to start a replication run for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StartOnDemandAppReplication.html
     */
    toStartOnDemandAppReplication() {
        this.to('sms:StartOnDemandAppReplication');
        return this;
    }
    /**
     * Grants permission to start a replication run for an existing replication job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StartOnDemandReplicationRun.html
     */
    toStartOnDemandReplicationRun() {
        this.to('sms:StartOnDemandReplicationRun');
        return this;
    }
    /**
     * Grants permission to stop and delete replication jobs for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_StopAppReplication.html
     */
    toStopAppReplication() {
        this.to('sms:StopAppReplication');
        return this;
    }
    /**
     * Grants permission to terminate the CloudFormation stack for an existing application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_TerminateApp.html
     */
    toTerminateApp() {
        this.to('sms:TerminateApp');
        return this;
    }
    /**
     * Grants permission to update an existing application configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_UpdateApp.html
     */
    toUpdateApp() {
        this.to('sms:UpdateApp');
        return this;
    }
    /**
     * Grants permission to update an existing job to migrate on-premise server onto AWS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/server-migration-service/latest/APIReference/API_UpdateReplicationJob.html
     */
    toUpdateReplicationJob() {
        this.to('sms:UpdateReplicationJob');
        return this;
    }
}
exports.Sms = Sms;
//# sourceMappingURL=data:application/json;base64,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