"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Storagegateway = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [storagegateway](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonstoragegateway.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Storagegateway extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [storagegateway](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonstoragegateway.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'storagegateway';
        this.accessLevelList = {
            "Write": [
                "ActivateGateway",
                "AddCache",
                "AddUploadBuffer",
                "AddWorkingStorage",
                "AssignTapePool",
                "AttachVolume",
                "BypassGovernanceRetention",
                "CancelArchival",
                "CancelRetrieval",
                "CreateCachediSCSIVolume",
                "CreateNFSFileShare",
                "CreateSMBFileShare",
                "CreateSnapshot",
                "CreateSnapshotFromVolumeRecoveryPoint",
                "CreateStorediSCSIVolume",
                "CreateTapePool",
                "CreateTapeWithBarcode",
                "CreateTapes",
                "DeleteAutomaticTapeCreationPolicy",
                "DeleteBandwidthRateLimit",
                "DeleteChapCredentials",
                "DeleteFileShare",
                "DeleteGateway",
                "DeleteSnapshotSchedule",
                "DeleteTape",
                "DeleteTapeArchive",
                "DeleteTapePool",
                "DeleteVolume",
                "DetachVolume",
                "DisableGateway",
                "JoinDomain",
                "NotifyWhenUploaded",
                "RefreshCache",
                "ResetCache",
                "RetrieveTapeArchive",
                "RetrieveTapeRecoveryPoint",
                "SetLocalConsolePassword",
                "SetSMBGuestPassword",
                "ShutdownGateway",
                "StartGateway",
                "UpdateAutomaticTapeCreationPolicy",
                "UpdateBandwidthRateLimit",
                "UpdateChapCredentials",
                "UpdateGatewayInformation",
                "UpdateGatewaySoftwareNow",
                "UpdateMaintenanceStartTime",
                "UpdateNFSFileShare",
                "UpdateSMBFileShare",
                "UpdateSnapshotSchedule",
                "UpdateVTLDeviceType"
            ],
            "Tagging": [
                "AddTagsToResource",
                "RemoveTagsFromResource"
            ],
            "Read": [
                "DescribeBandwidthRateLimit",
                "DescribeCache",
                "DescribeCachediSCSIVolumes",
                "DescribeChapCredentials",
                "DescribeGatewayInformation",
                "DescribeMaintenanceStartTime",
                "DescribeNFSFileShares",
                "DescribeSMBFileShares",
                "DescribeSMBSettings",
                "DescribeSnapshotSchedule",
                "DescribeStorediSCSIVolumes",
                "DescribeTapeArchives",
                "DescribeTapeRecoveryPoints",
                "DescribeTapes",
                "DescribeUploadBuffer",
                "DescribeVTLDevices",
                "DescribeWorkingStorage"
            ],
            "List": [
                "ListAutomaticTapeCreationPolicies",
                "ListFileShares",
                "ListGateways",
                "ListLocalDisks",
                "ListTagsForResource",
                "ListTapePools",
                "ListTapes",
                "ListVolumeInitiators",
                "ListVolumeRecoveryPoints",
                "ListVolumes"
            ]
        };
    }
    /**
     * Grants permission to activate the gateway you previously deployed on your host
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ActivateGateway.html
     */
    toActivateGateway() {
        this.to('storagegateway:ActivateGateway');
        return this;
    }
    /**
     * Grants permission to configure one or more gateway local disks as cache for a cached-volume gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddCache.html
     */
    toAddCache() {
        this.to('storagegateway:AddCache');
        return this;
    }
    /**
     * Grants permission to add one or more tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.to('storagegateway:AddTagsToResource');
        return this;
    }
    /**
     * Grants permission to configure one or more gateway local disks as upload buffer for a specified gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddUploadBuffer.html
     */
    toAddUploadBuffer() {
        this.to('storagegateway:AddUploadBuffer');
        return this;
    }
    /**
     * Grants permission to configure one or more gateway local disks as working storage for a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddWorkingStorage.html
     */
    toAddWorkingStorage() {
        this.to('storagegateway:AddWorkingStorage');
        return this;
    }
    /**
     * Grants permission to move a tape to the target pool specified
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AssignTapePool.html
     */
    toAssignTapePool() {
        this.to('storagegateway:AssignTapePool');
        return this;
    }
    /**
     * Grants permission to connect a volume to an iSCSI connection and then attaches the volume to the specified gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AttachVolume.html
     */
    toAttachVolume() {
        this.to('storagegateway:AttachVolume');
        return this;
    }
    /**
     * Grants permission to allow the governance retention lock on a pool to be bypassed
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/CreatingCustomTapePool.html#TapeRetentionLock
     */
    toBypassGovernanceRetention() {
        this.to('storagegateway:BypassGovernanceRetention');
        return this;
    }
    /**
     * Grants permission to cancel archiving of a virtual tape to the virtual tape shelf (VTS) after the archiving process is initiated
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CancelArchival.html
     */
    toCancelArchival() {
        this.to('storagegateway:CancelArchival');
        return this;
    }
    /**
     * Grants permission to cancel retrieval of a virtual tape from the virtual tape shelf (VTS) to a gateway after the retrieval process is initiated
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CancelRetrieval.html
     */
    toCancelRetrieval() {
        this.to('storagegateway:CancelRetrieval');
        return this;
    }
    /**
     * Grants permission to create a cached volume on a specified cached gateway. This operation is supported only for the gateway-cached volume architecture
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateCachediSCSIVolume.html
     */
    toCreateCachediSCSIVolume() {
        this.to('storagegateway:CreateCachediSCSIVolume');
        return this;
    }
    /**
     * Grants permission to create a NFS file share on an existing file gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateNFSFileShare.html
     */
    toCreateNFSFileShare() {
        this.to('storagegateway:CreateNFSFileShare');
        return this;
    }
    /**
     * Grants permission to create a SMB file share on an existing file gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateSMBFileShare.html
     */
    toCreateSMBFileShare() {
        this.to('storagegateway:CreateSMBFileShare');
        return this;
    }
    /**
     * Grants permission to initiate a snapshot of a volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        this.to('storagegateway:CreateSnapshot');
        return this;
    }
    /**
     * Grants permission to initiate a snapshot of a gateway from a volume recovery point
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateSnapshotFromVolumeRecoveryPoint.html
     */
    toCreateSnapshotFromVolumeRecoveryPoint() {
        this.to('storagegateway:CreateSnapshotFromVolumeRecoveryPoint');
        return this;
    }
    /**
     * Grants permission to create a volume on a specified gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateStorediSCSIVolume.html
     */
    toCreateStorediSCSIVolume() {
        this.to('storagegateway:CreateStorediSCSIVolume');
        return this;
    }
    /**
     * Grants permission to create a tape pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateTapePool.html
     */
    toCreateTapePool() {
        this.to('storagegateway:CreateTapePool');
        return this;
    }
    /**
     * Grants permission to create a virtual tape by using your own barcode
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateTapeWithBarcode.html
     */
    toCreateTapeWithBarcode() {
        this.to('storagegateway:CreateTapeWithBarcode');
        return this;
    }
    /**
     * Grants permission to create one or more virtual tapes. You write data to the virtual tapes and then archive the tapes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateTapes.html
     */
    toCreateTapes() {
        this.to('storagegateway:CreateTapes');
        return this;
    }
    /**
     * Grants permission to delete the automatic tape creation policy configured on a gateway-VTL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteAutomaticTapeCreationPolicy.html
     */
    toDeleteAutomaticTapeCreationPolicy() {
        this.to('storagegateway:DeleteAutomaticTapeCreationPolicy');
        return this;
    }
    /**
     * Grants permission to delete the bandwidth rate limits of a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteBandwidthRateLimit.html
     */
    toDeleteBandwidthRateLimit() {
        this.to('storagegateway:DeleteBandwidthRateLimit');
        return this;
    }
    /**
     * Grants permission to delete Challenge-Handshake Authentication Protocol (CHAP) credentials for a specified iSCSI target and initiator pair
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteChapCredentials.html
     */
    toDeleteChapCredentials() {
        this.to('storagegateway:DeleteChapCredentials');
        return this;
    }
    /**
     * Grants permission to delete a file share from a file gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteFileShare.html
     */
    toDeleteFileShare() {
        this.to('storagegateway:DeleteFileShare');
        return this;
    }
    /**
     * Grants permission to delete a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteGateway.html
     */
    toDeleteGateway() {
        this.to('storagegateway:DeleteGateway');
        return this;
    }
    /**
     * Grants permission to delete a snapshot of a volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteSnapshotSchedule.html
     */
    toDeleteSnapshotSchedule() {
        this.to('storagegateway:DeleteSnapshotSchedule');
        return this;
    }
    /**
     * Grants permission to delete the specified virtual tape
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteTape.html
     */
    toDeleteTape() {
        this.to('storagegateway:DeleteTape');
        return this;
    }
    /**
     * Grants permission to delete the specified virtual tape from the virtual tape shelf (VTS)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteTapeArchive.html
     */
    toDeleteTapeArchive() {
        this.to('storagegateway:DeleteTapeArchive');
        return this;
    }
    /**
     * Grants permission to delete the specified tape pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteTapePool.html
     */
    toDeleteTapePool() {
        this.to('storagegateway:DeleteTapePool');
        return this;
    }
    /**
     * Grants permission to delete the specified gateway volume that you previously created using the CreateCachediSCSIVolume or CreateStorediSCSIVolume API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteVolume.html
     */
    toDeleteVolume() {
        this.to('storagegateway:DeleteVolume');
        return this;
    }
    /**
     * Grants permission to get the bandwidth rate limits of a gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeBandwidthRateLimit.html
     */
    toDescribeBandwidthRateLimit() {
        this.to('storagegateway:DescribeBandwidthRateLimit');
        return this;
    }
    /**
     * Grants permission to get information about the cache of a gateway. This operation is supported only for the gateway-cached volume architecture
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeCache.html
     */
    toDescribeCache() {
        this.to('storagegateway:DescribeCache');
        return this;
    }
    /**
     * Grants permission to get a description of the gateway volumes specified in the request. This operation is supported only for the gateway-cached volume architecture
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeCachediSCSIVolumes.html
     */
    toDescribeCachediSCSIVolumes() {
        this.to('storagegateway:DescribeCachediSCSIVolumes');
        return this;
    }
    /**
     * Grants permission to get an array of Challenge-Handshake Authentication Protocol (CHAP) credentials information for a specified iSCSI target, one for each target-initiator pair
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeChapCredentials.html
     */
    toDescribeChapCredentials() {
        this.to('storagegateway:DescribeChapCredentials');
        return this;
    }
    /**
     * Grants permission to get metadata about a gateway such as its name, network interfaces, configured time zone, and the state (whether the gateway is running or not)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeGatewayInformation.html
     */
    toDescribeGatewayInformation() {
        this.to('storagegateway:DescribeGatewayInformation');
        return this;
    }
    /**
     * Grants permission to get your gateway's weekly maintenance start time including the day and time of the week
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeMaintenanceStartTime.html
     */
    toDescribeMaintenanceStartTime() {
        this.to('storagegateway:DescribeMaintenanceStartTime');
        return this;
    }
    /**
     * Grants permission to get a description for one or more file shares from a file gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeNFSFileShares.html
     */
    toDescribeNFSFileShares() {
        this.to('storagegateway:DescribeNFSFileShares');
        return this;
    }
    /**
     * Grants permission to get a description for one or more file shares from a file gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeSMBFileShares.html
     */
    toDescribeSMBFileShares() {
        this.to('storagegateway:DescribeSMBFileShares');
        return this;
    }
    /**
     * Grants permission to get a description of a Server Message Block (SMB) file share settings from a file gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeSMBSettings.html
     */
    toDescribeSMBSettings() {
        this.to('storagegateway:DescribeSMBSettings');
        return this;
    }
    /**
     * Grants permission to describe the snapshot schedule for the specified gateway volume
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeSnapshotSchedule.html
     */
    toDescribeSnapshotSchedule() {
        this.to('storagegateway:DescribeSnapshotSchedule');
        return this;
    }
    /**
     * Grants permission to get the description of the gateway volumes specified in the request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeStorediSCSIVolumes.html
     */
    toDescribeStorediSCSIVolumes() {
        this.to('storagegateway:DescribeStorediSCSIVolumes');
        return this;
    }
    /**
     * Grants permission to get a description of specified virtual tapes in the virtual tape shelf (VTS)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeTapeArchives.html
     */
    toDescribeTapeArchives() {
        this.to('storagegateway:DescribeTapeArchives');
        return this;
    }
    /**
     * Grants permission to get a list of virtual tape recovery points that are available for the specified gateway-VTL
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeTapeRecoveryPoints.html
     */
    toDescribeTapeRecoveryPoints() {
        this.to('storagegateway:DescribeTapeRecoveryPoints');
        return this;
    }
    /**
     * Grants permission to get a description of the specified Amazon Resource Name (ARN) of virtual tapes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeTapes.html
     */
    toDescribeTapes() {
        this.to('storagegateway:DescribeTapes');
        return this;
    }
    /**
     * Grants permission to get information about the upload buffer of a gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeUploadBuffer.html
     */
    toDescribeUploadBuffer() {
        this.to('storagegateway:DescribeUploadBuffer');
        return this;
    }
    /**
     * Grants permission to get a description of virtual tape library (VTL) devices for the specified gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeVTLDevices.html
     */
    toDescribeVTLDevices() {
        this.to('storagegateway:DescribeVTLDevices');
        return this;
    }
    /**
     * Grants permission to get information about the working storage of a gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeWorkingStorage.html
     */
    toDescribeWorkingStorage() {
        this.to('storagegateway:DescribeWorkingStorage');
        return this;
    }
    /**
     * Grants permission to disconnect a volume from an iSCSI connection and then detaches the volume from the specified gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DetachVolume.html
     */
    toDetachVolume() {
        this.to('storagegateway:DetachVolume');
        return this;
    }
    /**
     * Grants permission to disable a gateway when the gateway is no longer functioning
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DisableGateway.html
     */
    toDisableGateway() {
        this.to('storagegateway:DisableGateway');
        return this;
    }
    /**
     * Grants permission to enable you to join an Active Directory Domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_JoinDomain.html
     */
    toJoinDomain() {
        this.to('storagegateway:JoinDomain');
        return this;
    }
    /**
     * Grants permission to list the automatic tape creation policies configured on the specified gateway-VTL or all gateway-VTLs owned by your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListAutomaticTapeCreationPolicies.html
     */
    toListAutomaticTapeCreationPolicies() {
        this.to('storagegateway:ListAutomaticTapeCreationPolicies');
        return this;
    }
    /**
     * Grants permission to get a list of the file shares for a specific file gateway, or the list of file shares that belong to the calling user account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListFileShares.html
     */
    toListFileShares() {
        this.to('storagegateway:ListFileShares');
        return this;
    }
    /**
     * Grants permission to list gateways owned by an AWS account in a region specified in the request. The returned list is ordered by gateway Amazon Resource Name (ARN)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListGateways.html
     */
    toListGateways() {
        this.to('storagegateway:ListGateways');
        return this;
    }
    /**
     * Grants permission to get a list of the gateway's local disks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListLocalDisks.html
     */
    toListLocalDisks() {
        this.to('storagegateway:ListLocalDisks');
        return this;
    }
    /**
     * Grants permission to get the tags that have been added to the specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('storagegateway:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list tape pools owned by your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListTapePools.html
     */
    toListTapePools() {
        this.to('storagegateway:ListTapePools');
        return this;
    }
    /**
     * Grants permission to list virtual tapes in your virtual tape library (VTL) and your virtual tape shelf (VTS)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListTapes.html
     */
    toListTapes() {
        this.to('storagegateway:ListTapes');
        return this;
    }
    /**
     * Grants permission to list iSCSI initiators that are connected to a volume
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListVolumeInitiators.html
     */
    toListVolumeInitiators() {
        this.to('storagegateway:ListVolumeInitiators');
        return this;
    }
    /**
     * Grants permission to list the recovery points for a specified gateway
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListVolumeRecoveryPoints.html
     */
    toListVolumeRecoveryPoints() {
        this.to('storagegateway:ListVolumeRecoveryPoints');
        return this;
    }
    /**
     * Grants permission to list the iSCSI stored volumes of a gateway
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListVolumes.html
     */
    toListVolumes() {
        this.to('storagegateway:ListVolumes');
        return this;
    }
    /**
     * Grants permission to send you a notification through CloudWatch Events when all files written to your NFS file share have been uploaded to Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_NotifyWhenUploaded.html
     */
    toNotifyWhenUploaded() {
        this.to('storagegateway:NotifyWhenUploaded');
        return this;
    }
    /**
     * Grants permission to refresh the cache for the specified file share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RefreshCache.html
     */
    toRefreshCache() {
        this.to('storagegateway:RefreshCache');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.to('storagegateway:RemoveTagsFromResource');
        return this;
    }
    /**
     * Grants permission to reset all cache disks that have encountered a error and makes the disks available for reconfiguration as cache storage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ResetCache.html
     */
    toResetCache() {
        this.to('storagegateway:ResetCache');
        return this;
    }
    /**
     * Grants permission to retrieve an archived virtual tape from the virtual tape shelf (VTS) to a gateway-VTL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RetrieveTapeArchive.html
     */
    toRetrieveTapeArchive() {
        this.to('storagegateway:RetrieveTapeArchive');
        return this;
    }
    /**
     * Grants permission to retrieve the recovery point for the specified virtual tape
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RetrieveTapeRecoveryPoint.html
     */
    toRetrieveTapeRecoveryPoint() {
        this.to('storagegateway:RetrieveTapeRecoveryPoint');
        return this;
    }
    /**
     * Grants permission to set the password for your VM local console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_SetLocalConsolePassword.html
     */
    toSetLocalConsolePassword() {
        this.to('storagegateway:SetLocalConsolePassword');
        return this;
    }
    /**
     * Grants permission to set the password for SMB Guest user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_SetSMBGuestPassword.html
     */
    toSetSMBGuestPassword() {
        this.to('storagegateway:SetSMBGuestPassword');
        return this;
    }
    /**
     * Grants permission to shut down a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ShutdownGateway.html
     */
    toShutdownGateway() {
        this.to('storagegateway:ShutdownGateway');
        return this;
    }
    /**
     * Grants permission to start a gateway that you previously shut down
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_StartGateway.html
     */
    toStartGateway() {
        this.to('storagegateway:StartGateway');
        return this;
    }
    /**
     * Grants permission to update the automatic tape creation policy configured on a gateway-VTL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateAutomaticTapeCreationPolicy.html
     */
    toUpdateAutomaticTapeCreationPolicy() {
        this.to('storagegateway:UpdateAutomaticTapeCreationPolicy');
        return this;
    }
    /**
     * Grants permission to update the bandwidth rate limits of a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateBandwidthRateLimit.html
     */
    toUpdateBandwidthRateLimit() {
        this.to('storagegateway:UpdateBandwidthRateLimit');
        return this;
    }
    /**
     * Grants permission to update the Challenge-Handshake Authentication Protocol (CHAP) credentials for a specified iSCSI target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateChapCredentials.html
     */
    toUpdateChapCredentials() {
        this.to('storagegateway:UpdateChapCredentials');
        return this;
    }
    /**
     * Grants permission to update a gateway's metadata, which includes the gateway's name and time zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateGatewayInformation.html
     */
    toUpdateGatewayInformation() {
        this.to('storagegateway:UpdateGatewayInformation');
        return this;
    }
    /**
     * Grants permission to update the gateway virtual machine (VM) software
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateGatewaySoftwareNow.html
     */
    toUpdateGatewaySoftwareNow() {
        this.to('storagegateway:UpdateGatewaySoftwareNow');
        return this;
    }
    /**
     * Grants permission to update a gateway's weekly maintenance start time information, including day and time of the week. The maintenance time is the time in your gateway's time zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateMaintenanceStartTime.html
     */
    toUpdateMaintenanceStartTime() {
        this.to('storagegateway:UpdateMaintenanceStartTime');
        return this;
    }
    /**
     * Grants permission to update a NFS file share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateNFSFileShare.html
     */
    toUpdateNFSFileShare() {
        this.to('storagegateway:UpdateNFSFileShare');
        return this;
    }
    /**
     * Grants permission to update a SMB file share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateSMBFileShare.html
     */
    toUpdateSMBFileShare() {
        this.to('storagegateway:UpdateSMBFileShare');
        return this;
    }
    /**
     * Grants permission to update a snapshot schedule configured for a gateway volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateSnapshotSchedule.html
     */
    toUpdateSnapshotSchedule() {
        this.to('storagegateway:UpdateSnapshotSchedule');
        return this;
    }
    /**
     * Grants permission to update the type of medium changer in a gateway-VTL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateVTLDeviceType.html
     */
    toUpdateVTLDeviceType() {
        this.to('storagegateway:UpdateVTLDeviceType');
        return this;
    }
    /**
     * Adds a resource of type device to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/resource_vtl-devices.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param vtldevice - Identifier for the vtldevice.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDevice(gatewayId, vtldevice, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:gateway/${GatewayId}/device/${Vtldevice}';
        arn = arn.replace('${GatewayId}', gatewayId);
        arn = arn.replace('${Vtldevice}', vtldevice);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type gateway to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/StorageGatewayConcepts.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGateway(gatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:gateway/${GatewayId}';
        arn = arn.replace('${GatewayId}', gatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type share to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/GettingStartedCreateFileShare.html
     *
     * @param shareId - Identifier for the shareId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onShare(shareId, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:share/${ShareId}';
        arn = arn.replace('${ShareId}', shareId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type tape to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/StorageGatewayConcepts.html#storage-gateway-vtl-concepts
     *
     * @param tapeBarcode - Identifier for the tapeBarcode.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTape(tapeBarcode, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:tape/${TapeBarcode}';
        arn = arn.replace('${TapeBarcode}', tapeBarcode);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type tapepool to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/CreatingCustomTapePool.html
     *
     * @param poolId - Identifier for the poolId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTapepool(poolId, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:tapepool/${PoolId}';
        arn = arn.replace('${PoolId}', poolId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type target to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/GettingStartedCreateVolumes.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param iscsiTarget - Identifier for the iscsiTarget.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTarget(gatewayId, iscsiTarget, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:gateway/${GatewayId}/target/${IscsiTarget}';
        arn = arn.replace('${GatewayId}', gatewayId);
        arn = arn.replace('${IscsiTarget}', iscsiTarget);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type volume to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/StorageGatewayConcepts.html#volume-gateway-concepts
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param volumeId - Identifier for the volumeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVolume(gatewayId, volumeId, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:gateway/${GatewayId}/volume/${VolumeId}';
        arn = arn.replace('${GatewayId}', gatewayId);
        arn = arn.replace('${VolumeId}', volumeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Storagegateway = Storagegateway;
//# sourceMappingURL=data:application/json;base64,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