"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ssm = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ssm](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ssm extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ssm](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ssm';
        this.accessLevelList = {
            "Tagging": [
                "AddTagsToResource",
                "RemoveTagsFromResource"
            ],
            "Write": [
                "CancelCommand",
                "CancelMaintenanceWindowExecution",
                "CreateActivation",
                "CreateAssociation",
                "CreateAssociationBatch",
                "CreateDocument",
                "CreateMaintenanceWindow",
                "CreateOpsItem",
                "CreatePatchBaseline",
                "CreateResourceDataSync",
                "DeleteActivation",
                "DeleteAssociation",
                "DeleteDocument",
                "DeleteInventory",
                "DeleteMaintenanceWindow",
                "DeleteParameter",
                "DeleteParameters",
                "DeletePatchBaseline",
                "DeleteResourceDataSync",
                "DeregisterManagedInstance",
                "DeregisterPatchBaselineForPatchGroup",
                "DeregisterTargetFromMaintenanceWindow",
                "DeregisterTaskFromMaintenanceWindow",
                "LabelParameterVersion",
                "ModifyDocumentPermission",
                "PutComplianceItems",
                "PutInventory",
                "PutParameter",
                "RegisterDefaultPatchBaseline",
                "RegisterPatchBaselineForPatchGroup",
                "RegisterTargetWithMaintenanceWindow",
                "RegisterTaskWithMaintenanceWindow",
                "ResetServiceSetting",
                "ResumeSession",
                "SendAutomationSignal",
                "SendCommand",
                "StartAssociationsOnce",
                "StartAutomationExecution",
                "StartSession",
                "StopAutomationExecution",
                "TerminateSession",
                "UpdateAssociation",
                "UpdateAssociationStatus",
                "UpdateDocument",
                "UpdateDocumentDefaultVersion",
                "UpdateInstanceAssociationStatus",
                "UpdateInstanceInformation",
                "UpdateMaintenanceWindow",
                "UpdateMaintenanceWindowTarget",
                "UpdateMaintenanceWindowTask",
                "UpdateManagedInstanceRole",
                "UpdateOpsItem",
                "UpdatePatchBaseline",
                "UpdateResourceDataSync",
                "UpdateServiceSetting"
            ],
            "Read": [
                "DescribeActivations",
                "DescribeAssociation",
                "DescribeAssociationExecutionTargets",
                "DescribeAssociationExecutions",
                "DescribeAutomationExecutions",
                "DescribeAutomationStepExecutions",
                "DescribeAvailablePatches",
                "DescribeDocument",
                "DescribeDocumentParameters",
                "DescribeDocumentPermission",
                "DescribeEffectiveInstanceAssociations",
                "DescribeEffectivePatchesForPatchBaseline",
                "DescribeInstanceAssociationsStatus",
                "DescribeInstanceInformation",
                "DescribeInstancePatchStates",
                "DescribeInstancePatchStatesForPatchGroup",
                "DescribeInstancePatches",
                "DescribeInstanceProperties",
                "DescribeInventoryDeletions",
                "DescribeOpsItems",
                "DescribePatchGroupState",
                "GetAutomationExecution",
                "GetCalendarState",
                "GetCommandInvocation",
                "GetConnectionStatus",
                "GetDefaultPatchBaseline",
                "GetDeployablePatchSnapshotForInstance",
                "GetDocument",
                "GetInventory",
                "GetInventorySchema",
                "GetMaintenanceWindow",
                "GetMaintenanceWindowExecution",
                "GetMaintenanceWindowExecutionTask",
                "GetMaintenanceWindowExecutionTaskInvocation",
                "GetMaintenanceWindowTask",
                "GetManifest",
                "GetOpsItem",
                "GetOpsSummary",
                "GetParameter",
                "GetParameterHistory",
                "GetParameters",
                "GetParametersByPath",
                "GetPatchBaseline",
                "GetPatchBaselineForPatchGroup",
                "GetServiceSetting",
                "ListCommandInvocations",
                "ListCommands",
                "ListTagsForResource",
                "PutConfigurePackageResult"
            ],
            "List": [
                "DescribeMaintenanceWindowExecutionTaskInvocations",
                "DescribeMaintenanceWindowExecutionTasks",
                "DescribeMaintenanceWindowExecutions",
                "DescribeMaintenanceWindowSchedule",
                "DescribeMaintenanceWindowTargets",
                "DescribeMaintenanceWindowTasks",
                "DescribeMaintenanceWindows",
                "DescribeMaintenanceWindowsForTarget",
                "DescribeParameters",
                "DescribePatchBaselines",
                "DescribePatchGroups",
                "DescribePatchProperties",
                "DescribeSessions",
                "ListAssociationVersions",
                "ListAssociations",
                "ListComplianceItems",
                "ListComplianceSummaries",
                "ListDocumentVersions",
                "ListDocuments",
                "ListInstanceAssociations",
                "ListInventoryEntries",
                "ListResourceComplianceSummaries",
                "ListResourceDataSync"
            ]
        };
    }
    /**
     * Grants permission to add or overwrite one or more tags for a specified AWS resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.to('ssm:AddTagsToResource');
        return this;
    }
    /**
     * Grants permission to cancel a specified Run Command command
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CancelCommand.html
     */
    toCancelCommand() {
        this.to('ssm:CancelCommand');
        return this;
    }
    /**
     * Grants permission to cancel an in-progress maintenance window execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CancelMaintenanceWindowExecution.html
     */
    toCancelMaintenanceWindowExecution() {
        this.to('ssm:CancelMaintenanceWindowExecution');
        return this;
    }
    /**
     * Grants permission to create an activation that is used to register on-premises servers and virtual machines (VMs) with Systems Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateActivation.html
     */
    toCreateActivation() {
        this.to('ssm:CreateActivation');
        return this;
    }
    /**
     * Grants permission to associate a specified Systems Manager document with specified instances or other targets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateAssociation.html
     */
    toCreateAssociation() {
        this.to('ssm:CreateAssociation');
        return this;
    }
    /**
     * Grants permission to combine entries for multiple CreateAssociation operations in a single command
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateAssociationBatch.html
     */
    toCreateAssociationBatch() {
        this.to('ssm:CreateAssociationBatch');
        return this;
    }
    /**
     * Grants permission to create a Systems Manager SSM document
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateDocument.html
     */
    toCreateDocument() {
        this.to('ssm:CreateDocument');
        return this;
    }
    /**
     * Grants permission to create a maintenance window
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateMaintenanceWindow.html
     */
    toCreateMaintenanceWindow() {
        this.to('ssm:CreateMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to create an OpsItem in OpsCenter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateOpsItem.html
     */
    toCreateOpsItem() {
        this.to('ssm:CreateOpsItem');
        return this;
    }
    /**
     * Grants permission to create a patch baseline
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreatePatchBaseline.html
     */
    toCreatePatchBaseline() {
        this.to('ssm:CreatePatchBaseline');
        return this;
    }
    /**
     * Grants permission to create a resource data sync configuration, which regularly collects inventory data from managed instances and updates the data in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateResourceDataSync.html
     */
    toCreateResourceDataSync() {
        this.to('ssm:CreateResourceDataSync');
        return this;
    }
    /**
     * Grants permission to delete a specified activation for managed instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteActivation.html
     */
    toDeleteActivation() {
        this.to('ssm:DeleteActivation');
        return this;
    }
    /**
     * Grants permission to disassociate a specified SSM document from a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteAssociation.html
     */
    toDeleteAssociation() {
        this.to('ssm:DeleteAssociation');
        return this;
    }
    /**
     * Grants permission to delete a specified SSM document and its instance associations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteDocument.html
     */
    toDeleteDocument() {
        this.to('ssm:DeleteDocument');
        return this;
    }
    /**
     * Grants permission to delete a specified custom inventory type, or the data associated with a custom inventory type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteInventory.html
     */
    toDeleteInventory() {
        this.to('ssm:DeleteInventory');
        return this;
    }
    /**
     * Grants permission to delete a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteMaintenanceWindow.html
     */
    toDeleteMaintenanceWindow() {
        this.to('ssm:DeleteMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to delete a specified SSM parameter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteParameter.html
     */
    toDeleteParameter() {
        this.to('ssm:DeleteParameter');
        return this;
    }
    /**
     * Grants permission to delete multiple specified SSM parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteParameters.html
     */
    toDeleteParameters() {
        this.to('ssm:DeleteParameters');
        return this;
    }
    /**
     * Grants permission to delete a specified patch baseline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeletePatchBaseline.html
     */
    toDeletePatchBaseline() {
        this.to('ssm:DeletePatchBaseline');
        return this;
    }
    /**
     * Grants permission to delete a specified resource data sync
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteResourceDataSync.html
     */
    toDeleteResourceDataSync() {
        this.to('ssm:DeleteResourceDataSync');
        return this;
    }
    /**
     * Grants permission to deregister a specified on-premises server or virtual machine (VM) from Systems Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterManagedInstance.html
     */
    toDeregisterManagedInstance() {
        this.to('ssm:DeregisterManagedInstance');
        return this;
    }
    /**
     * Grants permission to deregister a specified patch baseline from being the default patch baseline for a specified patch group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterPatchBaselineForPatchGroup.html
     */
    toDeregisterPatchBaselineForPatchGroup() {
        this.to('ssm:DeregisterPatchBaselineForPatchGroup');
        return this;
    }
    /**
     * Grants permission to deregister a specified target from a maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterTargetFromMaintenanceWindow.html
     */
    toDeregisterTargetFromMaintenanceWindow() {
        this.to('ssm:DeregisterTargetFromMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to deregister a specified task from a maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterTaskFromMaintenanceWindow.html
     */
    toDeregisterTaskFromMaintenanceWindow() {
        this.to('ssm:DeregisterTaskFromMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to view details about a specified managed instance activation, such as when it was created and the number of instances registered using the activation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeActivations.html
     */
    toDescribeActivations() {
        this.to('ssm:DescribeActivations');
        return this;
    }
    /**
     * Grants permission to view details about the specified association for a specified instance or target
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociation.html
     */
    toDescribeAssociation() {
        this.to('ssm:DescribeAssociation');
        return this;
    }
    /**
     * Grants permission to view information about a specified association execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociationExecutionTargets.html
     */
    toDescribeAssociationExecutionTargets() {
        this.to('ssm:DescribeAssociationExecutionTargets');
        return this;
    }
    /**
     * Grants permission to view all executions for a specified association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociationExecutions.html
     */
    toDescribeAssociationExecutions() {
        this.to('ssm:DescribeAssociationExecutions');
        return this;
    }
    /**
     * Grants permission to view details about all active and terminated Automation executions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAutomationExecutions.html
     */
    toDescribeAutomationExecutions() {
        this.to('ssm:DescribeAutomationExecutions');
        return this;
    }
    /**
     * Grants permission to view information about all active and terminated step executions in an Automation workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAutomationStepExecutions.html
     */
    toDescribeAutomationStepExecutions() {
        this.to('ssm:DescribeAutomationStepExecutions');
        return this;
    }
    /**
     * Grants permission to view all patches eligible to include in a patch baseline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAvailablePatches.html
     */
    toDescribeAvailablePatches() {
        this.to('ssm:DescribeAvailablePatches');
        return this;
    }
    /**
     * Grants permission to view details about a specified SSM document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeDocument.html
     */
    toDescribeDocument() {
        this.to('ssm:DescribeDocument');
        return this;
    }
    /**
     * Grants permission to display information about SSM document parameters in the Systems Manager console (internal Systems Manager action)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toDescribeDocumentParameters() {
        this.to('ssm:DescribeDocumentParameters');
        return this;
    }
    /**
     * Grants permission to view the permissions for a specified SSM document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeDocumentPermission.html
     */
    toDescribeDocumentPermission() {
        this.to('ssm:DescribeDocumentPermission');
        return this;
    }
    /**
     * Grants permission to view all current associations for a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeEffectiveInstanceAssociations.html
     */
    toDescribeEffectiveInstanceAssociations() {
        this.to('ssm:DescribeEffectiveInstanceAssociations');
        return this;
    }
    /**
     * Grants permission to view details about the patches currently associated with the specified patch baseline (Windows only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeEffectivePatchesForPatchBaseline.html
     */
    toDescribeEffectivePatchesForPatchBaseline() {
        this.to('ssm:DescribeEffectivePatchesForPatchBaseline');
        return this;
    }
    /**
     * Grants permission to view the status of the associations for a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstanceAssociationsStatus.html
     */
    toDescribeInstanceAssociationsStatus() {
        this.to('ssm:DescribeInstanceAssociationsStatus');
        return this;
    }
    /**
     * Grants permission to view details about a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstanceInformation.html
     */
    toDescribeInstanceInformation() {
        this.to('ssm:DescribeInstanceInformation');
        return this;
    }
    /**
     * Grants permission to view status details about patches on a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatchStates.html
     */
    toDescribeInstancePatchStates() {
        this.to('ssm:DescribeInstancePatchStates');
        return this;
    }
    /**
     * Grants permission to describe the high-level patch state for the instances in the specified patch group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatchStatesForPatchGroup.html
     */
    toDescribeInstancePatchStatesForPatchGroup() {
        this.to('ssm:DescribeInstancePatchStatesForPatchGroup');
        return this;
    }
    /**
     * Grants permission to view general details about the patches on a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatches.html
     */
    toDescribeInstancePatches() {
        this.to('ssm:DescribeInstancePatches');
        return this;
    }
    /**
     * Grants permission to user's Amazon EC2 console to render managed instances' nodes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toDescribeInstanceProperties() {
        this.to('ssm:DescribeInstanceProperties');
        return this;
    }
    /**
     * Grants permission to view details about a specified inventory deletion
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInventoryDeletions.html
     */
    toDescribeInventoryDeletions() {
        this.to('ssm:DescribeInventoryDeletions');
        return this;
    }
    /**
     * Grants permission to view details of a specified task execution for a maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutionTaskInvocations.html
     */
    toDescribeMaintenanceWindowExecutionTaskInvocations() {
        this.to('ssm:DescribeMaintenanceWindowExecutionTaskInvocations');
        return this;
    }
    /**
     * Grants permission to view details about the tasks that ran during a specified maintenance window execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutionTasks.html
     */
    toDescribeMaintenanceWindowExecutionTasks() {
        this.to('ssm:DescribeMaintenanceWindowExecutionTasks');
        return this;
    }
    /**
     * Grants permission to view the executions of a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutions.html
     */
    toDescribeMaintenanceWindowExecutions() {
        this.to('ssm:DescribeMaintenanceWindowExecutions');
        return this;
    }
    /**
     * Grants permission to view details about upcoming executions of a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowSchedule.html
     */
    toDescribeMaintenanceWindowSchedule() {
        this.to('ssm:DescribeMaintenanceWindowSchedule');
        return this;
    }
    /**
     * Grants permission to view a list of the targets associated with a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowTargets.html
     */
    toDescribeMaintenanceWindowTargets() {
        this.to('ssm:DescribeMaintenanceWindowTargets');
        return this;
    }
    /**
     * Grants permission to view a list of the tasks associated with a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowTasks.html
     */
    toDescribeMaintenanceWindowTasks() {
        this.to('ssm:DescribeMaintenanceWindowTasks');
        return this;
    }
    /**
     * Grants permission to view information about all or specified maintenance windows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindows.html
     */
    toDescribeMaintenanceWindows() {
        this.to('ssm:DescribeMaintenanceWindows');
        return this;
    }
    /**
     * Grants permission to view information about the maintenance window targets and tasks associated with a specified instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowsForTarget.html
     */
    toDescribeMaintenanceWindowsForTarget() {
        this.to('ssm:DescribeMaintenanceWindowsForTarget');
        return this;
    }
    /**
     * Grants permission to view details about specified OpsItems
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeOpsItems.html
     */
    toDescribeOpsItems() {
        this.to('ssm:DescribeOpsItems');
        return this;
    }
    /**
     * Grants permission to view details about a specified SSM parameter
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeParameters.html
     */
    toDescribeParameters() {
        this.to('ssm:DescribeParameters');
        return this;
    }
    /**
     * Grants permission to view information about patch baselines that meet the specified criteria
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchBaselines.html
     */
    toDescribePatchBaselines() {
        this.to('ssm:DescribePatchBaselines');
        return this;
    }
    /**
     * Grants permission to view aggregated status details for patches for a specified patch group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchGroupState.html
     */
    toDescribePatchGroupState() {
        this.to('ssm:DescribePatchGroupState');
        return this;
    }
    /**
     * Grants permission to view information about the patch baseline for a specified patch group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchGroups.html
     */
    toDescribePatchGroups() {
        this.to('ssm:DescribePatchGroups');
        return this;
    }
    /**
     * Grants permission to view details of available patches for a specified operating system and patch property
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchProperties.html
     */
    toDescribePatchProperties() {
        this.to('ssm:DescribePatchProperties');
        return this;
    }
    /**
     * Grants permission to view a list of recent Session Manager sessions that meet the specified search criteria
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeSessions.html
     */
    toDescribeSessions() {
        this.to('ssm:DescribeSessions');
        return this;
    }
    /**
     * Grants permission to view details of a specified Automation execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_AutomationExecution.html
     */
    toGetAutomationExecution() {
        this.to('ssm:GetAutomationExecution');
        return this;
    }
    /**
     * Grants permission to view the calendar state for a change calendar or a list of change calendars
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetCalendarState.html
     */
    toGetCalendarState() {
        this.to('ssm:GetCalendarState');
        return this;
    }
    /**
     * Grants permission to view details about the command execution of a specified invocation or plugin
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetCommandInvocation.html
     */
    toGetCommandInvocation() {
        this.to('ssm:GetCommandInvocation');
        return this;
    }
    /**
     * Grants permission to view the Session Manager connection status for a specified managed instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetConnectionStatus.html
     */
    toGetConnectionStatus() {
        this.to('ssm:GetConnectionStatus');
        return this;
    }
    /**
     * Grants permission to view the current default patch baseline for a specified operating system type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDefaultPatchBaseline.html
     */
    toGetDefaultPatchBaseline() {
        this.to('ssm:GetDefaultPatchBaseline');
        return this;
    }
    /**
     * Grants permission to retrieve the current patch baseline snapshot for a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDeployablePatchSnapshotForInstance.html
     */
    toGetDeployablePatchSnapshotForInstance() {
        this.to('ssm:GetDeployablePatchSnapshotForInstance');
        return this;
    }
    /**
     * Grants permission to view the contents of a specified SSM document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDocument.html
     */
    toGetDocument() {
        this.to('ssm:GetDocument');
        return this;
    }
    /**
     * Grants permission to view instance inventory details per the specified criteria
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetInventory.html
     */
    toGetInventory() {
        this.to('ssm:GetInventory');
        return this;
    }
    /**
     * Grants permission to view a list of inventory types or attribute names for a specified inventory item type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetInventorySchema.html
     */
    toGetInventorySchema() {
        this.to('ssm:GetInventorySchema');
        return this;
    }
    /**
     * Grants permission to view details about a specified maintenance window
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindow.html
     */
    toGetMaintenanceWindow() {
        this.to('ssm:GetMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to view details about a specified maintenance window execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecution.html
     */
    toGetMaintenanceWindowExecution() {
        this.to('ssm:GetMaintenanceWindowExecution');
        return this;
    }
    /**
     * Grants permission to view details about a specified maintenance window execution task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecutionTask.html
     */
    toGetMaintenanceWindowExecutionTask() {
        this.to('ssm:GetMaintenanceWindowExecutionTask');
        return this;
    }
    /**
     * Grants permission to view details about a specific maintenance window task running on a specific target
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecutionTaskInvocation.html
     */
    toGetMaintenanceWindowExecutionTaskInvocation() {
        this.to('ssm:GetMaintenanceWindowExecutionTaskInvocation');
        return this;
    }
    /**
     * Grants permission to view details about tasks registered with a specified maintenance window
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowTask.html
     */
    toGetMaintenanceWindowTask() {
        this.to('ssm:GetMaintenanceWindowTask');
        return this;
    }
    /**
     * Used by Systems Manager and SSM Agent to determine package installation requirements for an instance (internal Systems Manager call)
     *
     * Access Level: Read
     */
    toGetManifest() {
        this.to('ssm:GetManifest');
        return this;
    }
    /**
     * Grants permission to view information about a specified OpsItem
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetOpsItem.html
     */
    toGetOpsItem() {
        this.to('ssm:GetOpsItem');
        return this;
    }
    /**
     * Grants permission to view summary information about OpsItems based on specified filters and aggregators
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetOpsSummary.html
     */
    toGetOpsSummary() {
        this.to('ssm:GetOpsSummary');
        return this;
    }
    /**
     * Grants permission to view information about a specified parameter
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameter.html
     */
    toGetParameter() {
        this.to('ssm:GetParameter');
        return this;
    }
    /**
     * Grants permission to view details and changes for a specified parameter
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameterHistory.html
     */
    toGetParameterHistory() {
        this.to('ssm:GetParameterHistory');
        return this;
    }
    /**
     * Grants permission to view information about multiple specified parameters
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameters.html
     */
    toGetParameters() {
        this.to('ssm:GetParameters');
        return this;
    }
    /**
     * Grants permission to view information about parameters in a specified hierarchy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParametersByPath.html
     */
    toGetParametersByPath() {
        this.to('ssm:GetParametersByPath');
        return this;
    }
    /**
     * Grants permission to view information about a specified patch baseline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetPatchBaseline.html
     */
    toGetPatchBaseline() {
        this.to('ssm:GetPatchBaseline');
        return this;
    }
    /**
     * Grants permission to view the ID of the current patch baseline for a specified patch group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetPatchBaselineForPatchGroup.html
     */
    toGetPatchBaselineForPatchGroup() {
        this.to('ssm:GetPatchBaselineForPatchGroup');
        return this;
    }
    /**
     * Grants permission to view the account-level setting for an AWS service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetServiceSetting.html
     */
    toGetServiceSetting() {
        this.to('ssm:GetServiceSetting');
        return this;
    }
    /**
     * Grants permission to apply an identifying label to a specified version of a parameter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_LabelParameterVersion.html
     */
    toLabelParameterVersion() {
        this.to('ssm:LabelParameterVersion');
        return this;
    }
    /**
     * Grants permission to list versions of the specified association
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListAssociationVersions.html
     */
    toListAssociationVersions() {
        this.to('ssm:ListAssociationVersions');
        return this;
    }
    /**
     * Grants permission to list the associations for a specified SSM document or managed instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListAssociations.html
     */
    toListAssociations() {
        this.to('ssm:ListAssociations');
        return this;
    }
    /**
     * Grants permission to list information about command invocations sent to a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListCommandInvocations.html
     */
    toListCommandInvocations() {
        this.to('ssm:ListCommandInvocations');
        return this;
    }
    /**
     * Grants permission to list the commands sent to a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListCommands.html
     */
    toListCommands() {
        this.to('ssm:ListCommands');
        return this;
    }
    /**
     * Grants permission to list compliance status for specified resource types on a specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListComplianceItems.html
     */
    toListComplianceItems() {
        this.to('ssm:ListComplianceItems');
        return this;
    }
    /**
     * Grants permission to list a summary count of compliant and noncompliant resources for a specified compliance type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListComplianceSummaries.html
     */
    toListComplianceSummaries() {
        this.to('ssm:ListComplianceSummaries');
        return this;
    }
    /**
     * Grants permission to list all versions of a specified document
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListDocumentVersions.html
     */
    toListDocumentVersions() {
        this.to('ssm:ListDocumentVersions');
        return this;
    }
    /**
     * Grants permission to view information about a specified SSM document
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListDocuments.html
     */
    toListDocuments() {
        this.to('ssm:ListDocuments');
        return this;
    }
    /**
     * Used by SSM Agent to check for new State Manager associations (internal Systems Manager call)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toListInstanceAssociations() {
        this.to('ssm:ListInstanceAssociations');
        return this;
    }
    /**
     * Grants permission to view a list of specified inventory types for a specified instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListInventoryEntries.html
     */
    toListInventoryEntries() {
        this.to('ssm:ListInventoryEntries');
        return this;
    }
    /**
     * Grants permission to list resource-level summary count
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListResourceComplianceSummaries.html
     */
    toListResourceComplianceSummaries() {
        this.to('ssm:ListResourceComplianceSummaries');
        return this;
    }
    /**
     * Grants permission to list information about resource data sync configurations in an account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListResourceDataSync.html
     */
    toListResourceDataSync() {
        this.to('ssm:ListResourceDataSync');
        return this;
    }
    /**
     * Grants permission to view a list of resource tags for a specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('ssm:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to share a custom SSM document publicly or privately with specified AWS accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ModifyDocumentPermission.html
     */
    toModifyDocumentPermission() {
        this.to('ssm:ModifyDocumentPermission');
        return this;
    }
    /**
     * Grants permission to register a compliance type and other compliance details on a specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutComplianceItems.html
     */
    toPutComplianceItems() {
        this.to('ssm:PutComplianceItems');
        return this;
    }
    /**
     * Used by SSM Agent to generate a report of the results of specific agent requests (internal Systems Manager call)
     *
     * Access Level: Read
     */
    toPutConfigurePackageResult() {
        this.to('ssm:PutConfigurePackageResult');
        return this;
    }
    /**
     * Grants permission to add or update inventory items on multiple specified managed instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutInventory.html
     */
    toPutInventory() {
        this.to('ssm:PutInventory');
        return this;
    }
    /**
     * Grants permission to create an SSM parameter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutParameter.html
     */
    toPutParameter() {
        this.to('ssm:PutParameter');
        return this;
    }
    /**
     * Grants permission to specify the default patch baseline for an operating system type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterDefaultPatchBaseline.html
     */
    toRegisterDefaultPatchBaseline() {
        this.to('ssm:RegisterDefaultPatchBaseline');
        return this;
    }
    /**
     * Grants permission to specify the default patch baseline for a specified patch group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterPatchBaselineForPatchGroup.html
     */
    toRegisterPatchBaselineForPatchGroup() {
        this.to('ssm:RegisterPatchBaselineForPatchGroup');
        return this;
    }
    /**
     * Grants permission to register a target with a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterTargetWithMaintenanceWindow.html
     */
    toRegisterTargetWithMaintenanceWindow() {
        this.to('ssm:RegisterTargetWithMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to register a task with a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterTaskWithMaintenanceWindow.html
     */
    toRegisterTaskWithMaintenanceWindow() {
        this.to('ssm:RegisterTaskWithMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to remove a specified tag key from a specified resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.to('ssm:RemoveTagsFromResource');
        return this;
    }
    /**
     * Grants permission to reset the service setting for an AWS account to the default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ResetServiceSetting.html
     */
    toResetServiceSetting() {
        this.to('ssm:ResetServiceSetting');
        return this;
    }
    /**
     * Grants permission to reconnect a Session Manager session to a managed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ResumeSession.html
     */
    toResumeSession() {
        this.to('ssm:ResumeSession');
        return this;
    }
    /**
     * Grants permission to send a signal to change the current behavior or status of a specified Automation execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_SendAutomationSignal.html
     */
    toSendAutomationSignal() {
        this.to('ssm:SendAutomationSignal');
        return this;
    }
    /**
     * Grants permission to run commands on one or more specified managed instances
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_SendCommand.html
     */
    toSendCommand() {
        this.to('ssm:SendCommand');
        return this;
    }
    /**
     * Grants permission to run a specified association manually
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartAssociationsOnce.html
     */
    toStartAssociationsOnce() {
        this.to('ssm:StartAssociationsOnce');
        return this;
    }
    /**
     * Grants permission to initiate the execution of an Automation document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartAutomationExecution.html
     */
    toStartAutomationExecution() {
        this.to('ssm:StartAutomationExecution');
        return this;
    }
    /**
     * Grants permission to initiate a connection to a specified target for a Session Manager session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSessionDocumentAccessCheck()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartSession.html
     */
    toStartSession() {
        this.to('ssm:StartSession');
        return this;
    }
    /**
     * Grants permission to stop a specified Automation execution that is already in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StopAutomationExecution.html
     */
    toStopAutomationExecution() {
        this.to('ssm:StopAutomationExecution');
        return this;
    }
    /**
     * Grants permission to permanently end a Session Manager connection to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_TerminateSession.html
     */
    toTerminateSession() {
        this.to('ssm:TerminateSession');
        return this;
    }
    /**
     * Grants permission to update an association and immediately run the association on the specified targets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateAssociation.html
     */
    toUpdateAssociation() {
        this.to('ssm:UpdateAssociation');
        return this;
    }
    /**
     * Grants permission to update the status of the SSM document associated with a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateAssociationStatus.html
     */
    toUpdateAssociationStatus() {
        this.to('ssm:UpdateAssociationStatus');
        return this;
    }
    /**
     * Grants permission to update one or more values for an SSM document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateDocument.html
     */
    toUpdateDocument() {
        this.to('ssm:UpdateDocument');
        return this;
    }
    /**
     * Grants permission to change the default version of an SSM document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateDocumentDefaultVersion.html
     */
    toUpdateDocumentDefaultVersion() {
        this.to('ssm:UpdateDocumentDefaultVersion');
        return this;
    }
    /**
     * Used by SSM Agent to update the status of the association that it is currently running (internal Systems Manager call)
     *
     * Access Level: Write
     */
    toUpdateInstanceAssociationStatus() {
        this.to('ssm:UpdateInstanceAssociationStatus');
        return this;
    }
    /**
     * Used by SSM Agent to send a heartbeat signal to the Systems Manager service in the cloud
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toUpdateInstanceInformation() {
        this.to('ssm:UpdateInstanceInformation');
        return this;
    }
    /**
     * Grants permission to update a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindow.html
     */
    toUpdateMaintenanceWindow() {
        this.to('ssm:UpdateMaintenanceWindow');
        return this;
    }
    /**
     * Grants permission to update a specified maintenance window target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindowTarget.html
     */
    toUpdateMaintenanceWindowTarget() {
        this.to('ssm:UpdateMaintenanceWindowTarget');
        return this;
    }
    /**
     * Grants permission to update a specified maintenance window task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindowTask.html
     */
    toUpdateMaintenanceWindowTask() {
        this.to('ssm:UpdateMaintenanceWindowTask');
        return this;
    }
    /**
     * Grants permission to assign or change the IAM role assigned to a specified managed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateManagedInstanceRole.html
     */
    toUpdateManagedInstanceRole() {
        this.to('ssm:UpdateManagedInstanceRole');
        return this;
    }
    /**
     * Grants permission to edit or change an OpsItem
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateOpsItem.html
     */
    toUpdateOpsItem() {
        this.to('ssm:UpdateOpsItem');
        return this;
    }
    /**
     * Grants permission to update a specified patch baseline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdatePatchBaseline.html
     */
    toUpdatePatchBaseline() {
        this.to('ssm:UpdatePatchBaseline');
        return this;
    }
    /**
     * Grants permission to update a resource data sync
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateResourceDataSync.html
     */
    toUpdateResourceDataSync() {
        this.to('ssm:UpdateResourceDataSync');
        return this;
    }
    /**
     * Grants permission to update the service setting for an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateServiceSetting.html
     */
    toUpdateServiceSetting() {
        this.to('ssm:UpdateServiceSetting');
        return this;
    }
    /**
     * Adds a resource of type association to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-associations.html
     *
     * @param associationId - Identifier for the associationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAssociation(associationId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:association/${AssociationId}';
        arn = arn.replace('${AssociationId}', associationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type automation-execution to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-working.html
     *
     * @param automationExecutionId - Identifier for the automationExecutionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAutomationExecution(automationExecutionId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:automation-execution/${AutomationExecutionId}';
        arn = arn.replace('${AutomationExecutionId}', automationExecutionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type automation-definition to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-documents.html
     *
     * @param automationDefinitionName - Identifier for the automationDefinitionName.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAutomationDefinition(automationDefinitionName, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:automation-definition/${AutomationDefinitionName}:${VersionId}';
        arn = arn.replace('${AutomationDefinitionName}', automationDefinitionName);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type document to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-ssm-docs.html
     *
     * @param documentName - Identifier for the documentName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onDocument(documentName, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:document/${DocumentName}';
        arn = arn.replace('${DocumentName}', documentName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/iam-policy-structure.html#EC2_ARN_Format
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onInstance(instanceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:instance/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type maintenancewindow to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-maintenance.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onMaintenancewindow(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:maintenancewindow/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type managed-instance to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/what-is-systems-manager.html
     *
     * @param managedInstanceName - Identifier for the managedInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onManagedInstance(managedInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:managed-instance/${ManagedInstanceName}';
        arn = arn.replace('${ManagedInstanceName}', managedInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type managed-instance-inventory to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-inventory-configuring.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onManagedInstanceInventory(instanceId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:managed-instance-inventory/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type opsitem to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOpsitem(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:opsitem/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type parameter to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-parameter-store.html
     *
     * @param fullyQualifiedParameterName - Identifier for the fullyQualifiedParameterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onParameter(fullyQualifiedParameterName, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:parameter/${FullyQualifiedParameterName}';
        arn = arn.replace('${FullyQualifiedParameterName}', fullyQualifiedParameterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type patchbaseline to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-patch.html
     *
     * @param patchBaselineIdResourceId - Identifier for the patchBaselineIdResourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onPatchbaseline(patchBaselineIdResourceId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:patchbaseline/${PatchBaselineIdResourceId}';
        arn = arn.replace('${PatchBaselineIdResourceId}', patchBaselineIdResourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type session to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager.html
     *
     * @param sessionId - Identifier for the sessionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSession(sessionId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:session/${SessionId}';
        arn = arn.replace('${SessionId}', sessionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resourcedatasync to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-inventory-datasync.html
     *
     * @param syncName - Identifier for the syncName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onResourcedatasync(syncName, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:resource-data-sync/${SyncName}';
        arn = arn.replace('${SyncName}', syncName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type servicesetting to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/API_ServiceSetting.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onServicesetting(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:servicesetting/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type windowtarget to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-assign-targets.html
     *
     * @param windowTargetId - Identifier for the windowTargetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWindowtarget(windowTargetId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:windowtarget/${WindowTargetId}';
        arn = arn.replace('${WindowTargetId}', windowTargetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type windowtask to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-assign-tasks.html
     *
     * @param windowTaskId - Identifier for the windowTaskId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWindowtask(windowTaskId, account, region, partition) {
        var arn = 'arn:${Partition}:ssm:${Region}:${Account}:windowtask/${WindowTaskId}';
        arn = arn.replace('${WindowTaskId}', windowTaskId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by controlling whether the values for specified resources can be overwritten.
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#policy-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOverwrite(value, operator) {
        return this.if(`ssm:Overwrite`, value, operator || 'StringLike');
    }
    /**
     * Filters access for resources created in a hierarchical structure.
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#policy-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecursive(value, operator) {
        return this.if(`ssm:Recursive`, value, operator || 'StringLike');
    }
    /**
     * Filters access by verifying that a user has permission to access either the default Session Manager configuration document or the custom configuration document specified in a request.
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/getting-started-sessiondocumentaccesscheck.html
     *
     * Applies to actions:
     * - .toStartSession()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSessionDocumentAccessCheck(value) {
        return this.if(`ssm:SessionDocumentAccessCheck`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by verifying that a user also has access to the ResourceDataSync SyncType specified in the request
     *
     * Applies to actions:
     * - .toCreateResourceDataSync()
     * - .toDeleteResourceDataSync()
     * - .toListResourceDataSync()
     * - .toUpdateResourceDataSync()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSyncType(value, operator) {
        return this.if(`ssm:SyncType`, value, operator || 'StringLike');
    }
    /**
     * Filters access based on a tag key-value pair assigned to the Systems Manager resource
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#policy-conditions
     *
     * Applies to actions:
     * - .toSendCommand()
     *
     * Applies to resource types:
     * - document
     * - instance
     * - maintenancewindow
     * - managed-instance
     * - parameter
     * - patchbaseline
     *
     * @param tagkey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagkey, value, operator) {
        return this.if(`ssm:resourceTag/${tagkey}`, value, operator || 'StringLike');
    }
}
exports.Ssm = Ssm;
//# sourceMappingURL=data:application/json;base64,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