"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transcribe = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [transcribe](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazontranscribe.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Transcribe extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [transcribe](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazontranscribe.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'transcribe';
        this.accessLevelList = {
            "Write": [
                "CreateLanguageModel",
                "CreateMedicalVocabulary",
                "CreateVocabulary",
                "CreateVocabularyFilter",
                "DeleteLanguageModel",
                "DeleteMedicalTranscriptionJob",
                "DeleteMedicalVocabulary",
                "DeleteTranscriptionJob",
                "DeleteVocabulary",
                "DeleteVocabularyFilter",
                "StartMedicalStreamTranscription",
                "StartMedicalStreamTranscriptionWebSocket",
                "StartMedicalTranscriptionJob",
                "StartStreamTranscription",
                "StartStreamTranscriptionWebSocket",
                "StartTranscriptionJob",
                "UpdateMedicalVocabulary",
                "UpdateVocabulary",
                "UpdateVocabularyFilter"
            ],
            "Read": [
                "DescribeLanguageModel",
                "GetMedicalTranscriptionJob",
                "GetMedicalVocabulary",
                "GetTranscriptionJob",
                "GetVocabulary",
                "GetVocabularyFilter"
            ],
            "List": [
                "ListLanguageModels",
                "ListMedicalTranscriptionJobs",
                "ListMedicalVocabularies",
                "ListTranscriptionJobs",
                "ListVocabularies",
                "ListVocabularyFilters"
            ]
        };
    }
    /**
     * Grants permission to create a new custom language model.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateLanguageModel.html
     */
    toCreateLanguageModel() {
        this.to('transcribe:CreateLanguageModel');
        return this;
    }
    /**
     * Grants permission to create a new custom vocabulary that you can use to change the way Amazon Transcribe Medical handles transcription of an audio file.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateMedicalVocabulary.html
     */
    toCreateMedicalVocabulary() {
        this.to('transcribe:CreateMedicalVocabulary');
        return this;
    }
    /**
     * Grants permission to create a new custom vocabulary that you can use to change the way Amazon Transcribe handles transcription of an audio file.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabulary.html
     */
    toCreateVocabulary() {
        this.to('transcribe:CreateVocabulary');
        return this;
    }
    /**
     * Grants permission to create a new vocabulary filter that you can use to filter out words from the transcription of an audio file generated by Amazon Transcribe
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabularyFilter.html
     */
    toCreateVocabularyFilter() {
        this.to('transcribe:CreateVocabularyFilter');
        return this;
    }
    /**
     * Grants permission to delete a previously created custom language model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteLanguageModel.html
     */
    toDeleteLanguageModel() {
        this.to('transcribe:DeleteLanguageModel');
        return this;
    }
    /**
     * Grants permission to delete a previously submitted medical transcription job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalTranscriptionJob.html
     */
    toDeleteMedicalTranscriptionJob() {
        this.to('transcribe:DeleteMedicalTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to delete a medical vocabulary from Amazon Transcribe.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalVocabulary.html
     */
    toDeleteMedicalVocabulary() {
        this.to('transcribe:DeleteMedicalVocabulary');
        return this;
    }
    /**
     * Grants permission to delete a previously submitted transcription job along with any other generated results such as the transcription, models, and so on.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteTranscriptionJob.html
     */
    toDeleteTranscriptionJob() {
        this.to('transcribe:DeleteTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to delete a vocabulary from Amazon Transcribe.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteVocabulary.html
     */
    toDeleteVocabulary() {
        this.to('transcribe:DeleteVocabulary');
        return this;
    }
    /**
     * Grants permission to delete a vocabulary filter from Amazon Transcribe.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteVocabularyFilter.html
     */
    toDeleteVocabularyFilter() {
        this.to('transcribe:DeleteVocabularyFilter');
        return this;
    }
    /**
     * Grants permission to return information about a custom language model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DescribeLanguageModel.html
     */
    toDescribeLanguageModel() {
        this.to('transcribe:DescribeLanguageModel');
        return this;
    }
    /**
     * Grants permission to return information about a medical transcription job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalTranscriptionJob.html
     */
    toGetMedicalTranscriptionJob() {
        this.to('transcribe:GetMedicalTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to get information about a medical vocabulary.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalVocabulary.html
     */
    toGetMedicalVocabulary() {
        this.to('transcribe:GetMedicalVocabulary');
        return this;
    }
    /**
     * Grants permission to return information about a transcription job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetTranscriptionJob.html
     */
    toGetTranscriptionJob() {
        this.to('transcribe:GetTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to to get information about a vocabulary.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetVocabulary.html
     */
    toGetVocabulary() {
        this.to('transcribe:GetVocabulary');
        return this;
    }
    /**
     * Grants permission to get information about a vocabulary filter.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetVocabularyFilter.html
     */
    toGetVocabularyFilter() {
        this.to('transcribe:GetVocabularyFilter');
        return this;
    }
    /**
     * Grants permission to list custom language models.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListLanguageModels.html
     */
    toListLanguageModels() {
        this.to('transcribe:ListLanguageModels');
        return this;
    }
    /**
     * Grants permission to list medical transcription jobs with the specified status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalTranscriptionJobs.html
     */
    toListMedicalTranscriptionJobs() {
        this.to('transcribe:ListMedicalTranscriptionJobs');
        return this;
    }
    /**
     * Grants permission to return a list of medical vocabularies that match the specified criteria. If no criteria are specified, returns the entire list of vocabularies.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalVocabularies.html
     */
    toListMedicalVocabularies() {
        this.to('transcribe:ListMedicalVocabularies');
        return this;
    }
    /**
     * Grants permission to list transcription jobs with the specified status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListTranscriptionJobs.html
     */
    toListTranscriptionJobs() {
        this.to('transcribe:ListTranscriptionJobs');
        return this;
    }
    /**
     * Grants permission to return a list of vocabularies that match the specified criteria. If no criteria are specified, returns the entire list of vocabularies.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListVocabularies.html
     */
    toListVocabularies() {
        this.to('transcribe:ListVocabularies');
        return this;
    }
    /**
     * Grants permission to return a list of vocabulary filters that match the specified criteria. If no criteria are specified, returns the at most 5 vocabulary filters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListVocabularyFilters.html
     */
    toListVocabularyFilters() {
        this.to('transcribe:ListVocabularyFilters');
        return this;
    }
    /**
     * Grants permission to start a protocol where audio is streamed to Transcribe Medical and the transcription results are streamed to your application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalStreamTranscription.html
     */
    toStartMedicalStreamTranscription() {
        this.to('transcribe:StartMedicalStreamTranscription');
        return this;
    }
    /**
     * Grants permission to start a WebSocket where audio is streamed to Transcribe Medical and the transcription results are streamed to your application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalStreamTranscriptionWebSocket.html
     */
    toStartMedicalStreamTranscriptionWebSocket() {
        this.to('transcribe:StartMedicalStreamTranscriptionWebSocket');
        return this;
    }
    /**
     * Grants permission to start an asynchronous job to transcribe medical speech to text.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartMedicalTranscriptionJob.html
     */
    toStartMedicalTranscriptionJob() {
        this.to('transcribe:StartMedicalTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to start a bidirectional HTTP2 stream to transcribe speech to text in real time.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartStreamTranscription.html
     */
    toStartStreamTranscription() {
        this.to('transcribe:StartStreamTranscription');
        return this;
    }
    /**
     * Grants permission to start a websocket stream to transcribe speech to text in real time.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartStreamTranscriptionWebSocket.html
     */
    toStartStreamTranscriptionWebSocket() {
        this.to('transcribe:StartStreamTranscriptionWebSocket');
        return this;
    }
    /**
     * Grants permission to start an asynchronous job to transcribe speech to text.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOutputBucketName()
     * - .ifOutputEncryptionKMSKeyId()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartTranscriptionJob.html
     */
    toStartTranscriptionJob() {
        this.to('transcribe:StartTranscriptionJob');
        return this;
    }
    /**
     * Grants permission to update an existing medical vocabulary with new values. The UpdateMedicalVocabulary operation overwrites all of the existing information with the values that you provide in the request.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateMedicalVocabulary.html
     */
    toUpdateMedicalVocabulary() {
        this.to('transcribe:UpdateMedicalVocabulary');
        return this;
    }
    /**
     * Grants permission to update an existing vocabulary with new values. The UpdateVocabulary operation overwrites all of the existing information with the values that you provide in the request.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateVocabulary.html
     */
    toUpdateVocabulary() {
        this.to('transcribe:UpdateVocabulary');
        return this;
    }
    /**
     * Grants permission to update an existing vocabulary filter with new values. The UpdateVocabularyFilter operation overwrites all of the existing information with the values that you provide in the request.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateVocabularyFilter.html
     */
    toUpdateVocabularyFilter() {
        this.to('transcribe:UpdateVocabularyFilter');
        return this;
    }
    /**
     * Enables you to control access based on the output bucket name included in the request
     *
     * Applies to actions:
     * - .toStartTranscriptionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputBucketName(value, operator) {
        return this.if(`transcribe:OutputBucketName`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the KMS key id included in the request
     *
     * Applies to actions:
     * - .toStartTranscriptionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputEncryptionKMSKeyId(value, operator) {
        return this.if(`transcribe:OutputEncryptionKMSKeyId`, value, operator || 'StringLike');
    }
}
exports.Transcribe = Transcribe;
//# sourceMappingURL=data:application/json;base64,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