"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Worklink = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [worklink](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworklink.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Worklink extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [worklink](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworklink.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'worklink';
        this.accessLevelList = {
            "Write": [
                "AssociateDomain",
                "AssociateWebsiteAuthorizationProvider",
                "AssociateWebsiteCertificateAuthority",
                "CreateFleet",
                "DeleteFleet",
                "DisassociateDomain",
                "DisassociateWebsiteAuthorizationProvider",
                "DisassociateWebsiteCertificateAuthority",
                "RestoreDomainAccess",
                "RevokeDomainAccess",
                "SignOutUser",
                "UpdateAuditStreamConfiguration",
                "UpdateCompanyNetworkConfiguration",
                "UpdateDevicePolicyConfiguration",
                "UpdateDomainMetadata",
                "UpdateFleetMetadata",
                "UpdateIdentityProviderConfiguration"
            ],
            "Read": [
                "DescribeAuditStreamConfiguration",
                "DescribeCompanyNetworkConfiguration",
                "DescribeDevice",
                "DescribeDevicePolicyConfiguration",
                "DescribeDomain",
                "DescribeFleetMetadata",
                "DescribeIdentityProviderConfiguration",
                "DescribeWebsiteCertificateAuthority",
                "ListTagsForResource"
            ],
            "List": [
                "ListDevices",
                "ListDomains",
                "ListFleets",
                "ListWebsiteAuthorizationProviders",
                "ListWebsiteCertificateAuthorities"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to associate a domain with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_AssociateDomain.html
     */
    toAssociateDomain() {
        this.to('worklink:AssociateDomain');
        return this;
    }
    /**
     * Grants permission to associate a website authorization provider with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_AssociateWebsiteAuthorizationProvider.html
     */
    toAssociateWebsiteAuthorizationProvider() {
        this.to('worklink:AssociateWebsiteAuthorizationProvider');
        return this;
    }
    /**
     * Grants permission to associate a website certificate authority with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_AssociateWebsiteCertificateAuthority.html
     */
    toAssociateWebsiteCertificateAuthority() {
        this.to('worklink:AssociateWebsiteCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to create an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_CreateFleet.html
     */
    toCreateFleet() {
        this.to('worklink:CreateFleet');
        return this;
    }
    /**
     * Grants permission to delete an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DeleteFleet.html
     */
    toDeleteFleet() {
        this.to('worklink:DeleteFleet');
        return this;
    }
    /**
     * Grants permission to describe the audit stream configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeAuditStreamConfiguration.html
     */
    toDescribeAuditStreamConfiguration() {
        this.to('worklink:DescribeAuditStreamConfiguration');
        return this;
    }
    /**
     * Grants permission to describe the company network configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeCompanyNetworkConfiguration.html
     */
    toDescribeCompanyNetworkConfiguration() {
        this.to('worklink:DescribeCompanyNetworkConfiguration');
        return this;
    }
    /**
     * Grants permission to describe details of a device associated with an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDevice.html
     */
    toDescribeDevice() {
        this.to('worklink:DescribeDevice');
        return this;
    }
    /**
     * Grants permission to describe the device policy configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDevicePolicyConfiguration.html
     */
    toDescribeDevicePolicyConfiguration() {
        this.to('worklink:DescribeDevicePolicyConfiguration');
        return this;
    }
    /**
     * Grants permission to describe details about a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDomain.html
     */
    toDescribeDomain() {
        this.to('worklink:DescribeDomain');
        return this;
    }
    /**
     * Grants permission to describe metadata of an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeFleetMetadata.html
     */
    toDescribeFleetMetadata() {
        this.to('worklink:DescribeFleetMetadata');
        return this;
    }
    /**
     * Grants permission to describe the identity provider configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeIdentityProviderConfiguration.html
     */
    toDescribeIdentityProviderConfiguration() {
        this.to('worklink:DescribeIdentityProviderConfiguration');
        return this;
    }
    /**
     * Grants permission to describe a website certificate authority associated with an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeWebsiteCertificateAuthority.html
     */
    toDescribeWebsiteCertificateAuthority() {
        this.to('worklink:DescribeWebsiteCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to disassociate a domain from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateDomain.html
     */
    toDisassociateDomain() {
        this.to('worklink:DisassociateDomain');
        return this;
    }
    /**
     * Grants permission to disassociate a website authorization provider from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateWebsiteAuthorizationProvider.html
     */
    toDisassociateWebsiteAuthorizationProvider() {
        this.to('worklink:DisassociateWebsiteAuthorizationProvider');
        return this;
    }
    /**
     * Grants permission to disassociate a website certificate authority from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateWebsiteCertificateAuthority.html
     */
    toDisassociateWebsiteCertificateAuthority() {
        this.to('worklink:DisassociateWebsiteCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to list the devices associated with an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListDevices.html
     */
    toListDevices() {
        this.to('worklink:ListDevices');
        return this;
    }
    /**
     * Grants permission to list the associated domains for an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListDomains.html
     */
    toListDomains() {
        this.to('worklink:ListDomains');
        return this;
    }
    /**
     * Grants permission to list the Amazon WorkLink fleets associated with the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListFleets.html
     */
    toListFleets() {
        this.to('worklink:ListFleets');
        return this;
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('worklink:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list the website authorization providers for an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListWebsiteAuthorizationProviders.html
     */
    toListWebsiteAuthorizationProviders() {
        this.to('worklink:ListWebsiteAuthorizationProviders');
        return this;
    }
    /**
     * Grants permission to list the website certificate authorities associated with an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListWebsiteCertificateAuthorities.html
     */
    toListWebsiteCertificateAuthorities() {
        this.to('worklink:ListWebsiteCertificateAuthorities');
        return this;
    }
    /**
     * Grants permission to restore access to a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_RestoreDomainAccess.html
     */
    toRestoreDomainAccess() {
        this.to('worklink:RestoreDomainAccess');
        return this;
    }
    /**
     * Grants permission to revoke access to a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_RevokeDomainAccess.html
     */
    toRevokeDomainAccess() {
        this.to('worklink:RevokeDomainAccess');
        return this;
    }
    /**
     * Grants permission to sign out a user from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_SignOutUser.html
     */
    toSignOutUser() {
        this.to('worklink:SignOutUser');
        return this;
    }
    /**
     * Grants permission to add one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.to('worklink:TagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.to('worklink:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the audit stream configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateAuditStreamConfiguration.html
     */
    toUpdateAuditStreamConfiguration() {
        this.to('worklink:UpdateAuditStreamConfiguration');
        return this;
    }
    /**
     * Grants permission to update the company network configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateCompanyNetworkConfiguration.html
     */
    toUpdateCompanyNetworkConfiguration() {
        this.to('worklink:UpdateCompanyNetworkConfiguration');
        return this;
    }
    /**
     * Grants permission to update the device policy configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateDevicePolicyConfiguration.html
     */
    toUpdateDevicePolicyConfiguration() {
        this.to('worklink:UpdateDevicePolicyConfiguration');
        return this;
    }
    /**
     * Grants permission to update the metadata for a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateDomainMetadata.html
     */
    toUpdateDomainMetadata() {
        this.to('worklink:UpdateDomainMetadata');
        return this;
    }
    /**
     * Grants permission to update the metadata of an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateFleetMetadata.html
     */
    toUpdateFleetMetadata() {
        this.to('worklink:UpdateFleetMetadata');
        return this;
    }
    /**
     * Grants permission to update the identity provider configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateIdentityProviderConfiguration.html
     */
    toUpdateIdentityProviderConfiguration() {
        this.to('worklink:UpdateIdentityProviderConfiguration');
        return this;
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/worklink/latest/api/worklink-resources.html#Fleet
     *
     * @param fleetName - Identifier for the fleetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFleet(fleetName, account, partition) {
        var arn = 'arn:${Partition}:worklink::${Account}:fleet/${FleetName}';
        arn = arn.replace('${FleetName}', fleetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Worklink = Worklink;
//# sourceMappingURL=data:application/json;base64,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