"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workmail = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [workmail](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkmail.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Workmail extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [workmail](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkmail.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'workmail';
        this.accessLevelList = {
            "Write": [
                "AddMembersToGroup",
                "AssociateDelegateToResource",
                "AssociateMemberToGroup",
                "CancelMailboxExportJob",
                "CreateAlias",
                "CreateGroup",
                "CreateInboundMailFlowRule",
                "CreateMailDomain",
                "CreateMailUser",
                "CreateOrganization",
                "CreateOutboundMailFlowRule",
                "CreateResource",
                "CreateSmtpGateway",
                "CreateUser",
                "DeleteAccessControlRule",
                "DeleteAlias",
                "DeleteGroup",
                "DeleteInboundMailFlowRule",
                "DeleteMailDomain",
                "DeleteMailboxPermissions",
                "DeleteMobileDevice",
                "DeleteOrganization",
                "DeleteOutboundMailFlowRule",
                "DeleteResource",
                "DeleteRetentionPolicy",
                "DeleteSmtpGateway",
                "DeleteUser",
                "DeregisterFromWorkMail",
                "DisableMailGroups",
                "DisableMailUsers",
                "DisassociateDelegateFromResource",
                "DisassociateMemberFromGroup",
                "EnableMailDomain",
                "EnableMailGroups",
                "EnableMailUsers",
                "PutAccessControlRule",
                "PutMailboxPermissions",
                "PutRetentionPolicy",
                "RegisterToWorkMail",
                "RemoveMembersFromGroup",
                "ResetPassword",
                "ResetUserPassword",
                "SetAdmin",
                "SetDefaultMailDomain",
                "SetJournalingRules",
                "SetMailGroupDetails",
                "SetMailUserDetails",
                "SetMobilePolicyDetails",
                "StartMailboxExportJob",
                "TestInboundMailFlowRules",
                "TestOutboundMailFlowRules",
                "UpdateInboundMailFlowRule",
                "UpdateMailboxQuota",
                "UpdateOutboundMailFlowRule",
                "UpdatePrimaryEmailAddress",
                "UpdateResource",
                "UpdateSmtpGateway",
                "WipeMobileDevice"
            ],
            "List": [
                "DescribeDirectories",
                "DescribeGroup",
                "DescribeKmsKeys",
                "DescribeMailDomains",
                "DescribeMailGroups",
                "DescribeMailUsers",
                "DescribeOrganization",
                "DescribeOrganizations",
                "DescribeResource",
                "DescribeUser",
                "ListAccessControlRules",
                "ListAliases",
                "ListGroupMembers",
                "ListGroups",
                "ListInboundMailFlowRules",
                "ListMailboxExportJobs",
                "ListMailboxPermissions",
                "ListOrganizations",
                "ListOutboundMailFlowRules",
                "ListResourceDelegates",
                "ListResources",
                "ListSmtpGateways",
                "ListTagsForResource",
                "ListUsers"
            ],
            "Read": [
                "DescribeInboundMailFlowRule",
                "DescribeMailboxExportJob",
                "DescribeOutboundMailFlowRule",
                "DescribeSmtpGateway",
                "GetAccessControlEffect",
                "GetDefaultRetentionPolicy",
                "GetJournalingRules",
                "GetMailDomainDetails",
                "GetMailGroupDetails",
                "GetMailUserDetails",
                "GetMailboxDetails",
                "GetMobileDeviceDetails",
                "GetMobileDevicesForUser",
                "GetMobilePolicyDetails",
                "ListMembersInMailGroup",
                "SearchMembers"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Adds a list of members (users or groups) to a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toAddMembersToGroup() {
        this.to('workmail:AddMembersToGroup');
        return this;
    }
    /**
     * Adds a member (user or group) to the resource's set of delegates.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_AssociateDelegateToResource.html
     */
    toAssociateDelegateToResource() {
        this.to('workmail:AssociateDelegateToResource');
        return this;
    }
    /**
     * Adds a member (user or group) to the group's set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_AssociateMemberToGroup.html
     */
    toAssociateMemberToGroup() {
        this.to('workmail:AssociateMemberToGroup');
        return this;
    }
    /**
     * Cancels a currently running mailbox export job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CancelMailboxExportJob.html
     */
    toCancelMailboxExportJob() {
        this.to('workmail:CancelMailboxExportJob');
        return this;
    }
    /**
     * Adds an alias to the set of a given member (user or group) of WorkMail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateAlias.html
     */
    toCreateAlias() {
        this.to('workmail:CreateAlias');
        return this;
    }
    /**
     * Creates a group that can be used in WorkMail by calling the RegisterToWorkMail operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        this.to('workmail:CreateGroup');
        return this;
    }
    /**
     * Create an inbound email flow rule which will apply to all email sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/create-email-rules.html
     */
    toCreateInboundMailFlowRule() {
        this.to('workmail:CreateInboundMailFlowRule');
        return this;
    }
    /**
     * Creates a mail domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_domain.html
     */
    toCreateMailDomain() {
        this.to('workmail:CreateMailDomain');
        return this;
    }
    /**
     * Creates a user in the directory and the WorkMail storage but does not enable the user for mail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html
     */
    toCreateMailUser() {
        this.to('workmail:CreateMailUser');
        return this;
    }
    /**
     * Creates a new Amazon WorkMail organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateOrganization.html
     */
    toCreateOrganization() {
        this.to('workmail:CreateOrganization');
        return this;
    }
    /**
     * Create an outbound email flow rule which will apply to all email sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/create-email-rules.html
     */
    toCreateOutboundMailFlowRule() {
        this.to('workmail:CreateOutboundMailFlowRule');
        return this;
    }
    /**
     * Creates a new WorkMail resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateResource.html
     */
    toCreateResource() {
        this.to('workmail:CreateResource');
        return this;
    }
    /**
     * Register an SMTP device against a WorkMail organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toCreateSmtpGateway() {
        this.to('workmail:CreateSmtpGateway');
        return this;
    }
    /**
     * Creates a user who can be used in WorkMail by calling the RegisterToWorkMail operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.to('workmail:CreateUser');
        return this;
    }
    /**
     * Deletes an access control rule for the specified WorkMail organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteAccessControlRule.html
     */
    toDeleteAccessControlRule() {
        this.to('workmail:DeleteAccessControlRule');
        return this;
    }
    /**
     * Remove one or more specified aliases from a set of aliases for a given user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteAlias.html
     */
    toDeleteAlias() {
        this.to('workmail:DeleteAlias');
        return this;
    }
    /**
     * Deletes a group from WorkMail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        this.to('workmail:DeleteGroup');
        return this;
    }
    /**
     * Remove an inbound email flow rule to no longer apply to emails sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove-email-flow-rule.html
     */
    toDeleteInboundMailFlowRule() {
        this.to('workmail:DeleteInboundMailFlowRule');
        return this;
    }
    /**
     * Removes an unused mail domain from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove_domain.html
     */
    toDeleteMailDomain() {
        this.to('workmail:DeleteMailDomain');
        return this;
    }
    /**
     * Deletes permissions granted to a member (user or group).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteMailboxPermissions.html
     */
    toDeleteMailboxPermissions() {
        this.to('workmail:DeleteMailboxPermissions');
        return this;
    }
    /**
     * Removes a mobile device from a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html#remove_mobile_device
     */
    toDeleteMobileDevice() {
        this.to('workmail:DeleteMobileDevice');
        return this;
    }
    /**
     * Deletes an Amazon WorkMail organization and all underlying AWS resources managed by Amazon WorkMail as part of the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteOrganization.html
     */
    toDeleteOrganization() {
        this.to('workmail:DeleteOrganization');
        return this;
    }
    /**
     * Remove an outbound email flow rule to no longer apply to emails sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove-email-flow-rule.html
     */
    toDeleteOutboundMailFlowRule() {
        this.to('workmail:DeleteOutboundMailFlowRule');
        return this;
    }
    /**
     * Deletes the specified resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteResource.html
     */
    toDeleteResource() {
        this.to('workmail:DeleteResource');
        return this;
    }
    /**
     * Deletes the retention policy based on the supplied organization and policy identifiers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteRetentionPolicy.html
     */
    toDeleteRetentionPolicy() {
        this.to('workmail:DeleteRetentionPolicy');
        return this;
    }
    /**
     * Remove an SMTP device from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toDeleteSmtpGateway() {
        this.to('workmail:DeleteSmtpGateway');
        return this;
    }
    /**
     * Deletes a user from WorkMail and all subsequent systems. The action cannot be undone.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        this.to('workmail:DeleteUser');
        return this;
    }
    /**
     * Mark a user, group, or resource as no longer used in WorkMail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeregisterFromWorkMail.html
     */
    toDeregisterFromWorkMail() {
        this.to('workmail:DeregisterFromWorkMail');
        return this;
    }
    /**
     * Shows a list of directories available for use in creating an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_organization.html
     */
    toDescribeDirectories() {
        this.to('workmail:DescribeDirectories');
        return this;
    }
    /**
     * Returns the data available for the group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeGroup.html
     */
    toDescribeGroup() {
        this.to('workmail:DescribeGroup');
        return this;
    }
    /**
     * Returns the details of an inbound mail flow rule configured for an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-actions
     */
    toDescribeInboundMailFlowRule() {
        this.to('workmail:DescribeInboundMailFlowRule');
        return this;
    }
    /**
     * Shows a list of KMS Keys available for use in creating an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_organization.html
     */
    toDescribeKmsKeys() {
        this.to('workmail:DescribeKmsKeys');
        return this;
    }
    /**
     * Shows the details of all mail domains associated with the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/domains_overview.html
     */
    toDescribeMailDomains() {
        this.to('workmail:DescribeMailDomains');
        return this;
    }
    /**
     * Shows the details of all groups associated with the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toDescribeMailGroups() {
        this.to('workmail:DescribeMailGroups');
        return this;
    }
    /**
     * Shows the details of all users associated with the orgaization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html
     */
    toDescribeMailUsers() {
        this.to('workmail:DescribeMailUsers');
        return this;
    }
    /**
     * Retrieve details of a mailbox export job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeMailboxExportJob.html
     */
    toDescribeMailboxExportJob() {
        this.to('workmail:DescribeMailboxExportJob');
        return this;
    }
    /**
     * Provides more information regarding a given organization based on its identifier.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeOrganization.html
     */
    toDescribeOrganization() {
        this.to('workmail:DescribeOrganization');
        return this;
    }
    /**
     * Shows a summary of all organizations associated with the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/organizations_overview.html
     */
    toDescribeOrganizations() {
        this.to('workmail:DescribeOrganizations');
        return this;
    }
    /**
     * Returns the details of an outbound mail flow rule configured for an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-outbound
     */
    toDescribeOutboundMailFlowRule() {
        this.to('workmail:DescribeOutboundMailFlowRule');
        return this;
    }
    /**
     * Returns the data available for the resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeResource.html
     */
    toDescribeResource() {
        this.to('workmail:DescribeResource');
        return this;
    }
    /**
     * Returns the details of an SMTP device registered against an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toDescribeSmtpGateway() {
        this.to('workmail:DescribeSmtpGateway');
        return this;
    }
    /**
     * Provides information regarding the user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeUser.html
     */
    toDescribeUser() {
        this.to('workmail:DescribeUser');
        return this;
    }
    /**
     * Disable a mail group when it is not being used and, to allow it to be deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove_group.html
     */
    toDisableMailGroups() {
        this.to('workmail:DisableMailGroups');
        return this;
    }
    /**
     * Disable a user mailbox when it is no longer being used, and to allow it to be deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-mailboxes.html#delete_user_mailbox
     */
    toDisableMailUsers() {
        this.to('workmail:DisableMailUsers');
        return this;
    }
    /**
     * Removes a member from the resource's set of delegates.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DisassociateDelegateFromResource.html
     */
    toDisassociateDelegateFromResource() {
        this.to('workmail:DisassociateDelegateFromResource');
        return this;
    }
    /**
     * Removes a member from a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DisassociateMemberFromGroup.html
     */
    toDisassociateMemberFromGroup() {
        this.to('workmail:DisassociateMemberFromGroup');
        return this;
    }
    /**
     * Enable a mail domain in the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_domain.html
     */
    toEnableMailDomain() {
        this.to('workmail:EnableMailDomain');
        return this;
    }
    /**
     * Enable a mail group after it has been created to allow it to receive mail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/enable_existing_group.html
     */
    toEnableMailGroups() {
        this.to('workmail:EnableMailGroups');
        return this;
    }
    /**
     * Enable a user's mailbox after it has been created to allow it to receive mail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html#enable_existing_user
     */
    toEnableMailUsers() {
        this.to('workmail:EnableMailUsers');
        return this;
    }
    /**
     * Gets the effects of an organization's access control rules as they apply to a specified IPv4 address, access protocol action, or user ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetAccessControlEffect.html
     */
    toGetAccessControlEffect() {
        this.to('workmail:GetAccessControlEffect');
        return this;
    }
    /**
     * Retrieves the retention policy associated at an organizational level.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetDefaultRetentionPolicy.html
     */
    toGetDefaultRetentionPolicy() {
        this.to('workmail:GetDefaultRetentionPolicy');
        return this;
    }
    /**
     * Returns journaling and fallback email addresses configured for email journaling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/journaling_overview.html
     */
    toGetJournalingRules() {
        this.to('workmail:GetJournalingRules');
        return this;
    }
    /**
     * Get the details of the mail domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/domains_overview.html
     */
    toGetMailDomainDetails() {
        this.to('workmail:GetMailDomainDetails');
        return this;
    }
    /**
     * Get the details of the mail group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toGetMailGroupDetails() {
        this.to('workmail:GetMailGroupDetails');
        return this;
    }
    /**
     * Get the details of the user's mailbox and account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html
     */
    toGetMailUserDetails() {
        this.to('workmail:GetMailUserDetails');
        return this;
    }
    /**
     * Returns the details of the user's mailbox.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetMailboxDetails.html
     */
    toGetMailboxDetails() {
        this.to('workmail:GetMailboxDetails');
        return this;
    }
    /**
     * Get the details of the mobile device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html
     */
    toGetMobileDeviceDetails() {
        this.to('workmail:GetMobileDeviceDetails');
        return this;
    }
    /**
     * Get a list of the mobile devices associated with the user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html
     */
    toGetMobileDevicesForUser() {
        this.to('workmail:GetMobileDevicesForUser');
        return this;
    }
    /**
     * Get the details of the mobile device policy associated with the organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/edit_organization_mobile_policy.html
     */
    toGetMobilePolicyDetails() {
        this.to('workmail:GetMobilePolicyDetails');
        return this;
    }
    /**
     * Lists the access control rules for the specified organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListAccessControlRules.html
     */
    toListAccessControlRules() {
        this.to('workmail:ListAccessControlRules');
        return this;
    }
    /**
     * Creates a paginated call to list the aliases associated with a given entity.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListAliases.html
     */
    toListAliases() {
        this.to('workmail:ListAliases');
        return this;
    }
    /**
     * Returns an overview of the members of a group. Users and groups can be members of a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListGroupMembers.html
     */
    toListGroupMembers() {
        this.to('workmail:ListGroupMembers');
        return this;
    }
    /**
     * Returns summaries of the organization's groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListGroups.html
     */
    toListGroups() {
        this.to('workmail:ListGroups');
        return this;
    }
    /**
     * Returns a list of inbound mail flow rules configured for an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-actions
     */
    toListInboundMailFlowRules() {
        this.to('workmail:ListInboundMailFlowRules');
        return this;
    }
    /**
     * List mailbox export jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListMailboxExportJobs.html
     */
    toListMailboxExportJobs() {
        this.to('workmail:ListMailboxExportJobs');
        return this;
    }
    /**
     * Lists the mailbox permissions associated with a user, group, or resource mailbox.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListMailboxPermissions.html
     */
    toListMailboxPermissions() {
        this.to('workmail:ListMailboxPermissions');
        return this;
    }
    /**
     * Get a list of all the members in a mail group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toListMembersInMailGroup() {
        this.to('workmail:ListMembersInMailGroup');
        return this;
    }
    /**
     * Returns summaries of the customer's non-deleted organizations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListOrganizations.html
     */
    toListOrganizations() {
        this.to('workmail:ListOrganizations');
        return this;
    }
    /**
     * Returns a list of outbound mail flow rules configured for an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-outbound
     */
    toListOutboundMailFlowRules() {
        this.to('workmail:ListOutboundMailFlowRules');
        return this;
    }
    /**
     * Lists the delegates associated with a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListResourceDelegates.html
     */
    toListResourceDelegates() {
        this.to('workmail:ListResourceDelegates');
        return this;
    }
    /**
     * Returns summaries of the organization's resources.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListResources.html
     */
    toListResources() {
        this.to('workmail:ListResources');
        return this;
    }
    /**
     * Returns a list of SMTP devices registered against the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toListSmtpGateways() {
        this.to('workmail:ListSmtpGateways');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon WorkMail organization resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('workmail:ListTagsForResource');
        return this;
    }
    /**
     * Returns summaries of the organization's users.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        this.to('workmail:ListUsers');
        return this;
    }
    /**
     * Adds a new access control rule for the specified organization. The rule allows or denies access to the organization for the specified IPv4 addresses, access protocol actions, and user IDs. Adding a new rule with the same name as an existing rule replaces the older rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutAccessControlRule.html
     */
    toPutAccessControlRule() {
        this.to('workmail:PutAccessControlRule');
        return this;
    }
    /**
     * Sets permissions for a user, group, or resource. This replaces any pre-existing permissions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutMailboxPermissions.html
     */
    toPutMailboxPermissions() {
        this.to('workmail:PutMailboxPermissions');
        return this;
    }
    /**
     * Adds or updates the retention policy for the specified organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutRetentionPolicy.html
     */
    toPutRetentionPolicy() {
        this.to('workmail:PutRetentionPolicy');
        return this;
    }
    /**
     * Registers an existing and disabled user, group, or resource for use by associating a mailbox and calendaring capabilities.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_RegisterToWorkMail.html
     */
    toRegisterToWorkMail() {
        this.to('workmail:RegisterToWorkMail');
        return this;
    }
    /**
     * Remove members from a mail group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toRemoveMembersFromGroup() {
        this.to('workmail:RemoveMembersFromGroup');
        return this;
    }
    /**
     * Allows the administrator to reset the password for a user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ResetPassword.html
     */
    toResetPassword() {
        this.to('workmail:ResetPassword');
        return this;
    }
    /**
     * Reset the password for a user's account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html#reset_user_password
     */
    toResetUserPassword() {
        this.to('workmail:ResetUserPassword');
        return this;
    }
    /**
     * Prefix search to find a specific user in a mail group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toSearchMembers() {
        this.to('workmail:SearchMembers');
        return this;
    }
    /**
     * Mark a user as being an administrator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html
     */
    toSetAdmin() {
        this.to('workmail:SetAdmin');
        return this;
    }
    /**
     * Set the default mail domain for the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/default_domain.html
     */
    toSetDefaultMailDomain() {
        this.to('workmail:SetDefaultMailDomain');
        return this;
    }
    /**
     * Set journaling and fallback email addresses for email journaling
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/journaling_overview.html
     */
    toSetJournalingRules() {
        this.to('workmail:SetJournalingRules');
        return this;
    }
    /**
     * Set the details of the mail group which has just been created
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_group.html
     */
    toSetMailGroupDetails() {
        this.to('workmail:SetMailGroupDetails');
        return this;
    }
    /**
     * Set the details for the user account which has just been created
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html
     */
    toSetMailUserDetails() {
        this.to('workmail:SetMailUserDetails');
        return this;
    }
    /**
     * Set the details of a mobile policy associated with the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/edit_organization_mobile_policy.html
     */
    toSetMobilePolicyDetails() {
        this.to('workmail:SetMobilePolicyDetails');
        return this;
    }
    /**
     * Start a new mailbox export job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_StartMailboxExportJob.html
     */
    toStartMailboxExportJob() {
        this.to('workmail:StartMailboxExportJob');
        return this;
    }
    /**
     * Grants permission to tag the specified Amazon WorkMail organization resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('workmail:TagResource');
        return this;
    }
    /**
     * Test what inbound rules will apply to an email with a given sender and recipient
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/test-email-flow-rule.html
     */
    toTestInboundMailFlowRules() {
        this.to('workmail:TestInboundMailFlowRules');
        return this;
    }
    /**
     * Test what outbound rules will apply to an email with a given sender and recipient
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/test-email-flow-rule.html
     */
    toTestOutboundMailFlowRules() {
        this.to('workmail:TestOutboundMailFlowRules');
        return this;
    }
    /**
     * Grants permission to untag the specified Amazon WorkMail organization resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('workmail:UntagResource');
        return this;
    }
    /**
     * Update the details of an inbound email flow rule which will apply to all email sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/modify-email-flow-rule.html
     */
    toUpdateInboundMailFlowRule() {
        this.to('workmail:UpdateInboundMailFlowRule');
        return this;
    }
    /**
     * Updates the maximum size (in MB) of the user's mailbox.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateMailboxQuota.html
     */
    toUpdateMailboxQuota() {
        this.to('workmail:UpdateMailboxQuota');
        return this;
    }
    /**
     * Update the details of an outbound email flow rule which will apply to all email sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/modify-email-flow-rule.html
     */
    toUpdateOutboundMailFlowRule() {
        this.to('workmail:UpdateOutboundMailFlowRule');
        return this;
    }
    /**
     * Updates the primary email for a user, group, or resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdatePrimaryEmailAddress.html
     */
    toUpdatePrimaryEmailAddress() {
        this.to('workmail:UpdatePrimaryEmailAddress');
        return this;
    }
    /**
     * Updates data for the resource. To retrieve the latest information, it must be preceded by a DescribeResource call.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateResource.html
     */
    toUpdateResource() {
        this.to('workmail:UpdateResource');
        return this;
    }
    /**
     * Update the details of an existing SMTP device registered against an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toUpdateSmtpGateway() {
        this.to('workmail:UpdateSmtpGateway');
        return this;
    }
    /**
     * Remotely wipe the mobile device associated with a user's account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html#remote_wipe_device
     */
    toWipeMobileDevice() {
        this.to('workmail:WipeMobileDevice');
        return this;
    }
    /**
     * Adds a resource of type organization to the statement
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/organizations_overview.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOrganization(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:workmail:${Region}:${Account}:organization/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Workmail = Workmail;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid29ya21haWwuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ3b3JrbWFpbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSxzQ0FBNEM7QUFFNUM7Ozs7R0FJRztBQUNILE1BQWEsUUFBUyxTQUFRLHdCQUFlO0lBRzNDOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBUk4sa0JBQWEsR0FBRyxVQUFVLENBQUM7UUEyckN4QixvQkFBZSxHQUFvQjtZQUMzQyxPQUFPLEVBQUU7Z0JBQ1AsbUJBQW1CO2dCQUNuQiw2QkFBNkI7Z0JBQzdCLHdCQUF3QjtnQkFDeEIsd0JBQXdCO2dCQUN4QixhQUFhO2dCQUNiLGFBQWE7Z0JBQ2IsMkJBQTJCO2dCQUMzQixrQkFBa0I7Z0JBQ2xCLGdCQUFnQjtnQkFDaEIsb0JBQW9CO2dCQUNwQiw0QkFBNEI7Z0JBQzVCLGdCQUFnQjtnQkFDaEIsbUJBQW1CO2dCQUNuQixZQUFZO2dCQUNaLHlCQUF5QjtnQkFDekIsYUFBYTtnQkFDYixhQUFhO2dCQUNiLDJCQUEyQjtnQkFDM0Isa0JBQWtCO2dCQUNsQiwwQkFBMEI7Z0JBQzFCLG9CQUFvQjtnQkFDcEIsb0JBQW9CO2dCQUNwQiw0QkFBNEI7Z0JBQzVCLGdCQUFnQjtnQkFDaEIsdUJBQXVCO2dCQUN2QixtQkFBbUI7Z0JBQ25CLFlBQVk7Z0JBQ1osd0JBQXdCO2dCQUN4QixtQkFBbUI7Z0JBQ25CLGtCQUFrQjtnQkFDbEIsa0NBQWtDO2dCQUNsQyw2QkFBNkI7Z0JBQzdCLGtCQUFrQjtnQkFDbEIsa0JBQWtCO2dCQUNsQixpQkFBaUI7Z0JBQ2pCLHNCQUFzQjtnQkFDdEIsdUJBQXVCO2dCQUN2QixvQkFBb0I7Z0JBQ3BCLG9CQUFvQjtnQkFDcEIsd0JBQXdCO2dCQUN4QixlQUFlO2dCQUNmLG1CQUFtQjtnQkFDbkIsVUFBVTtnQkFDVixzQkFBc0I7Z0JBQ3RCLG9CQUFvQjtnQkFDcEIscUJBQXFCO2dCQUNyQixvQkFBb0I7Z0JBQ3BCLHdCQUF3QjtnQkFDeEIsdUJBQXVCO2dCQUN2QiwwQkFBMEI7Z0JBQzFCLDJCQUEyQjtnQkFDM0IsMkJBQTJCO2dCQUMzQixvQkFBb0I7Z0JBQ3BCLDRCQUE0QjtnQkFDNUIsMkJBQTJCO2dCQUMzQixnQkFBZ0I7Z0JBQ2hCLG1CQUFtQjtnQkFDbkIsa0JBQWtCO2FBQ25CO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLHFCQUFxQjtnQkFDckIsZUFBZTtnQkFDZixpQkFBaUI7Z0JBQ2pCLHFCQUFxQjtnQkFDckIsb0JBQW9CO2dCQUNwQixtQkFBbUI7Z0JBQ25CLHNCQUFzQjtnQkFDdEIsdUJBQXVCO2dCQUN2QixrQkFBa0I7Z0JBQ2xCLGNBQWM7Z0JBQ2Qsd0JBQXdCO2dCQUN4QixhQUFhO2dCQUNiLGtCQUFrQjtnQkFDbEIsWUFBWTtnQkFDWiwwQkFBMEI7Z0JBQzFCLHVCQUF1QjtnQkFDdkIsd0JBQXdCO2dCQUN4QixtQkFBbUI7Z0JBQ25CLDJCQUEyQjtnQkFDM0IsdUJBQXVCO2dCQUN2QixlQUFlO2dCQUNmLGtCQUFrQjtnQkFDbEIscUJBQXFCO2dCQUNyQixXQUFXO2FBQ1o7WUFDRCxNQUFNLEVBQUU7Z0JBQ04sNkJBQTZCO2dCQUM3QiwwQkFBMEI7Z0JBQzFCLDhCQUE4QjtnQkFDOUIscUJBQXFCO2dCQUNyQix3QkFBd0I7Z0JBQ3hCLDJCQUEyQjtnQkFDM0Isb0JBQW9CO2dCQUNwQixzQkFBc0I7Z0JBQ3RCLHFCQUFxQjtnQkFDckIsb0JBQW9CO2dCQUNwQixtQkFBbUI7Z0JBQ25CLHdCQUF3QjtnQkFDeEIseUJBQXlCO2dCQUN6Qix3QkFBd0I7Z0JBQ3hCLHdCQUF3QjtnQkFDeEIsZUFBZTthQUNoQjtZQUNELFNBQVMsRUFBRTtnQkFDVCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBL3hDRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQzlDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsRUFBRSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDL0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsRUFBRSxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsRUFBRSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDL0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLElBQUksQ0FBQyxFQUFFLENBQUMsMkNBQTJDLENBQUMsQ0FBQztRQUNyRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQzlDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDN0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM3QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQzlDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBaUhEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGNBQWMsQ0FBQyxVQUFrQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzdGLElBQUksR0FBRyxHQUFHLDJFQUEyRSxDQUFDO1FBQ3RGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUMvQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztDQUNGO0FBaDBDRCw0QkFnMENDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSBcIi4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWxcIjtcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gXCIuLi9zaGFyZWRcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3dvcmttYWlsXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b253b3JrbWFpbC5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIFdvcmttYWlsIGV4dGVuZHMgUG9saWN5U3RhdGVtZW50IHtcbiAgcHVibGljIHNlcnZpY2VQcmVmaXggPSAnd29ya21haWwnO1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3dvcmttYWlsXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b253b3JrbWFpbC5odG1sKS5cbiAgICpcbiAgICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gICAqL1xuICBjb25zdHJ1Y3RvciAoc2lkPzogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2lkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgbGlzdCBvZiBtZW1iZXJzICh1c2VycyBvciBncm91cHMpIHRvIGEgZ3JvdXAuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL2dyb3Vwc19vdmVydmlldy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9BZGRNZW1iZXJzVG9Hcm91cCgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpBZGRNZW1iZXJzVG9Hcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBtZW1iZXIgKHVzZXIgb3IgZ3JvdXApIHRvIHRoZSByZXNvdXJjZSdzIHNldCBvZiBkZWxlZ2F0ZXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Fzc29jaWF0ZURlbGVnYXRlVG9SZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVEZWxlZ2F0ZVRvUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6QXNzb2NpYXRlRGVsZWdhdGVUb1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIG1lbWJlciAodXNlciBvciBncm91cCkgdG8gdGhlIGdyb3VwJ3Mgc2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Bc3NvY2lhdGVNZW1iZXJUb0dyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fzc29jaWF0ZU1lbWJlclRvR3JvdXAoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6QXNzb2NpYXRlTWVtYmVyVG9Hcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENhbmNlbHMgYSBjdXJyZW50bHkgcnVubmluZyBtYWlsYm94IGV4cG9ydCBqb2IuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NhbmNlbE1haWxib3hFeHBvcnRKb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ2FuY2VsTWFpbGJveEV4cG9ydEpvYigpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpDYW5jZWxNYWlsYm94RXhwb3J0Sm9iJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBhbGlhcyB0byB0aGUgc2V0IG9mIGEgZ2l2ZW4gbWVtYmVyICh1c2VyIG9yIGdyb3VwKSBvZiBXb3JrTWFpbC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQWxpYXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQWxpYXMoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6Q3JlYXRlQWxpYXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgZ3JvdXAgdGhhdCBjYW4gYmUgdXNlZCBpbiBXb3JrTWFpbCBieSBjYWxsaW5nIHRoZSBSZWdpc3RlclRvV29ya01haWwgb3BlcmF0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVHcm91cCgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpDcmVhdGVHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhbiBpbmJvdW5kIGVtYWlsIGZsb3cgcnVsZSB3aGljaCB3aWxsIGFwcGx5IHRvIGFsbCBlbWFpbCBzZW50IHRvIGFuIG9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS9jcmVhdGUtZW1haWwtcnVsZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSW5ib3VuZE1haWxGbG93UnVsZSgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpDcmVhdGVJbmJvdW5kTWFpbEZsb3dSdWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIG1haWwgZG9tYWluLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS9hZGRfZG9tYWluLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZU1haWxEb21haW4oKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6Q3JlYXRlTWFpbERvbWFpbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSB1c2VyIGluIHRoZSBkaXJlY3RvcnkgYW5kIHRoZSBXb3JrTWFpbCBzdG9yYWdlIGJ1dCBkb2VzIG5vdCBlbmFibGUgdGhlIHVzZXIgZm9yIG1haWwuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL21hbmFnZS11c2Vycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVNYWlsVXNlcigpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpDcmVhdGVNYWlsVXNlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgQW1hem9uIFdvcmtNYWlsIG9yZ2FuaXphdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlT3JnYW5pemF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZU9yZ2FuaXphdGlvbigpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpDcmVhdGVPcmdhbml6YXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYW4gb3V0Ym91bmQgZW1haWwgZmxvdyBydWxlIHdoaWNoIHdpbGwgYXBwbHkgdG8gYWxsIGVtYWlsIHNlbnQgZnJvbSBhbiBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvY3JlYXRlLWVtYWlsLXJ1bGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZU91dGJvdW5kTWFpbEZsb3dSdWxlKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkNyZWF0ZU91dGJvdW5kTWFpbEZsb3dSdWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIG5ldyBXb3JrTWFpbCByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6Q3JlYXRlUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWdpc3RlciBhbiBTTVRQIGRldmljZSBhZ2FpbnN0IGEgV29ya01haWwgb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL3NtdHAtZ2F0ZXdheS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTbXRwR2F0ZXdheSgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpDcmVhdGVTbXRwR2F0ZXdheScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSB1c2VyIHdobyBjYW4gYmUgdXNlZCBpbiBXb3JrTWFpbCBieSBjYWxsaW5nIHRoZSBSZWdpc3RlclRvV29ya01haWwgb3BlcmF0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVVc2VyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVVzZXIoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6Q3JlYXRlVXNlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYW4gYWNjZXNzIGNvbnRyb2wgcnVsZSBmb3IgdGhlIHNwZWNpZmllZCBXb3JrTWFpbCBvcmdhbml6YXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUFjY2Vzc0NvbnRyb2xSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUFjY2Vzc0NvbnRyb2xSdWxlKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlbGV0ZUFjY2Vzc0NvbnRyb2xSdWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlIG9uZSBvciBtb3JlIHNwZWNpZmllZCBhbGlhc2VzIGZyb20gYSBzZXQgb2YgYWxpYXNlcyBmb3IgYSBnaXZlbiB1c2VyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVBbGlhcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVBbGlhcygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpEZWxldGVBbGlhcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBncm91cCBmcm9tIFdvcmtNYWlsLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVHcm91cCgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpEZWxldGVHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZSBhbiBpbmJvdW5kIGVtYWlsIGZsb3cgcnVsZSB0byBubyBsb25nZXIgYXBwbHkgdG8gZW1haWxzIHNlbnQgdG8gYW4gb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL3JlbW92ZS1lbWFpbC1mbG93LXJ1bGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlSW5ib3VuZE1haWxGbG93UnVsZSgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpEZWxldGVJbmJvdW5kTWFpbEZsb3dSdWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlcyBhbiB1bnVzZWQgbWFpbCBkb21haW4gZnJvbSBhbiBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvcmVtb3ZlX2RvbWFpbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVNYWlsRG9tYWluKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlbGV0ZU1haWxEb21haW4nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIHBlcm1pc3Npb25zIGdyYW50ZWQgdG8gYSBtZW1iZXIgKHVzZXIgb3IgZ3JvdXApLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVNYWlsYm94UGVybWlzc2lvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlTWFpbGJveFBlcm1pc3Npb25zKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlbGV0ZU1haWxib3hQZXJtaXNzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZXMgYSBtb2JpbGUgZGV2aWNlIGZyb20gYSB1c2VyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL21hbmFnZS1kZXZpY2VzLmh0bWwjcmVtb3ZlX21vYmlsZV9kZXZpY2VcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU1vYmlsZURldmljZSgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpEZWxldGVNb2JpbGVEZXZpY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGFuIEFtYXpvbiBXb3JrTWFpbCBvcmdhbml6YXRpb24gYW5kIGFsbCB1bmRlcmx5aW5nIEFXUyByZXNvdXJjZXMgbWFuYWdlZCBieSBBbWF6b24gV29ya01haWwgYXMgcGFydCBvZiB0aGUgb3JnYW5pemF0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVPcmdhbml6YXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlT3JnYW5pemF0aW9uKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlbGV0ZU9yZ2FuaXphdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZSBhbiBvdXRib3VuZCBlbWFpbCBmbG93IHJ1bGUgdG8gbm8gbG9uZ2VyIGFwcGx5IHRvIGVtYWlscyBzZW50IGZyb20gYW4gb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL3JlbW92ZS1lbWFpbC1mbG93LXJ1bGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlT3V0Ym91bmRNYWlsRmxvd1J1bGUoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6RGVsZXRlT3V0Ym91bmRNYWlsRmxvd1J1bGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVJlc291cmNlKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlbGV0ZVJlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyB0aGUgcmV0ZW50aW9uIHBvbGljeSBiYXNlZCBvbiB0aGUgc3VwcGxpZWQgb3JnYW5pemF0aW9uIGFuZCBwb2xpY3kgaWRlbnRpZmllcnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVJldGVudGlvblBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVSZXRlbnRpb25Qb2xpY3koKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6RGVsZXRlUmV0ZW50aW9uUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlIGFuIFNNVFAgZGV2aWNlIGZyb20gYW4gb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL3NtdHAtZ2F0ZXdheS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTbXRwR2F0ZXdheSgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpEZWxldGVTbXRwR2F0ZXdheScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSB1c2VyIGZyb20gV29ya01haWwgYW5kIGFsbCBzdWJzZXF1ZW50IHN5c3RlbXMuIFRoZSBhY3Rpb24gY2Fubm90IGJlIHVuZG9uZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlVXNlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVVc2VyKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlbGV0ZVVzZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXJrIGEgdXNlciwgZ3JvdXAsIG9yIHJlc291cmNlIGFzIG5vIGxvbmdlciB1c2VkIGluIFdvcmtNYWlsLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXJlZ2lzdGVyRnJvbVdvcmtNYWlsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlcmVnaXN0ZXJGcm9tV29ya01haWwoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6RGVyZWdpc3RlckZyb21Xb3JrTWFpbCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFNob3dzIGEgbGlzdCBvZiBkaXJlY3RvcmllcyBhdmFpbGFibGUgZm9yIHVzZSBpbiBjcmVhdGluZyBhbiBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS9hZGRfbmV3X29yZ2FuaXphdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURpcmVjdG9yaWVzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlc2NyaWJlRGlyZWN0b3JpZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBkYXRhIGF2YWlsYWJsZSBmb3IgdGhlIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVHcm91cCgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpEZXNjcmliZUdyb3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgZGV0YWlscyBvZiBhbiBpbmJvdW5kIG1haWwgZmxvdyBydWxlIGNvbmZpZ3VyZWQgZm9yIGFuIG9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL2VtYWlsLWZsb3dzLmh0bWwjZW1haWwtZmxvd3MtcnVsZS1hY3Rpb25zXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUluYm91bmRNYWlsRmxvd1J1bGUoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6RGVzY3JpYmVJbmJvdW5kTWFpbEZsb3dSdWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU2hvd3MgYSBsaXN0IG9mIEtNUyBLZXlzIGF2YWlsYWJsZSBmb3IgdXNlIGluIGNyZWF0aW5nIGFuIG9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL2FkZF9uZXdfb3JnYW5pemF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlS21zS2V5cygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpEZXNjcmliZUttc0tleXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTaG93cyB0aGUgZGV0YWlscyBvZiBhbGwgbWFpbCBkb21haW5zIGFzc29jaWF0ZWQgd2l0aCB0aGUgb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvZG9tYWluc19vdmVydmlldy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZU1haWxEb21haW5zKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlc2NyaWJlTWFpbERvbWFpbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTaG93cyB0aGUgZGV0YWlscyBvZiBhbGwgZ3JvdXBzIGFzc29jaWF0ZWQgd2l0aCB0aGUgb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvZ3JvdXBzX292ZXJ2aWV3Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlTWFpbEdyb3VwcygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpEZXNjcmliZU1haWxHcm91cHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTaG93cyB0aGUgZGV0YWlscyBvZiBhbGwgdXNlcnMgYXNzb2NpYXRlZCB3aXRoIHRoZSBvcmdhaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL3VzZXJzX292ZXJ2aWV3Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlTWFpbFVzZXJzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlc2NyaWJlTWFpbFVzZXJzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmUgZGV0YWlscyBvZiBhIG1haWxib3ggZXhwb3J0IGpvYi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZU1haWxib3hFeHBvcnRKb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVNYWlsYm94RXhwb3J0Sm9iKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlc2NyaWJlTWFpbGJveEV4cG9ydEpvYicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFByb3ZpZGVzIG1vcmUgaW5mb3JtYXRpb24gcmVnYXJkaW5nIGEgZ2l2ZW4gb3JnYW5pemF0aW9uIGJhc2VkIG9uIGl0cyBpZGVudGlmaWVyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlT3JnYW5pemF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlT3JnYW5pemF0aW9uKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlc2NyaWJlT3JnYW5pemF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU2hvd3MgYSBzdW1tYXJ5IG9mIGFsbCBvcmdhbml6YXRpb25zIGFzc29jaWF0ZWQgd2l0aCB0aGUgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL29yZ2FuaXphdGlvbnNfb3ZlcnZpZXcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVPcmdhbml6YXRpb25zKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlc2NyaWJlT3JnYW5pemF0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIGRldGFpbHMgb2YgYW4gb3V0Ym91bmQgbWFpbCBmbG93IHJ1bGUgY29uZmlndXJlZCBmb3IgYW4gb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvZW1haWwtZmxvd3MuaHRtbCNlbWFpbC1mbG93cy1ydWxlLW91dGJvdW5kXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZU91dGJvdW5kTWFpbEZsb3dSdWxlKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlc2NyaWJlT3V0Ym91bmRNYWlsRmxvd1J1bGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBkYXRhIGF2YWlsYWJsZSBmb3IgdGhlIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVSZXNvdXJjZSgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpEZXNjcmliZVJlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgZGV0YWlscyBvZiBhbiBTTVRQIGRldmljZSByZWdpc3RlcmVkIGFnYWluc3QgYW4gb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvc210cC1nYXRld2F5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlU210cEdhdGV3YXkoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6RGVzY3JpYmVTbXRwR2F0ZXdheScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFByb3ZpZGVzIGluZm9ybWF0aW9uIHJlZ2FyZGluZyB0aGUgdXNlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVVzZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVVc2VyKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRlc2NyaWJlVXNlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERpc2FibGUgYSBtYWlsIGdyb3VwIHdoZW4gaXQgaXMgbm90IGJlaW5nIHVzZWQgYW5kLCB0byBhbGxvdyBpdCB0byBiZSBkZWxldGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL3JlbW92ZV9ncm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EaXNhYmxlTWFpbEdyb3VwcygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpEaXNhYmxlTWFpbEdyb3VwcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERpc2FibGUgYSB1c2VyIG1haWxib3ggd2hlbiBpdCBpcyBubyBsb25nZXIgYmVpbmcgdXNlZCwgYW5kIHRvIGFsbG93IGl0IHRvIGJlIGRlbGV0ZWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvbWFuYWdlLW1haWxib3hlcy5odG1sI2RlbGV0ZV91c2VyX21haWxib3hcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2FibGVNYWlsVXNlcnMoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6RGlzYWJsZU1haWxVc2VycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZXMgYSBtZW1iZXIgZnJvbSB0aGUgcmVzb3VyY2UncyBzZXQgb2YgZGVsZWdhdGVzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVEZWxlZ2F0ZUZyb21SZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EaXNhc3NvY2lhdGVEZWxlZ2F0ZUZyb21SZXNvdXJjZSgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpEaXNhc3NvY2lhdGVEZWxlZ2F0ZUZyb21SZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZXMgYSBtZW1iZXIgZnJvbSBhIGdyb3VwLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVNZW1iZXJGcm9tR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlTWVtYmVyRnJvbUdyb3VwKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkRpc2Fzc29jaWF0ZU1lbWJlckZyb21Hcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuYWJsZSBhIG1haWwgZG9tYWluIGluIHRoZSBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvYWRkX2RvbWFpbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9FbmFibGVNYWlsRG9tYWluKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkVuYWJsZU1haWxEb21haW4nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFbmFibGUgYSBtYWlsIGdyb3VwIGFmdGVyIGl0IGhhcyBiZWVuIGNyZWF0ZWQgdG8gYWxsb3cgaXQgdG8gcmVjZWl2ZSBtYWlsXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL2VuYWJsZV9leGlzdGluZ19ncm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9FbmFibGVNYWlsR3JvdXBzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkVuYWJsZU1haWxHcm91cHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFbmFibGUgYSB1c2VyJ3MgbWFpbGJveCBhZnRlciBpdCBoYXMgYmVlbiBjcmVhdGVkIHRvIGFsbG93IGl0IHRvIHJlY2VpdmUgbWFpbFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS9tYW5hZ2UtdXNlcnMuaHRtbCNlbmFibGVfZXhpc3RpbmdfdXNlclxuICAgKi9cbiAgcHVibGljIHRvRW5hYmxlTWFpbFVzZXJzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkVuYWJsZU1haWxVc2VycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldHMgdGhlIGVmZmVjdHMgb2YgYW4gb3JnYW5pemF0aW9uJ3MgYWNjZXNzIGNvbnRyb2wgcnVsZXMgYXMgdGhleSBhcHBseSB0byBhIHNwZWNpZmllZCBJUHY0IGFkZHJlc3MsIGFjY2VzcyBwcm90b2NvbCBhY3Rpb24sIG9yIHVzZXIgSUQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0QWNjZXNzQ29udHJvbEVmZmVjdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRBY2Nlc3NDb250cm9sRWZmZWN0KCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkdldEFjY2Vzc0NvbnRyb2xFZmZlY3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgdGhlIHJldGVudGlvbiBwb2xpY3kgYXNzb2NpYXRlZCBhdCBhbiBvcmdhbml6YXRpb25hbCBsZXZlbC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXREZWZhdWx0UmV0ZW50aW9uUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldERlZmF1bHRSZXRlbnRpb25Qb2xpY3koKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6R2V0RGVmYXVsdFJldGVudGlvblBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgam91cm5hbGluZyBhbmQgZmFsbGJhY2sgZW1haWwgYWRkcmVzc2VzIGNvbmZpZ3VyZWQgZm9yIGVtYWlsIGpvdXJuYWxpbmdcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS9qb3VybmFsaW5nX292ZXJ2aWV3Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEpvdXJuYWxpbmdSdWxlcygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpHZXRKb3VybmFsaW5nUnVsZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXQgdGhlIGRldGFpbHMgb2YgdGhlIG1haWwgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvZG9tYWluc19vdmVydmlldy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRNYWlsRG9tYWluRGV0YWlscygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpHZXRNYWlsRG9tYWluRGV0YWlscycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldCB0aGUgZGV0YWlscyBvZiB0aGUgbWFpbCBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL2dyb3Vwc19vdmVydmlldy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRNYWlsR3JvdXBEZXRhaWxzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkdldE1haWxHcm91cERldGFpbHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXQgdGhlIGRldGFpbHMgb2YgdGhlIHVzZXIncyBtYWlsYm94IGFuZCBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvdXNlcnNfb3ZlcnZpZXcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0TWFpbFVzZXJEZXRhaWxzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkdldE1haWxVc2VyRGV0YWlscycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIGRldGFpbHMgb2YgdGhlIHVzZXIncyBtYWlsYm94LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldE1haWxib3hEZXRhaWxzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE1haWxib3hEZXRhaWxzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkdldE1haWxib3hEZXRhaWxzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR2V0IHRoZSBkZXRhaWxzIG9mIHRoZSBtb2JpbGUgZGV2aWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvbWFuYWdlLWRldmljZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0TW9iaWxlRGV2aWNlRGV0YWlscygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpHZXRNb2JpbGVEZXZpY2VEZXRhaWxzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR2V0IGEgbGlzdCBvZiB0aGUgbW9iaWxlIGRldmljZXMgYXNzb2NpYXRlZCB3aXRoIHRoZSB1c2VyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvbWFuYWdlLWRldmljZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0TW9iaWxlRGV2aWNlc0ZvclVzZXIoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6R2V0TW9iaWxlRGV2aWNlc0ZvclVzZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXQgdGhlIGRldGFpbHMgb2YgdGhlIG1vYmlsZSBkZXZpY2UgcG9saWN5IGFzc29jaWF0ZWQgd2l0aCB0aGUgb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvZWRpdF9vcmdhbml6YXRpb25fbW9iaWxlX3BvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRNb2JpbGVQb2xpY3lEZXRhaWxzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkdldE1vYmlsZVBvbGljeURldGFpbHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0cyB0aGUgYWNjZXNzIGNvbnRyb2wgcnVsZXMgZm9yIHRoZSBzcGVjaWZpZWQgb3JnYW5pemF0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RBY2Nlc3NDb250cm9sUnVsZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEFjY2Vzc0NvbnRyb2xSdWxlcygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpMaXN0QWNjZXNzQ29udHJvbFJ1bGVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIHBhZ2luYXRlZCBjYWxsIHRvIGxpc3QgdGhlIGFsaWFzZXMgYXNzb2NpYXRlZCB3aXRoIGEgZ2l2ZW4gZW50aXR5LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RBbGlhc2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RBbGlhc2VzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkxpc3RBbGlhc2VzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbiBvdmVydmlldyBvZiB0aGUgbWVtYmVycyBvZiBhIGdyb3VwLiBVc2VycyBhbmQgZ3JvdXBzIGNhbiBiZSBtZW1iZXJzIG9mIGEgZ3JvdXAuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEdyb3VwTWVtYmVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0R3JvdXBNZW1iZXJzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkxpc3RHcm91cE1lbWJlcnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHN1bW1hcmllcyBvZiB0aGUgb3JnYW5pemF0aW9uJ3MgZ3JvdXBzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEdyb3VwcygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpMaXN0R3JvdXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIGxpc3Qgb2YgaW5ib3VuZCBtYWlsIGZsb3cgcnVsZXMgY29uZmlndXJlZCBmb3IgYW4gb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvZW1haWwtZmxvd3MuaHRtbCNlbWFpbC1mbG93cy1ydWxlLWFjdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJbmJvdW5kTWFpbEZsb3dSdWxlcygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpMaXN0SW5ib3VuZE1haWxGbG93UnVsZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0IG1haWxib3ggZXhwb3J0IGpvYnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdE1haWxib3hFeHBvcnRKb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RNYWlsYm94RXhwb3J0Sm9icygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpMaXN0TWFpbGJveEV4cG9ydEpvYnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0cyB0aGUgbWFpbGJveCBwZXJtaXNzaW9ucyBhc3NvY2lhdGVkIHdpdGggYSB1c2VyLCBncm91cCwgb3IgcmVzb3VyY2UgbWFpbGJveC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0TWFpbGJveFBlcm1pc3Npb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RNYWlsYm94UGVybWlzc2lvbnMoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6TGlzdE1haWxib3hQZXJtaXNzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldCBhIGxpc3Qgb2YgYWxsIHRoZSBtZW1iZXJzIGluIGEgbWFpbCBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL2dyb3Vwc19vdmVydmlldy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TWVtYmVyc0luTWFpbEdyb3VwKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkxpc3RNZW1iZXJzSW5NYWlsR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHN1bW1hcmllcyBvZiB0aGUgY3VzdG9tZXIncyBub24tZGVsZXRlZCBvcmdhbml6YXRpb25zLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RPcmdhbml6YXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RPcmdhbml6YXRpb25zKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkxpc3RPcmdhbml6YXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIGxpc3Qgb2Ygb3V0Ym91bmQgbWFpbCBmbG93IHJ1bGVzIGNvbmZpZ3VyZWQgZm9yIGFuIG9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL2VtYWlsLWZsb3dzLmh0bWwjZW1haWwtZmxvd3MtcnVsZS1vdXRib3VuZFxuICAgKi9cbiAgcHVibGljIHRvTGlzdE91dGJvdW5kTWFpbEZsb3dSdWxlcygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpMaXN0T3V0Ym91bmRNYWlsRmxvd1J1bGVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgdGhlIGRlbGVnYXRlcyBhc3NvY2lhdGVkIHdpdGggYSByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0UmVzb3VyY2VEZWxlZ2F0ZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFJlc291cmNlRGVsZWdhdGVzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkxpc3RSZXNvdXJjZURlbGVnYXRlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgc3VtbWFyaWVzIG9mIHRoZSBvcmdhbml6YXRpb24ncyByZXNvdXJjZXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFJlc291cmNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UmVzb3VyY2VzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkxpc3RSZXNvdXJjZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgbGlzdCBvZiBTTVRQIGRldmljZXMgcmVnaXN0ZXJlZCBhZ2FpbnN0IHRoZSBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS9zbXRwLWdhdGV3YXkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFNtdHBHYXRld2F5cygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpMaXN0U210cEdhdGV3YXlzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgdGFncyBhcHBsaWVkIHRvIGFuIEFtYXpvbiBXb3JrTWFpbCBvcmdhbml6YXRpb24gcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOkxpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHN1bW1hcmllcyBvZiB0aGUgb3JnYW5pemF0aW9uJ3MgdXNlcnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFVzZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RVc2VycygpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpMaXN0VXNlcnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgbmV3IGFjY2VzcyBjb250cm9sIHJ1bGUgZm9yIHRoZSBzcGVjaWZpZWQgb3JnYW5pemF0aW9uLiBUaGUgcnVsZSBhbGxvd3Mgb3IgZGVuaWVzIGFjY2VzcyB0byB0aGUgb3JnYW5pemF0aW9uIGZvciB0aGUgc3BlY2lmaWVkIElQdjQgYWRkcmVzc2VzLCBhY2Nlc3MgcHJvdG9jb2wgYWN0aW9ucywgYW5kIHVzZXIgSURzLiBBZGRpbmcgYSBuZXcgcnVsZSB3aXRoIHRoZSBzYW1lIG5hbWUgYXMgYW4gZXhpc3RpbmcgcnVsZSByZXBsYWNlcyB0aGUgb2xkZXIgcnVsZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0QWNjZXNzQ29udHJvbFJ1bGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0QWNjZXNzQ29udHJvbFJ1bGUoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6UHV0QWNjZXNzQ29udHJvbFJ1bGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTZXRzIHBlcm1pc3Npb25zIGZvciBhIHVzZXIsIGdyb3VwLCBvciByZXNvdXJjZS4gVGhpcyByZXBsYWNlcyBhbnkgcHJlLWV4aXN0aW5nIHBlcm1pc3Npb25zLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRNYWlsYm94UGVybWlzc2lvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0TWFpbGJveFBlcm1pc3Npb25zKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOlB1dE1haWxib3hQZXJtaXNzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgb3IgdXBkYXRlcyB0aGUgcmV0ZW50aW9uIHBvbGljeSBmb3IgdGhlIHNwZWNpZmllZCBvcmdhbml6YXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dFJldGVudGlvblBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRSZXRlbnRpb25Qb2xpY3koKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6UHV0UmV0ZW50aW9uUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVnaXN0ZXJzIGFuIGV4aXN0aW5nIGFuZCBkaXNhYmxlZCB1c2VyLCBncm91cCwgb3IgcmVzb3VyY2UgZm9yIHVzZSBieSBhc3NvY2lhdGluZyBhIG1haWxib3ggYW5kIGNhbGVuZGFyaW5nIGNhcGFiaWxpdGllcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVnaXN0ZXJUb1dvcmtNYWlsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlZ2lzdGVyVG9Xb3JrTWFpbCgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpSZWdpc3RlclRvV29ya01haWwnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdmUgbWVtYmVycyBmcm9tIGEgbWFpbCBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS9ncm91cHNfb3ZlcnZpZXcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVtb3ZlTWVtYmVyc0Zyb21Hcm91cCgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpSZW1vdmVNZW1iZXJzRnJvbUdyb3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWxsb3dzIHRoZSBhZG1pbmlzdHJhdG9yIHRvIHJlc2V0IHRoZSBwYXNzd29yZCBmb3IgYSB1c2VyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZXNldFBhc3N3b3JkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Jlc2V0UGFzc3dvcmQoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6UmVzZXRQYXNzd29yZCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlc2V0IHRoZSBwYXNzd29yZCBmb3IgYSB1c2VyJ3MgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS9tYW5hZ2UtdXNlcnMuaHRtbCNyZXNldF91c2VyX3Bhc3N3b3JkXG4gICAqL1xuICBwdWJsaWMgdG9SZXNldFVzZXJQYXNzd29yZCgpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpSZXNldFVzZXJQYXNzd29yZCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFByZWZpeCBzZWFyY2ggdG8gZmluZCBhIHNwZWNpZmljIHVzZXIgaW4gYSBtYWlsIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvZ3JvdXBzX292ZXJ2aWV3Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1NlYXJjaE1lbWJlcnMoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6U2VhcmNoTWVtYmVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hcmsgYSB1c2VyIGFzIGJlaW5nIGFuIGFkbWluaXN0cmF0b3JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvdXNlcnNfb3ZlcnZpZXcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2V0QWRtaW4oKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6U2V0QWRtaW4nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTZXQgdGhlIGRlZmF1bHQgbWFpbCBkb21haW4gZm9yIHRoZSBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvZGVmYXVsdF9kb21haW4uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2V0RGVmYXVsdE1haWxEb21haW4oKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6U2V0RGVmYXVsdE1haWxEb21haW4nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTZXQgam91cm5hbGluZyBhbmQgZmFsbGJhY2sgZW1haWwgYWRkcmVzc2VzIGZvciBlbWFpbCBqb3VybmFsaW5nXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL2pvdXJuYWxpbmdfb3ZlcnZpZXcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2V0Sm91cm5hbGluZ1J1bGVzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOlNldEpvdXJuYWxpbmdSdWxlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFNldCB0aGUgZGV0YWlscyBvZiB0aGUgbWFpbCBncm91cCB3aGljaCBoYXMganVzdCBiZWVuIGNyZWF0ZWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvYWRkX25ld19ncm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXRNYWlsR3JvdXBEZXRhaWxzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOlNldE1haWxHcm91cERldGFpbHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTZXQgdGhlIGRldGFpbHMgZm9yIHRoZSB1c2VyIGFjY291bnQgd2hpY2ggaGFzIGp1c3QgYmVlbiBjcmVhdGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL21hbmFnZS11c2Vycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXRNYWlsVXNlckRldGFpbHMoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6U2V0TWFpbFVzZXJEZXRhaWxzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU2V0IHRoZSBkZXRhaWxzIG9mIGEgbW9iaWxlIHBvbGljeSBhc3NvY2lhdGVkIHdpdGggdGhlIG9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS9lZGl0X29yZ2FuaXphdGlvbl9tb2JpbGVfcG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1NldE1vYmlsZVBvbGljeURldGFpbHMoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6U2V0TW9iaWxlUG9saWN5RGV0YWlscycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXJ0IGEgbmV3IG1haWxib3ggZXhwb3J0IGpvYi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnRNYWlsYm94RXhwb3J0Sm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0TWFpbGJveEV4cG9ydEpvYigpIHtcbiAgICB0aGlzLnRvKCd3b3JrbWFpbDpTdGFydE1haWxib3hFeHBvcnRKb2InKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB0YWcgdGhlIHNwZWNpZmllZCBBbWF6b24gV29ya01haWwgb3JnYW5pemF0aW9uIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1RhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOlRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVGVzdCB3aGF0IGluYm91bmQgcnVsZXMgd2lsbCBhcHBseSB0byBhbiBlbWFpbCB3aXRoIGEgZ2l2ZW4gc2VuZGVyIGFuZCByZWNpcGllbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvdGVzdC1lbWFpbC1mbG93LXJ1bGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGVzdEluYm91bmRNYWlsRmxvd1J1bGVzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOlRlc3RJbmJvdW5kTWFpbEZsb3dSdWxlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFRlc3Qgd2hhdCBvdXRib3VuZCBydWxlcyB3aWxsIGFwcGx5IHRvIGFuIGVtYWlsIHdpdGggYSBnaXZlbiBzZW5kZXIgYW5kIHJlY2lwaWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS90ZXN0LWVtYWlsLWZsb3ctcnVsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UZXN0T3V0Ym91bmRNYWlsRmxvd1J1bGVzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOlRlc3RPdXRib3VuZE1haWxGbG93UnVsZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1bnRhZyB0aGUgc3BlY2lmaWVkIEFtYXpvbiBXb3JrTWFpbCBvcmdhbml6YXRpb24gcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOlVudGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGUgdGhlIGRldGFpbHMgb2YgYW4gaW5ib3VuZCBlbWFpbCBmbG93IHJ1bGUgd2hpY2ggd2lsbCBhcHBseSB0byBhbGwgZW1haWwgc2VudCB0byBhbiBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L2FkbWluZ3VpZGUvbW9kaWZ5LWVtYWlsLWZsb3ctcnVsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVJbmJvdW5kTWFpbEZsb3dSdWxlKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOlVwZGF0ZUluYm91bmRNYWlsRmxvd1J1bGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIHRoZSBtYXhpbXVtIHNpemUgKGluIE1CKSBvZiB0aGUgdXNlcidzIG1haWxib3guXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZU1haWxib3hRdW90YS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVNYWlsYm94UXVvdGEoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6VXBkYXRlTWFpbGJveFF1b3RhJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlIHRoZSBkZXRhaWxzIG9mIGFuIG91dGJvdW5kIGVtYWlsIGZsb3cgcnVsZSB3aGljaCB3aWxsIGFwcGx5IHRvIGFsbCBlbWFpbCBzZW50IGZyb20gYW4gb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL21vZGlmeS1lbWFpbC1mbG93LXJ1bGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlT3V0Ym91bmRNYWlsRmxvd1J1bGUoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6VXBkYXRlT3V0Ym91bmRNYWlsRmxvd1J1bGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIHRoZSBwcmltYXJ5IGVtYWlsIGZvciBhIHVzZXIsIGdyb3VwLCBvciByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlUHJpbWFyeUVtYWlsQWRkcmVzcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVQcmltYXJ5RW1haWxBZGRyZXNzKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOlVwZGF0ZVByaW1hcnlFbWFpbEFkZHJlc3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIGRhdGEgZm9yIHRoZSByZXNvdXJjZS4gVG8gcmV0cmlldmUgdGhlIGxhdGVzdCBpbmZvcm1hdGlvbiwgaXQgbXVzdCBiZSBwcmVjZWRlZCBieSBhIERlc2NyaWJlUmVzb3VyY2UgY2FsbC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya21haWwvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6VXBkYXRlUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGUgdGhlIGRldGFpbHMgb2YgYW4gZXhpc3RpbmcgU01UUCBkZXZpY2UgcmVnaXN0ZXJlZCBhZ2FpbnN0IGFuIG9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS9zbXRwLWdhdGV3YXkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlU210cEdhdGV3YXkoKSB7XG4gICAgdGhpcy50bygnd29ya21haWw6VXBkYXRlU210cEdhdGV3YXknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdGVseSB3aXBlIHRoZSBtb2JpbGUgZGV2aWNlIGFzc29jaWF0ZWQgd2l0aCBhIHVzZXIncyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmttYWlsL2xhdGVzdC9hZG1pbmd1aWRlL21hbmFnZS1kZXZpY2VzLmh0bWwjcmVtb3RlX3dpcGVfZGV2aWNlXG4gICAqL1xuICBwdWJsaWMgdG9XaXBlTW9iaWxlRGV2aWNlKCkge1xuICAgIHRoaXMudG8oJ3dvcmttYWlsOldpcGVNb2JpbGVEZXZpY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBcIldyaXRlXCI6IFtcbiAgICAgIFwiQWRkTWVtYmVyc1RvR3JvdXBcIixcbiAgICAgIFwiQXNzb2NpYXRlRGVsZWdhdGVUb1Jlc291cmNlXCIsXG4gICAgICBcIkFzc29jaWF0ZU1lbWJlclRvR3JvdXBcIixcbiAgICAgIFwiQ2FuY2VsTWFpbGJveEV4cG9ydEpvYlwiLFxuICAgICAgXCJDcmVhdGVBbGlhc1wiLFxuICAgICAgXCJDcmVhdGVHcm91cFwiLFxuICAgICAgXCJDcmVhdGVJbmJvdW5kTWFpbEZsb3dSdWxlXCIsXG4gICAgICBcIkNyZWF0ZU1haWxEb21haW5cIixcbiAgICAgIFwiQ3JlYXRlTWFpbFVzZXJcIixcbiAgICAgIFwiQ3JlYXRlT3JnYW5pemF0aW9uXCIsXG4gICAgICBcIkNyZWF0ZU91dGJvdW5kTWFpbEZsb3dSdWxlXCIsXG4gICAgICBcIkNyZWF0ZVJlc291cmNlXCIsXG4gICAgICBcIkNyZWF0ZVNtdHBHYXRld2F5XCIsXG4gICAgICBcIkNyZWF0ZVVzZXJcIixcbiAgICAgIFwiRGVsZXRlQWNjZXNzQ29udHJvbFJ1bGVcIixcbiAgICAgIFwiRGVsZXRlQWxpYXNcIixcbiAgICAgIFwiRGVsZXRlR3JvdXBcIixcbiAgICAgIFwiRGVsZXRlSW5ib3VuZE1haWxGbG93UnVsZVwiLFxuICAgICAgXCJEZWxldGVNYWlsRG9tYWluXCIsXG4gICAgICBcIkRlbGV0ZU1haWxib3hQZXJtaXNzaW9uc1wiLFxuICAgICAgXCJEZWxldGVNb2JpbGVEZXZpY2VcIixcbiAgICAgIFwiRGVsZXRlT3JnYW5pemF0aW9uXCIsXG4gICAgICBcIkRlbGV0ZU91dGJvdW5kTWFpbEZsb3dSdWxlXCIsXG4gICAgICBcIkRlbGV0ZVJlc291cmNlXCIsXG4gICAgICBcIkRlbGV0ZVJldGVudGlvblBvbGljeVwiLFxuICAgICAgXCJEZWxldGVTbXRwR2F0ZXdheVwiLFxuICAgICAgXCJEZWxldGVVc2VyXCIsXG4gICAgICBcIkRlcmVnaXN0ZXJGcm9tV29ya01haWxcIixcbiAgICAgIFwiRGlzYWJsZU1haWxHcm91cHNcIixcbiAgICAgIFwiRGlzYWJsZU1haWxVc2Vyc1wiLFxuICAgICAgXCJEaXNhc3NvY2lhdGVEZWxlZ2F0ZUZyb21SZXNvdXJjZVwiLFxuICAgICAgXCJEaXNhc3NvY2lhdGVNZW1iZXJGcm9tR3JvdXBcIixcbiAgICAgIFwiRW5hYmxlTWFpbERvbWFpblwiLFxuICAgICAgXCJFbmFibGVNYWlsR3JvdXBzXCIsXG4gICAgICBcIkVuYWJsZU1haWxVc2Vyc1wiLFxuICAgICAgXCJQdXRBY2Nlc3NDb250cm9sUnVsZVwiLFxuICAgICAgXCJQdXRNYWlsYm94UGVybWlzc2lvbnNcIixcbiAgICAgIFwiUHV0UmV0ZW50aW9uUG9saWN5XCIsXG4gICAgICBcIlJlZ2lzdGVyVG9Xb3JrTWFpbFwiLFxuICAgICAgXCJSZW1vdmVNZW1iZXJzRnJvbUdyb3VwXCIsXG4gICAgICBcIlJlc2V0UGFzc3dvcmRcIixcbiAgICAgIFwiUmVzZXRVc2VyUGFzc3dvcmRcIixcbiAgICAgIFwiU2V0QWRtaW5cIixcbiAgICAgIFwiU2V0RGVmYXVsdE1haWxEb21haW5cIixcbiAgICAgIFwiU2V0Sm91cm5hbGluZ1J1bGVzXCIsXG4gICAgICBcIlNldE1haWxHcm91cERldGFpbHNcIixcbiAgICAgIFwiU2V0TWFpbFVzZXJEZXRhaWxzXCIsXG4gICAgICBcIlNldE1vYmlsZVBvbGljeURldGFpbHNcIixcbiAgICAgIFwiU3RhcnRNYWlsYm94RXhwb3J0Sm9iXCIsXG4gICAgICBcIlRlc3RJbmJvdW5kTWFpbEZsb3dSdWxlc1wiLFxuICAgICAgXCJUZXN0T3V0Ym91bmRNYWlsRmxvd1J1bGVzXCIsXG4gICAgICBcIlVwZGF0ZUluYm91bmRNYWlsRmxvd1J1bGVcIixcbiAgICAgIFwiVXBkYXRlTWFpbGJveFF1b3RhXCIsXG4gICAgICBcIlVwZGF0ZU91dGJvdW5kTWFpbEZsb3dSdWxlXCIsXG4gICAgICBcIlVwZGF0ZVByaW1hcnlFbWFpbEFkZHJlc3NcIixcbiAgICAgIFwiVXBkYXRlUmVzb3VyY2VcIixcbiAgICAgIFwiVXBkYXRlU210cEdhdGV3YXlcIixcbiAgICAgIFwiV2lwZU1vYmlsZURldmljZVwiXG4gICAgXSxcbiAgICBcIkxpc3RcIjogW1xuICAgICAgXCJEZXNjcmliZURpcmVjdG9yaWVzXCIsXG4gICAgICBcIkRlc2NyaWJlR3JvdXBcIixcbiAgICAgIFwiRGVzY3JpYmVLbXNLZXlzXCIsXG4gICAgICBcIkRlc2NyaWJlTWFpbERvbWFpbnNcIixcbiAgICAgIFwiRGVzY3JpYmVNYWlsR3JvdXBzXCIsXG4gICAgICBcIkRlc2NyaWJlTWFpbFVzZXJzXCIsXG4gICAgICBcIkRlc2NyaWJlT3JnYW5pemF0aW9uXCIsXG4gICAgICBcIkRlc2NyaWJlT3JnYW5pemF0aW9uc1wiLFxuICAgICAgXCJEZXNjcmliZVJlc291cmNlXCIsXG4gICAgICBcIkRlc2NyaWJlVXNlclwiLFxuICAgICAgXCJMaXN0QWNjZXNzQ29udHJvbFJ1bGVzXCIsXG4gICAgICBcIkxpc3RBbGlhc2VzXCIsXG4gICAgICBcIkxpc3RHcm91cE1lbWJlcnNcIixcbiAgICAgIFwiTGlzdEdyb3Vwc1wiLFxuICAgICAgXCJMaXN0SW5ib3VuZE1haWxGbG93UnVsZXNcIixcbiAgICAgIFwiTGlzdE1haWxib3hFeHBvcnRKb2JzXCIsXG4gICAgICBcIkxpc3RNYWlsYm94UGVybWlzc2lvbnNcIixcbiAgICAgIFwiTGlzdE9yZ2FuaXphdGlvbnNcIixcbiAgICAgIFwiTGlzdE91dGJvdW5kTWFpbEZsb3dSdWxlc1wiLFxuICAgICAgXCJMaXN0UmVzb3VyY2VEZWxlZ2F0ZXNcIixcbiAgICAgIFwiTGlzdFJlc291cmNlc1wiLFxuICAgICAgXCJMaXN0U210cEdhdGV3YXlzXCIsXG4gICAgICBcIkxpc3RUYWdzRm9yUmVzb3VyY2VcIixcbiAgICAgIFwiTGlzdFVzZXJzXCJcbiAgICBdLFxuICAgIFwiUmVhZFwiOiBbXG4gICAgICBcIkRlc2NyaWJlSW5ib3VuZE1haWxGbG93UnVsZVwiLFxuICAgICAgXCJEZXNjcmliZU1haWxib3hFeHBvcnRKb2JcIixcbiAgICAgIFwiRGVzY3JpYmVPdXRib3VuZE1haWxGbG93UnVsZVwiLFxuICAgICAgXCJEZXNjcmliZVNtdHBHYXRld2F5XCIsXG4gICAgICBcIkdldEFjY2Vzc0NvbnRyb2xFZmZlY3RcIixcbiAgICAgIFwiR2V0RGVmYXVsdFJldGVudGlvblBvbGljeVwiLFxuICAgICAgXCJHZXRKb3VybmFsaW5nUnVsZXNcIixcbiAgICAgIFwiR2V0TWFpbERvbWFpbkRldGFpbHNcIixcbiAgICAgIFwiR2V0TWFpbEdyb3VwRGV0YWlsc1wiLFxuICAgICAgXCJHZXRNYWlsVXNlckRldGFpbHNcIixcbiAgICAgIFwiR2V0TWFpbGJveERldGFpbHNcIixcbiAgICAgIFwiR2V0TW9iaWxlRGV2aWNlRGV0YWlsc1wiLFxuICAgICAgXCJHZXRNb2JpbGVEZXZpY2VzRm9yVXNlclwiLFxuICAgICAgXCJHZXRNb2JpbGVQb2xpY3lEZXRhaWxzXCIsXG4gICAgICBcIkxpc3RNZW1iZXJzSW5NYWlsR3JvdXBcIixcbiAgICAgIFwiU2VhcmNoTWVtYmVyc1wiXG4gICAgXSxcbiAgICBcIlRhZ2dpbmdcIjogW1xuICAgICAgXCJUYWdSZXNvdXJjZVwiLFxuICAgICAgXCJVbnRhZ1Jlc291cmNlXCJcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIG9yZ2FuaXphdGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbWFpbC9sYXRlc3QvYWRtaW5ndWlkZS9vcmdhbml6YXRpb25zX292ZXJ2aWV3Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Pcmdhbml6YXRpb24ocmVzb3VyY2VJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTp3b3JrbWFpbDoke1JlZ2lvbn06JHtBY2NvdW50fTpvcmdhbml6YXRpb24vJHtSZXNvdXJjZUlkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVzb3VyY2VJZH0nLCByZXNvdXJjZUlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxufVxuIl19