"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithResources = void 0;
const _3_actions_1 = require("./3-actions");
/**
 * Adds "resource" functionality to the Policy Statement
 */
class PolicyStatementWithResources extends _3_actions_1.PolicyStatementWithActions {
    constructor() {
        super(...arguments);
        this.useNotResources = false;
        this.resources = [];
        this.skipAutoResource = false;
        this.cdkResourcesApplied = false;
    }
    /**
     * Injects resources into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyResources();
            return super.toJSON();
        }
        const mode = this.useNotResources ? 'NotResource' : 'Resource';
        const statement = super.toJSON();
        const self = this;
        this.ensureResource();
        if (this.resources.length) {
            statement[mode] = this.resources.filter((elem, pos) => {
                return self.resources.indexOf(elem) == pos;
            });
        }
        return statement;
    }
    toStatementJson() {
        this.ensureResource();
        this.cdkApplyResources();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyResources() {
        if (!this.cdkResourcesApplied) {
            const mode = this.useNotResources ? 'addNotResources' : 'addResources';
            const self = this;
            const uniqueResources = this.resources.filter((elem, pos) => {
                return self.resources.indexOf(elem) == pos;
            });
            // @ts-ignore only available after swapping 1-base
            this[mode](...uniqueResources);
            this.cdkResourcesApplied = true;
        }
    }
    /**
     * Switches the statement to use [`NotResource`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notresource.html).
     */
    notResources() {
        this.useNotResources = true;
        return this;
    }
    /**
     * Checks weather any resource was applied to the policy.
     */
    hasResources() {
        return this.resources.length > 0;
    }
    /**
     * Limit statement to specified resources.
     *
     * To allow all resources, pass `*`
     */
    on(...arns) {
        this.resources.push(...arns);
        return this;
    }
    /**
     * Add all resources (`*`) to the statement
     *
     * This is the default behavior, unless the statement has principals.
     */
    onAllResources() {
        this.resources.push('*');
        return this;
    }
    ensureResource() {
        if (this.hasResources())
            return;
        // @ts-ignore only available after swapping 1-base
        if (this.hasResource)
            return;
        // @ts-ignore only available after swapping 1-base
        if (this.hasPrincipal)
            return; //assume policies may not have resources
        if (this.skipAutoResource)
            return;
        // a statement requires resources. if none was added, we assume the user wants all resources
        this.onAllResources();
    }
    /**
     * Dummy method. Will be overridden by 6-principal.ts
     *
     * We just need it here so we can reference it in method `ensureResource`
     */
    hasPrincipals() {
        return false;
    }
}
exports.PolicyStatementWithResources = PolicyStatementWithResources;
//# sourceMappingURL=data:application/json;base64,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