"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventBus = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const archive_1 = require("./archive");
const events_generated_1 = require("./events.generated");
class EventBusBase extends core_1.Resource {
    /**
     * Create an EventBridge archive to send events to.
     *
     * When you create an archive, incoming events might not immediately start being sent to the archive.
     * Allow a short period of time for changes to take effect.
     *
     * @stability stable
     */
    archive(id, props) {
        return new archive_1.Archive(this, id, {
            sourceEventBus: this,
            description: props.description || `Event Archive for ${this.eventBusName} Event Bus`,
            eventPattern: props.eventPattern,
            retention: props.retention,
            archiveName: props.archiveName,
        });
    }
    /**
     * Grants an IAM Principal to send custom events to the eventBus so that they can be matched to rules.
     *
     * @stability stable
     */
    grantPutEventsTo(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: [this.eventBusArn],
        });
    }
}
/**
 * Define an EventBridge EventBus.
 *
 * @stability stable
 * @resource AWS::Events::EventBus
 */
class EventBus extends EventBusBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        const { eventBusName, eventSourceName } = EventBus.eventBusProps(core_1.Lazy.string({ produce: () => core_1.Names.uniqueId(this) }), props);
        super(scope, id, { physicalName: eventBusName });
        const eventBus = new events_generated_1.CfnEventBus(this, 'Resource', {
            name: this.physicalName,
            eventSourceName,
        });
        this.eventBusArn = this.getResourceArnAttribute(eventBus.attrArn, {
            service: 'events',
            resource: 'event-bus',
            resourceName: eventBus.name,
        });
        this.eventBusName = this.getResourceNameAttribute(eventBus.ref);
        this.eventBusPolicy = eventBus.attrPolicy;
        this.eventSourceName = eventBus.eventSourceName;
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param eventBusArn ARN of imported event bus.
     * @stability stable
     */
    static fromEventBusArn(scope, id, eventBusArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventBusArn);
        return new ImportedEventBus(scope, id, {
            eventBusArn: eventBusArn,
            eventBusName: parts.resourceName || '',
            eventBusPolicy: '',
        });
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param eventBusName Name of imported event bus.
     * @stability stable
     */
    static fromEventBusName(scope, id, eventBusName) {
        const eventBusArn = core_1.Stack.of(scope).formatArn({
            resource: 'event-bus',
            service: 'events',
            resourceName: eventBusName,
        });
        return EventBus.fromEventBusAttributes(scope, id, {
            eventBusName: eventBusName,
            eventBusArn: eventBusArn,
            eventBusPolicy: '',
        });
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param attrs Imported event bus properties.
     * @stability stable
     */
    static fromEventBusAttributes(scope, id, attrs) {
        return new ImportedEventBus(scope, id, attrs);
    }
    /**
     * (deprecated) Permits an IAM Principal to send custom events to EventBridge so that they can be matched to rules.
     *
     * @param grantee The principal (no-op if undefined).
     * @deprecated use grantAllPutEvents instead
     */
    static grantPutEvents(grantee) {
        // It's currently not possible to restrict PutEvents to specific resources.
        // See https://docs.aws.amazon.com/eventbridge/latest/userguide/permissions-reference-eventbridge.html
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: ['*'],
        });
    }
    /**
     * Permits an IAM Principal to send custom events to EventBridge so that they can be matched to rules.
     *
     * @param grantee The principal (no-op if undefined).
     * @stability stable
     */
    static grantAllPutEvents(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: ['*'],
        });
    }
    static eventBusProps(defaultEventBusName, props) {
        if (props) {
            const { eventBusName, eventSourceName } = props;
            const eventBusNameRegex = /^[\/\.\-_A-Za-z0-9]{1,256}$/;
            if (eventBusName !== undefined && eventSourceName !== undefined) {
                throw new Error('\'eventBusName\' and \'eventSourceName\' cannot both be provided');
            }
            else if (eventBusName !== undefined && !core_1.Token.isUnresolved(eventBusName)) {
                if (eventBusName === 'default') {
                    throw new Error('\'eventBusName\' must not be \'default\'');
                }
                else if (eventBusName.indexOf('/') > -1) {
                    throw new Error('\'eventBusName\' must not contain \'/\'');
                }
                else if (!eventBusNameRegex.test(eventBusName)) {
                    throw new Error(`'eventBusName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName };
            }
            else if (eventSourceName !== undefined) {
                // Ex: aws.partner/PartnerName/acct1/repo1
                const eventSourceNameRegex = /^aws\.partner(\/[\.\-_A-Za-z0-9]+){2,}$/;
                if (!eventSourceNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventSourceNameRegex}`);
                }
                else if (!eventBusNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName: eventSourceName, eventSourceName };
            }
            else {
                return { eventBusName: props.eventBusName };
            }
        }
        return { eventBusName: defaultEventBusName };
    }
}
exports.EventBus = EventBus;
_a = JSII_RTTI_SYMBOL_1;
EventBus[_a] = { fqn: "@aws-cdk/aws-events.EventBus", version: "1.131.0" };
class ImportedEventBus extends EventBusBase {
    constructor(scope, id, attrs) {
        const arnParts = core_1.Stack.of(scope).parseArn(attrs.eventBusArn);
        super(scope, id, {
            account: arnParts.account,
            region: arnParts.region,
        });
        this.eventBusArn = attrs.eventBusArn;
        this.eventBusName = attrs.eventBusName;
        this.eventBusPolicy = attrs.eventBusPolicy;
        this.eventSourceName = attrs.eventSourceName;
    }
}
//# sourceMappingURL=data:application/json;base64,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