"""Tests the functions in post-processing related to range parsing
and expanding.
"""
import pytest

from naluconfigs.exceptions import (
    PostProcessingError,
    RangeParsingError,
)
from naluconfigs.postprocess import (
    process_configuration,
    _expand_range_str,
)


@pytest.fixture
def valid_ranges():
    ranges = {
        '0..5': range(6),
        '0 .. 5': range(6),
        '2..5': range(2, 6),
        '-2..2': range(-2, 3),
        '2..-2': range(-2, 3),
        '0..5,10..15': [*range(0, 6), *range(10, 16)],
        '1,2,3': [1, 2, 3],
        '1, 2, 3': [1, 2, 3],
        '1, 2..4, 5': range(1, 6),
    }
    return {k: list(v) for k, v in ranges.items()}


@pytest.fixture
def invalid_range_strings():
    """Range strings that are not parsable.
    """
    return [
        ',',
        '..',
        '0..',
        '..0',
        '0..1..2',
        '1..2, 3..',
        'asdf..asdf'
    ]


@pytest.fixture
def invalid_range_keys(invalid_range_strings):
    """Range strings that generate a valid range

    The difference between range keys and simply strings
    is that a string only needs to be parsable, while a key
    must be valid to be used in a dictionary: i.e., no duplicates!
    """
    return invalid_range_strings + ['0..31, 16..31']


def test_range_expansion(valid_ranges, invalid_range_keys):
    """Tests whether valid range keys generate the correct dictionaries,
    and invalid range keys raise an error.
    """
    for range_str, expected in valid_ranges.items():
        result = process_configuration({range_str: 'some value'})
        assert set(result.keys()) == set(expected) # order doesn't matter
        assert set(result.values()) == {'some value'}

    for range_str in invalid_range_keys:
        with pytest.raises(PostProcessingError):
            process_configuration({range_str: 'some_value'})


def test_deepcopy_values():
    """Test whether keys get properly deep-copied when key range is expanded.
    Each and every value generated by the expansion should have different ids
    for mutable types
    """
    reference_value = {'key': 0}
    config = {'0..15': reference_value}
    result = process_configuration(config)

    assert len(result) == 16
    for key in result:
        assert result[key] is not reference_value
        assert result[key] == reference_value


def test_recursive_expansion():
    """Test recursive expansion of ranged keys. Each and every resulting
    element must be unique objects (if mutable).
    """
    config = {
        '0..15': {
            '-10..10': {}
        }
    }
    expected_inner = {k: {} for k in range(-10, 11)}
    expected_outer = {k: expected_inner for k in range(16)}

    result = process_configuration(config)
    assert result == expected_outer

    # Each generated value should have a unique id
    assert len(set(id(x) for x in result.values())) == len(result)


def test_range_parsing(valid_ranges, invalid_range_strings):
    """Tests whether the parsing a range string results in the
    correct outcome. This tests the internal function.
    """
    for range_str, expected_result in valid_ranges.items():
        result = _expand_range_str(range_str)
        assert result == expected_result

    for range_str in invalid_range_strings:
        with pytest.raises(RangeParsingError):
            _expand_range_str(range_str)

    with pytest.raises(TypeError):
        _expand_range_str(0)
    with pytest.raises(TypeError):
        _expand_range_str([])

