import os
import platform
import re
import subprocess
import sys

from distutils.version import LooseVersion
from setuptools import setup, Extension
from setuptools.command.build_ext import build_ext


class CMakeExtension(Extension):
    def __init__(self, name, sourcedir=""):
        Extension.__init__(self, name, sources=[])
        self.sourcedir = os.path.abspath(sourcedir)


class CMakeBuild(build_ext):
    def run(self):
        try:
            out = subprocess.check_output(["cmake", "--version"])
        except OSError:
            raise RuntimeError(
                "CMake must be installed to build the following extensions: "
                + ", ".join(e.name for e in self.extensions)
            )

        cmake_version = LooseVersion(
            re.search(r"version\s*([\d.]+)", out.decode()).group(1)
        )
        if cmake_version < "3.3.0":
            raise RuntimeError("CMake >= 3.3.0 is required")

        for ext in self.extensions:
            self.build_extension(ext)

    def build_extension(self, ext):
        extdir = os.path.abspath(os.path.dirname(self.get_ext_fullpath(ext.name)))
        cmake_args = [
            "-DCMAKE_LIBRARY_OUTPUT_DIRECTORY=" + extdir,
            "-DBUILD_PYTHON_BINDINGS:BOOL=ON",
            # Unit tests somehow pull in MPI
            "-DBUILD_UNIT_TESTS:BOOL=OFF",
            "-DMVDTool_VERSION=" + self.distribution.get_version(),
            "-DPYTHON_EXECUTABLE=" + sys.executable,
            "-DEXTLIB_FROM_SUBMODULES=ON",
            "-DMVD_INSTALL_DEV=OFF",
        ]

        cfg = "Debug" if self.debug else "Release"
        build_args = ["--config", cfg]

        if platform.system() == "Windows":
            cmake_args += [
                "-DCMAKE_LIBRARY_OUTPUT_DIRECTORY_{}={}".format(cfg.upper(), extdir),
            ]
            if sys.maxsize > 2 ** 32:
                cmake_args += ["-A", "x64"]
            build_args += ["--", "/m"]
        else:
            cmake_args += ["-DCMAKE_BUILD_TYPE=" + cfg]
            build_args += ["--", "-j4"]

        env = os.environ.copy()
        env["CXXFLAGS"] = '{} -DVERSION_INFO=\\"{}\\"'.format(
            env.get("CXXFLAGS", ""), self.distribution.get_version()
        )
        if not os.path.exists(self.build_temp):
            os.makedirs(self.build_temp)
        subprocess.check_call(
            ["cmake", ext.sourcedir] + cmake_args, cwd=self.build_temp, env=env
        )
        subprocess.check_call(
            ["cmake", "--build", "."] + build_args, cwd=self.build_temp
        )

setup(
    name='MVDTool',
    use_scm_version={'local_scheme': "no-local-version"},
    author='Blue Brain Project, EPFL',
    author_email='bbp-ou-hpc@groupes.epfl.ch',
    url='https://github.com/BlueBrain/MVDTool',
    description='Python bindings to the MVD3 neuroscience file format',
    long_description='',
    packages=[],
    ext_modules=[
        CMakeExtension('mvdtool')
    ],
    install_requires=[
        'numpy',
    ],
    setup_requires=['pytest-runner', 'setuptools_scm'],
    tests_require=['pytest'],
    cmdclass=dict(
        build_ext=CMakeBuild,
    ),
    zip_safe=False,
)
