#!/usr/bin/env python
# -*- coding: utf-8 -*-

import re
import black
import configparser


def normalize_string_quotes(leaf: black.Leaf) -> None:
    """
    Prefer SINGLE quotes but only if it doesn't cause more escaping.
    Prefer double quotes for docstrings.

    Adds or removes backslashes as appropriate. Doesn't parse and fix
    strings nested in f-strings (yet).

    Note: Mutates its argument.
    """
    value = leaf.value.lstrip('furbFURB')
    if value[:3] == "'''":
        return

    elif value[:3] == '"""':
        orig_quote = '"""'
        new_quote = '"""'
    elif value[0] == '"':
        orig_quote = '"'
        new_quote = "'"
    else:
        orig_quote = "'"
        new_quote = '"'
    first_quote_pos = leaf.value.find(orig_quote)
    if first_quote_pos == -1:
        return  # There's an internal error

    prefix = leaf.value[:first_quote_pos]
    unescaped_new_quote = re.compile(rf'(([^\\]|^)(\\\\)*){new_quote}')
    escaped_new_quote = re.compile(rf'([^\\]|^)\\((?:\\\\)*){new_quote}')
    escaped_orig_quote = re.compile(rf'([^\\]|^)\\((?:\\\\)*){orig_quote}')
    body = leaf.value[first_quote_pos + len(orig_quote): -len(orig_quote)]
    if 'r' in prefix.casefold():
        if unescaped_new_quote.search(body):
            # There's at least one unescaped new_quote in this raw string
            # so converting is impossible
            return
        # Do not introduce or remove backslashes in raw strings
        new_body = body
    else:
        # remove unnecessary escapes
        new_body = black.sub_twice(
            escaped_new_quote, rf'\1\2{new_quote}', body
        )
        if body != new_body:
            # Consider the string without unnecessary escapes as the original
            body = new_body
            leaf.value = f'{prefix}{orig_quote}{body}{orig_quote}'
        new_body = black.sub_twice(
            escaped_orig_quote, rf'\1\2{orig_quote}', new_body
        )
        new_body = black.sub_twice(
            unescaped_new_quote, rf'\1\\{new_quote}', new_body
        )
    if 'f' in prefix.casefold():
        matches = re.findall(r'^{]\{(.*?)\}[^}]', new_body)
        for m in matches:
            if '\\' in str(m):
                # Do not introduce backslashes in interpolated expressions
                return
    if new_quote == '"""' and new_body[-1:] == '"':
        # edge case:
        new_body = new_body[:-1] + '\\"'
    orig_escape_count = body.count('\\')
    new_escape_count = new_body.count('\\')
    if new_escape_count > orig_escape_count:
        return  # Do not introduce more escaping

    if new_escape_count == orig_escape_count and orig_quote == "'":
        return  # Prefer single quotes

    leaf.value = f'{prefix}{new_quote}{new_body}{new_quote}'


def read_config_file(ctx, param, value):
    if not value:
        root = black.find_project_root(ctx.params.get('src', ()))
        path = root / 'setup.cfg'
        if path.is_file():
            value = str(path)
        else:
            return None

    config = configparser.ConfigParser()
    config.read(value)
    try:
        config = dict(config['tool:brunette'])
    except KeyError:
        return None
    if not config:
        return None

    try:
        if config['single-quotes'].lower() == 'true':
            black.normalize_string_quotes = normalize_string_quotes
    except KeyError:
        pass

    if ctx.default_map is None:
        ctx.default_map = {}
    ctx.default_map.update(  # type: ignore  # bad types in .pyi
        {k.replace('--', '').replace('-', '_'): v for k, v in config.items()}
    )
    return value


def main():
    config_file_opt = [p for p in black.main.params if p.name == 'config'][0]
    config_file_opt.callback = read_config_file
    options = [p for p in black.main.params if p.name != 'config']
    options.append(config_file_opt)
    black.main.params = options
    return black.main()


def cli_main():
    black.freeze_support()
    black.patch_click()
    black.main = main


if __name__ == '__main__':
    cli_main()
