"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderEventPattern = exports.sameEnvDimension = exports.mergeEventPattern = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * Merge the `src` event pattern into the `dest` event pattern by adding all
 * values from `src` into the fields in `dest`.
 *
 * See `rule.addEventPattern` for details.
 */
function mergeEventPattern(dest, src) {
    dest = dest || {};
    mergeObject(dest, src);
    return dest;
    function mergeObject(destObj, srcObj) {
        if (typeof (srcObj) !== 'object') {
            throw new Error(`Invalid event pattern '${JSON.stringify(srcObj)}', expecting an object or an array`);
        }
        for (const field of Object.keys(srcObj)) {
            const srcValue = srcObj[field];
            const destValue = destObj[field];
            if (srcValue === undefined) {
                continue;
            }
            if (typeof (srcValue) !== 'object') {
                throw new Error(`Invalid event pattern field { ${field}: ${JSON.stringify(srcValue)} }. All fields must be arrays`);
            }
            // dest doesn't have this field
            if (destObj[field] === undefined) {
                destObj[field] = srcValue;
                continue;
            }
            if (Array.isArray(srcValue) !== Array.isArray(destValue)) {
                throw new Error(`Invalid event pattern field ${field}. ` +
                    `Type mismatch between existing pattern ${JSON.stringify(destValue)} and added pattern ${JSON.stringify(srcValue)}`);
            }
            // if this is an array, concat and deduplicate the values
            if (Array.isArray(srcValue)) {
                const result = [...destValue, ...srcValue];
                const resultJson = result.map(i => JSON.stringify(i));
                destObj[field] = result.filter((value, index) => resultJson.indexOf(JSON.stringify(value)) === index);
                continue;
            }
            // otherwise, it's an object, so recurse
            mergeObject(destObj[field], srcValue);
        }
    }
}
exports.mergeEventPattern = mergeEventPattern;
/**
 * Whether two string probably contain the same environment dimension (region or account)
 *
 * Used to compare either accounts or regions, and also returns true if both
 * are unresolved (in which case both are expted to be "current region" or "current account").
 * @internal
 */
function sameEnvDimension(dim1, dim2) {
    return [core_1.TokenComparison.SAME, core_1.TokenComparison.BOTH_UNRESOLVED].includes(core_1.Token.compareStrings(dim1, dim2));
}
exports.sameEnvDimension = sameEnvDimension;
/**
 * Transform an eventPattern object into a valid Event Rule Pattern
 * by changing detailType into detail-type when present.
 */
function renderEventPattern(eventPattern) {
    if (Object.keys(eventPattern).length === 0) {
        return undefined;
    }
    // rename 'detailType' to 'detail-type'
    const out = {};
    for (let key of Object.keys(eventPattern)) {
        const value = eventPattern[key];
        if (key === 'detailType') {
            key = 'detail-type';
        }
        out[key] = value;
    }
    return out;
}
exports.renderEventPattern = renderEventPattern;
//# sourceMappingURL=data:application/json;base64,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