"""
Functions to compute throughput.
"""
from typing import List, Dict, Tuple

import numpy as np
import pandas as pd

from applefy.utils.photometry import AperturePhotometryMode, get_flux


def _compute_throughput_fixed_contrast(
        fake_planet_results,
        fp_residual,
        stellar_flux,
        inserted_flux_ratio,
        photometry_mode: AperturePhotometryMode):

    """
    Computes the throughput for a given list of fake planet residuals and
    flux ratio. The brightness of the inserted fake planets has to be fixed.
    In general the function compute_throughput_table should be used instead of
    this function.

    Args:
        fake_planet_results: List containing the residuals (2d np.array) and
            position (tuple of floats) of the fake planet experiments. Format:
            [(res_planet_a, pos_planet_a), (res_planet_b, pos_planet_b),...]
        fp_residual: A 2D numpy array of the residual without any fake planets
        stellar_flux: The stellar flux estimated by estimate_stellar_flux.
            Note: The settings for the used computation mode have to be the same
            as used by this function
        inserted_flux_ratio: The flux ratio of the fake planets inserted (float)
            Has to be the same for all input residuals
        photometry_mode: An instance of AperturePhotometryMode which defines
            how the flux is measured at the planet positions.

    Returns:
        List of throughput values for every planet
    """

    flux_in = stellar_flux * inserted_flux_ratio
    all_throughput_results = []

    # Loop over all residuals
    for tmp_planet_result in fake_planet_results:
        tmp_planet_residual, tmp_position = tmp_planet_result

        # Note: This line assumes that a linear PSF-subtraction method was used
        planet_signal_offset = tmp_planet_residual - fp_residual

        # sometimes negative self-subtraction wings are very close to the
        # planet. In oder to remove this effect we set these values to 0
        planet_signal_offset[planet_signal_offset < 0] = 0

        _, tmp_flux_out = get_flux(planet_signal_offset,
                                   tmp_position[:2],
                                   photometry_mode=photometry_mode)

        tmp_throughput = tmp_flux_out / flux_in

        # negative throughput can happen due to numerical problems
        tmp_throughput = max(tmp_throughput, 0)

        all_throughput_results.append(tmp_throughput)

    return all_throughput_results


def compute_throughput_table(
        planet_dict: Dict[int,
                          List[Tuple[np.ndarray,
                                     List[float]]]],
        fp_residual: np.ndarray,
        idx_table: pd.DataFrame,
        stellar_flux: float,
        photometry_mode_planet: AperturePhotometryMode
) -> Tuple[Dict[str, List[float]], pd.DataFrame]:
    """
    Computes the throughput for a series of experiments as generated by function
    sort_results.

    Args:
        planet_dict: A dictionary with keys = Experiment ID. For every ID a
            list is given which contains tuples of the residual and the position
            of the corresponding fake planet. E.g.:

            .. highlight:: python
            .. code-block:: python

                planet_dict["0001"] = [
                    (res_planet_a, pos_planet_a),
                    (res_planet_b, pos_planet_b),
                    (res_planet_c, pos_planet_c),
                    (res_planet_d, pos_planet_d),
                    (res_planet_e, pos_planet_e),
                    (res_planet_f, pos_planet_f)]
        fp_residual: A 2D numpy array of shape `(width, height)` containing
            the data on which to run the noise statistics. This is usually the
            residual without fake planets.
        idx_table: Pandas table which links separation and flux_ratio
            to its experiment ID as used by planet_dict.
        stellar_flux: The stellar flux measured with
            :meth:`~applefy.utils.photometry.estimate_stellar_flux`. The mode
            used to estimate the stellar_flux has to be compatible with the
            photometry_mode_noise used here.
        photometry_mode_planet: An instance of AperturePhotometryMode which
            defines how the flux is measured at the planet positions.

    Returns:
        1. throughput_dict - A dictionary with keys = Experiment ID.
        For every ID a list is given which contains the throughput for every
        residual.

        2. throughput_table - Pandas table which contains the median throughput
        as a function of separation and inserted flux_ratio
    """

    throughput_dict = dict()
    throughput_table = idx_table.copy()

    for separation, row in idx_table.iterrows():
        for tmp_flux_ratio, tmp_idx in row.items():

            tmp_fake_planet_results = planet_dict[tmp_idx]

            tmp_throughput_list = _compute_throughput_fixed_contrast(
                tmp_fake_planet_results,
                fp_residual,
                stellar_flux,
                tmp_flux_ratio,
                photometry_mode=photometry_mode_planet)

            throughput_dict[tmp_idx] = tmp_throughput_list
            throughput_table.loc[separation, tmp_flux_ratio] =\
                np.median(tmp_throughput_list)

    return throughput_dict, throughput_table
