"""
URLconf for login-protected views of the cms package. These urls are processed by
:mod:`~integreat_cms.core.middleware.access_control_middleware.AccessControlMiddleware`.
Views which should not have login protection go into :mod:`~integreat_cms.cms.urls.public`.
"""
from django.urls import include, path

from ..forms import (
    LanguageForm,
    OfferTemplateForm,
    OrganizationForm,
    RegionForm,
    EventTranslationForm,
    POITranslationForm,
)
from ..models import Event, Language, OfferTemplate, Organization, Page, POI, Role

from ..views import (
    analytics,
    bulk_action_views,
    chat,
    dashboard,
    delete_views,
    events,
    form_views,
    language_tree,
    list_views,
    linkcheck,
    media,
    imprint,
    pages,
    pois,
    push_notifications,
    regions,
    roles,
    settings,
    statistics,
    users,
    utils,
    feedback,
)

#: The media library ajax url patterns are reused twice (for the admin media library and the region media library)
media_ajax_urlpatterns = [
    path(
        "media/",
        include(
            [
                path(
                    "directory/",
                    include(
                        [
                            path(
                                "path/",
                                media.get_directory_path_ajax,
                                name="mediacenter_directory_path",
                            ),
                            path(
                                "content/",
                                media.get_directory_content_ajax,
                                name="mediacenter_get_directory_content",
                            ),
                            path(
                                "create/",
                                media.create_directory_ajax,
                                name="mediacenter_create_directory",
                            ),
                            path(
                                "update/",
                                media.edit_directory_ajax,
                                name="mediacenter_edit_directory",
                            ),
                            path(
                                "delete/",
                                media.delete_directory_ajax,
                                name="mediacenter_delete_directory",
                            ),
                        ]
                    ),
                ),
                path(
                    "file/",
                    include(
                        [
                            path(
                                "upload/",
                                media.upload_file_ajax,
                                name="mediacenter_upload_file",
                            ),
                            path(
                                "edit/",
                                media.edit_file_ajax,
                                name="mediacenter_edit_file",
                            ),
                            path(
                                "delete/",
                                media.delete_file_ajax,
                                name="mediacenter_delete_file",
                            ),
                            path(
                                "replace/",
                                media.replace_file_ajax,
                                name="mediacenter_replace_file",
                            ),
                        ]
                    ),
                ),
                path(
                    "search/",
                    media.get_query_search_results_ajax,
                    name="mediacenter_get_search_result",
                ),
            ]
        ),
    ),
]

#: The user setting url patterns are reused twice (for the staff area and the region area)
user_settings_urlpatterns = [
    path(
        "user-settings/",
        include(
            [
                path("", settings.UserSettingsView.as_view(), name="user_settings"),
                path(
                    "mfa/",
                    include(
                        [
                            path(
                                "authenticate/",
                                settings.AuthenticateModifyMfaView.as_view(),
                                name="authenticate_modify_mfa",
                            ),
                            path(
                                "get-challenge/",
                                settings.GetMfaChallengeView.as_view(),
                                name="get_mfa_challenge",
                            ),
                            path(
                                "register/",
                                settings.RegisterUserMfaKeyView.as_view(),
                                name="register_new_mfa_key",
                            ),
                            path(
                                "delete/<int:key_id>/",
                                settings.DeleteUserMfaKeyView.as_view(),
                                name="delete_mfa_key",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
]

#: The page order table url patterns are reused twice (for root pages and for subpages)
page_order_table_urlpatterns = [
    path(
        "",
        pages.get_page_order_table_ajax,
        name="get_page_order_table_ajax",
    ),
    path(
        "page-<int:page_id>/",
        pages.get_page_order_table_ajax,
        name="get_page_order_table_ajax",
    ),
]

#: The url patterns of this module (see :doc:`topics/http/urls`)
urlpatterns = [
    path(
        "admin-dashboard/",
        dashboard.AdminDashboardView.as_view(),
        name="admin_dashboard",
    ),
    path(
        "regions/",
        include(
            [
                path("", regions.RegionListView.as_view(), name="regions"),
                path(
                    "new/",
                    form_views.CustomCreateView.as_view(form_class=RegionForm),
                    name="new_region",
                ),
                path(
                    "<slug>/",
                    include(
                        [
                            path(
                                "edit/",
                                form_views.CustomUpdateView.as_view(
                                    form_class=RegionForm
                                ),
                                name="edit_region",
                            ),
                            path(
                                "delete/", regions.delete_region, name="delete_region"
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path("media-library/", media.AdminMediaListView.as_view(), name="media_admin"),
    path(
        "languages/",
        include(
            [
                path(
                    "",
                    list_views.ModelListView.as_view(
                        queryset=Language.objects.all().prefetch_related(
                            "language_tree_nodes"
                        )
                    ),
                    name="languages",
                ),
                path(
                    "new/",
                    form_views.CustomCreateView.as_view(form_class=LanguageForm),
                    name="new_language",
                ),
                path(
                    "<slug:slug>/",
                    include(
                        [
                            path(
                                "edit/",
                                form_views.CustomUpdateView.as_view(
                                    form_class=LanguageForm
                                ),
                                name="edit_language",
                            ),
                            path(
                                "delete/",
                                delete_views.CustomDeleteView.as_view(
                                    model=Language,
                                ),
                                name="delete_language",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path(
        "users/",
        include(
            [
                path("", users.UserListView.as_view(), name="users"),
                path("new/", users.UserFormView.as_view(), name="new_user"),
                path(
                    "<int:user_id>/",
                    include(
                        [
                            path(
                                "edit/",
                                users.UserFormView.as_view(),
                                name="edit_user",
                            ),
                            path("delete/", users.delete_user, name="delete_user"),
                            path(
                                "resend-activation-link",
                                users.resend_activation_link,
                                name="resend_activation_link",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path(
        "roles/",
        include(
            [
                path("", list_views.ModelListView.as_view(model=Role), name="roles"),
                path("new/", roles.RoleFormView.as_view(), name="new_role"),
                path(
                    "<int:role_id>/",
                    include(
                        [
                            path(
                                "edit/",
                                roles.RoleFormView.as_view(),
                                name="edit_role",
                            ),
                            path(
                                "delete/",
                                roles.RoleFormView.as_view(),
                                name="delete_role",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path(
        "organizations/",
        include(
            [
                path(
                    "",
                    list_views.ModelListView.as_view(model=Organization),
                    name="organizations",
                ),
                path(
                    "new/",
                    form_views.CustomCreateView.as_view(form_class=OrganizationForm),
                    name="new_organization",
                ),
                path(
                    "<slug>/",
                    include(
                        [
                            path(
                                "edit/",
                                form_views.CustomUpdateView.as_view(
                                    form_class=OrganizationForm
                                ),
                                name="edit_organization",
                            ),
                            path(
                                "delete/",
                                delete_views.CustomDeleteView.as_view(
                                    model=Organization,
                                ),
                                name="delete_organization",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path(
        "feedback/",
        include(
            [
                path(
                    "",
                    feedback.AdminFeedbackListView.as_view(),
                    name="admin_feedback",
                ),
                path(
                    "mark-read/",
                    feedback.mark_admin_feedback_as_read,
                    name="mark_admin_feedback_as_read",
                ),
                path(
                    "mark-unread/",
                    feedback.mark_admin_feedback_as_unread,
                    name="mark_admin_feedback_as_unread",
                ),
                path(
                    "delete/",
                    feedback.delete_admin_feedback,
                    name="delete_admin_feedback",
                ),
            ]
        ),
    ),
    path(
        "offer-templates/",
        include(
            [
                path(
                    "",
                    list_views.ModelListView.as_view(
                        queryset=OfferTemplate.objects.all().prefetch_related("regions")
                    ),
                    name="offertemplates",
                ),
                path(
                    "new/",
                    form_views.CustomCreateView.as_view(form_class=OfferTemplateForm),
                    name="new_offertemplate",
                ),
                path(
                    "<slug:slug>/",
                    include(
                        [
                            path(
                                "edit/",
                                form_views.CustomUpdateView.as_view(
                                    form_class=OfferTemplateForm
                                ),
                                name="edit_offertemplate",
                            ),
                            path(
                                "delete/",
                                delete_views.CustomDeleteView.as_view(
                                    model=OfferTemplate, protect_manytomany="regions"
                                ),
                                name="delete_offertemplate",
                            ),
                        ]
                    ),
                ),
            ]
        ),
    ),
    path("", include(user_settings_urlpatterns)),
    path(
        "ajax/",
        include(
            [
                path("", include(media_ajax_urlpatterns)),
                path(
                    "chat/",
                    include(
                        [
                            path(
                                "send-message/",
                                chat.send_chat_message,
                                name="send_chat_message",
                            ),
                            path(
                                "delete-message/<int:message_id>/",
                                chat.delete_chat_message,
                                name="delete_chat_message",
                            ),
                        ]
                    ),
                ),
                path(
                    "search/",
                    utils.search_content_ajax,
                    name="search_content_ajax",
                ),
            ]
        ),
    ),
    path(
        "<region_slug>/",
        include(
            [
                path("", dashboard.DashboardView.as_view(), name="dashboard"),
                path(
                    "ajax/",
                    include(
                        [
                            path("", include(media_ajax_urlpatterns)),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "<int:page_id>/",
                                            include(
                                                [
                                                    path(
                                                        "cancel-translation-process/",
                                                        pages.cancel_translation_process_ajax,
                                                        name="cancel_translation_process_ajax",
                                                    ),
                                                    path(
                                                        "preview/",
                                                        pages.preview_page_ajax,
                                                        name="preview_page_ajax",
                                                    ),
                                                    path(
                                                        "content/",
                                                        pages.get_page_content_ajax,
                                                        name="get_page_content_ajax",
                                                    ),
                                                ]
                                            ),
                                        ),
                                        path(
                                            "search/",
                                            utils.search_content_ajax,
                                            name="search_content_ajax",
                                        ),
                                        path(
                                            "<model_type>/slugify/",
                                            utils.slugify_ajax,
                                            name="slugify_ajax",
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "render/",
                                include(
                                    [
                                        path(
                                            "<slug:language_slug>/",
                                            include(
                                                [
                                                    path(
                                                        "mirrored-page-field/",
                                                        pages.render_mirrored_page_field,
                                                        name="render_mirrored_page_field",
                                                    ),
                                                    path(
                                                        "partial-page-tree/tree-<int:tree_id>/",
                                                        pages.PartialPageTreeView.as_view(),
                                                        name="get_page_tree_ajax",
                                                    ),
                                                ]
                                            ),
                                        ),
                                        path(
                                            "page-order-table/",
                                            include(
                                                [
                                                    path(
                                                        "",
                                                        include(
                                                            page_order_table_urlpatterns
                                                        ),
                                                    ),
                                                    path(
                                                        "parent-<int:parent_id>/",
                                                        include(
                                                            page_order_table_urlpatterns
                                                        ),
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "chat/",
                                include(
                                    [
                                        path(
                                            "send-message/",
                                            chat.send_chat_message,
                                            name="send_chat_message",
                                        ),
                                        path(
                                            "delete-message/<int:message_id>/",
                                            chat.delete_chat_message,
                                            name="delete_chat_message",
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "statistics/",
                                include(
                                    [
                                        path(
                                            "total-views/",
                                            statistics.get_total_visits_ajax,
                                            name="statistics_total_visits",
                                        ),
                                        path(
                                            "update-chart/",
                                            statistics.get_visits_per_language_ajax,
                                            name="statistics_visits_per_language",
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "page-permission/",
                                include(
                                    [
                                        path(
                                            "grant/",
                                            pages.grant_page_permission_ajax,
                                            name="grant_page_permission_ajax",
                                        ),
                                        path(
                                            "revoke/",
                                            pages.revoke_page_permission_ajax,
                                            name="revoke_page_permission_ajax",
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "content-edit-lock/",
                                include(
                                    [
                                        path(
                                            "heartbeat/",
                                            utils.content_edit_lock_heartbeat,
                                            name="content_edit_lock_heartbeat",
                                        ),
                                        path(
                                            "release/",
                                            utils.content_edit_lock_release,
                                            name="content_edit_lock_release",
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "search-poi/",
                                events.search_poi_ajax,
                                name="search_poi_ajax",
                            ),
                            path(
                                "search/",
                                utils.search_content_ajax,
                                name="search_content_ajax",
                            ),
                            path(
                                "dismiss-tutorial/<slug:slug>/",
                                settings.DismissTutorial.as_view(),
                                name="dismiss_tutorial",
                            ),
                        ]
                    ),
                ),
                path(
                    "analytics/",
                    include(
                        [
                            path(
                                "",
                                analytics.AnalyticsView.as_view(),
                                name="analytics",
                            ),
                            path(
                                "statistics/",
                                statistics.AnalyticsView.as_view(),
                                name="statistics",
                            ),
                            path(
                                "translation-coverage/",
                                analytics.TranslationCoverageView.as_view(),
                                name="translation_coverage",
                            ),
                            path(
                                "linkcheck/",
                                include(
                                    [
                                        path(
                                            "",
                                            linkcheck.LinkcheckRedirectView.as_view(),
                                            name="linkcheck_landing",
                                        ),
                                        path(
                                            "stats",
                                            linkcheck.LinkcheckStatsView.as_view(),
                                            name="linkcheck_stats",
                                        ),
                                        path(
                                            "<slug:url_filter>/",
                                            include(
                                                [
                                                    path(
                                                        "",
                                                        linkcheck.LinkcheckListView.as_view(),
                                                        name="linkcheck",
                                                    ),
                                                    path(
                                                        "<int:url_id>/",
                                                        linkcheck.LinkcheckListView.as_view(),
                                                        name="edit_url",
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "pages/",
                    include(
                        [
                            path("", pages.PageTreeView.as_view(), name="pages"),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "",
                                            pages.PageTreeView.as_view(),
                                            name="pages",
                                        ),
                                        path(
                                            "new/",
                                            pages.PageFormView.as_view(),
                                            name="new_page",
                                        ),
                                        path(
                                            "archived/",
                                            pages.PageTreeView.as_view(archived=True),
                                            name="archived_pages",
                                        ),
                                        path(
                                            "xliff/",
                                            include(
                                                [
                                                    path(
                                                        "download/",
                                                        pages.ExportXliffView.as_view(
                                                            prefetch_translations=True
                                                        ),
                                                        name="download_xliff",
                                                    ),
                                                    path(
                                                        "only-public/",
                                                        pages.ExportXliffView.as_view(
                                                            only_public=True,
                                                            prefetch_public_translations=True,
                                                        ),
                                                        name="download_xliff_only_public",
                                                    ),
                                                    path(
                                                        "upload/",
                                                        pages.upload_xliff,
                                                        name="upload_xliff",
                                                    ),
                                                    path(
                                                        "import/<uuid:xliff_dir>/",
                                                        pages.PageXliffImportView.as_view(),
                                                        name="import_xliff",
                                                    ),
                                                ],
                                            ),
                                        ),
                                        path(
                                            "export/",
                                            pages.GeneratePdfView.as_view(),
                                            name="export_pdf",
                                        ),
                                        path(
                                            "bulk-archive/",
                                            bulk_action_views.BulkArchiveView.as_view(
                                                model=Page,
                                                archived_field="explicitly_archived",
                                            ),
                                            name="bulk_archive_pages",
                                        ),
                                        path(
                                            "bulk-restore/",
                                            bulk_action_views.BulkRestoreView.as_view(
                                                model=Page,
                                                archived_field="explicitly_archived",
                                            ),
                                            name="bulk_restore_pages",
                                        ),
                                        path(
                                            "<int:page_id>/",
                                            include(
                                                [
                                                    path(
                                                        "edit/",
                                                        include(
                                                            [
                                                                path(
                                                                    "",
                                                                    pages.PageFormView.as_view(),
                                                                    name="edit_page",
                                                                ),
                                                                path(
                                                                    "side-by-side/",
                                                                    pages.PageSideBySideView.as_view(),
                                                                    name="sbs_edit_page",
                                                                ),
                                                            ],
                                                        ),
                                                    ),
                                                    path(
                                                        "revisions/",
                                                        include(
                                                            [
                                                                path(
                                                                    "",
                                                                    pages.PageRevisionView.as_view(),
                                                                    name="page_revisions",
                                                                ),
                                                                path(
                                                                    "<int:selected_revision>/",
                                                                    pages.PageRevisionView.as_view(),
                                                                    name="page_revisions",
                                                                ),
                                                            ],
                                                        ),
                                                    ),
                                                    path(
                                                        "archive/",
                                                        pages.archive_page,
                                                        name="archive_page",
                                                    ),
                                                    path(
                                                        "restore/",
                                                        pages.restore_page,
                                                        name="restore_page",
                                                    ),
                                                    path(
                                                        "refresh/",
                                                        pages.refresh_date,
                                                        name="refresh_page",
                                                    ),
                                                    path(
                                                        "delete/",
                                                        pages.delete_page,
                                                        name="delete_page",
                                                    ),
                                                    # warning: the move url is also hardcoded in src/integreat_cms/static/js/tree_drag_and_drop.js
                                                    path(
                                                        "move/<int:target_id>/<slug:position>/",
                                                        pages.move_page,
                                                        name="move_page",
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "imprint/",
                    include(
                        [
                            path(
                                "",
                                imprint.ImprintFormView.as_view(),
                                name="edit_imprint",
                            ),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "edit/",
                                            include(
                                                [
                                                    path(
                                                        "",
                                                        imprint.ImprintFormView.as_view(),
                                                        name="edit_imprint",
                                                    ),
                                                    path(
                                                        "side-by-side/",
                                                        imprint.ImprintSideBySideView.as_view(),
                                                        name="sbs_edit_imprint",
                                                    ),
                                                ],
                                            ),
                                        ),
                                        path(
                                            "revisions/",
                                            include(
                                                [
                                                    path(
                                                        "",
                                                        imprint.ImprintRevisionView.as_view(),
                                                        name="imprint_revisions",
                                                    ),
                                                    path(
                                                        "<int:selected_revision>",
                                                        imprint.ImprintRevisionView.as_view(),
                                                        name="imprint_revisions",
                                                    ),
                                                ],
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                            path(
                                "delete/",
                                imprint.delete_imprint,
                                name="delete_imprint",
                            ),
                        ]
                    ),
                ),
                # TODO: Change destination for delete_event, add view_event
                path(
                    "events/",
                    include(
                        [
                            path("", events.EventListView.as_view(), name="events"),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "",
                                            events.EventListView.as_view(),
                                            name="events",
                                        ),
                                        path(
                                            "archived/",
                                            events.EventListView.as_view(archived=True),
                                            name="events_archived",
                                        ),
                                        path(
                                            "new/",
                                            events.EventFormView.as_view(),
                                            name="new_event",
                                        ),
                                        path(
                                            "auto-translate/",
                                            bulk_action_views.BulkAutoTranslateView.as_view(
                                                model=Event, form=EventTranslationForm
                                            ),
                                            name="automatic_translation_events",
                                        ),
                                        path(
                                            "bulk-archive/",
                                            bulk_action_views.BulkArchiveView.as_view(
                                                model=Event
                                            ),
                                            name="bulk_archive_events",
                                        ),
                                        path(
                                            "bulk-restore/",
                                            bulk_action_views.BulkRestoreView.as_view(
                                                model=Event
                                            ),
                                            name="bulk_restore_events",
                                        ),
                                        path(
                                            "<int:event_id>/",
                                            include(
                                                [
                                                    path(
                                                        "edit/",
                                                        events.EventFormView.as_view(),
                                                        name="edit_event",
                                                    ),
                                                    path(
                                                        "copy/",
                                                        events.copy,
                                                        name="copy_event",
                                                    ),
                                                    path(
                                                        "archive/",
                                                        events.archive,
                                                        name="archive_event",
                                                    ),
                                                    path(
                                                        "restore/",
                                                        events.restore,
                                                        name="restore_event",
                                                    ),
                                                    path(
                                                        "delete/",
                                                        events.delete,
                                                        name="delete_event",
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "pois/",
                    include(
                        [
                            path("", pois.POIListView.as_view(), name="pois"),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "",
                                            pois.POIListView.as_view(),
                                            name="pois",
                                        ),
                                        path(
                                            "archived/",
                                            pois.POIListView.as_view(archived=True),
                                            name="archived_pois",
                                        ),
                                        path(
                                            "new/",
                                            pois.POIFormView.as_view(),
                                            name="new_poi",
                                        ),
                                        path(
                                            "auto-translate/",
                                            bulk_action_views.BulkAutoTranslateView.as_view(
                                                model=POI, form=POITranslationForm
                                            ),
                                            name="automatic_translation_pois",
                                        ),
                                        path(
                                            "bulk-archive/",
                                            bulk_action_views.BulkArchiveView.as_view(
                                                model=POI
                                            ),
                                            name="bulk_archive_pois",
                                        ),
                                        path(
                                            "bulk-restore/",
                                            bulk_action_views.BulkRestoreView.as_view(
                                                model=POI
                                            ),
                                            name="bulk_restore_pois",
                                        ),
                                        path(
                                            "<int:poi_id>/",
                                            include(
                                                [
                                                    path(
                                                        "view/",
                                                        pois.view_poi,
                                                        name="view_poi",
                                                    ),
                                                    path(
                                                        "edit/",
                                                        pois.POIFormView.as_view(),
                                                        name="edit_poi",
                                                    ),
                                                    path(
                                                        "archive/",
                                                        pois.archive_poi,
                                                        name="archive_poi",
                                                    ),
                                                    path(
                                                        "restore/",
                                                        pois.restore_poi,
                                                        name="restore_poi",
                                                    ),
                                                    path(
                                                        "delete/",
                                                        pois.delete_poi,
                                                        name="delete_poi",
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "feedback/",
                    include(
                        [
                            path(
                                "",
                                feedback.RegionFeedbackListView.as_view(),
                                name="region_feedback",
                            ),
                            path(
                                "mark-read/",
                                feedback.mark_region_feedback_as_read,
                                name="mark_region_feedback_as_read",
                            ),
                            path(
                                "mark-unread/",
                                feedback.mark_region_feedback_as_unread,
                                name="mark_region_feedback_as_unread",
                            ),
                            path(
                                "delete/",
                                feedback.delete_region_feedback,
                                name="delete_region_feedback",
                            ),
                        ]
                    ),
                ),
                path(
                    "push-notifications/",
                    include(
                        [
                            path(
                                "",
                                push_notifications.PushNotificationListView.as_view(),
                                name="push_notifications",
                            ),
                            path(
                                "<slug:language_slug>/",
                                include(
                                    [
                                        path(
                                            "",
                                            push_notifications.PushNotificationListView.as_view(),
                                            name="push_notifications",
                                        ),
                                        path(
                                            "new/",
                                            push_notifications.PushNotificationFormView.as_view(),
                                            name="new_push_notification",
                                        ),
                                        path(
                                            "<int:push_notification_id>/",
                                            include(
                                                [
                                                    path(
                                                        "edit/",
                                                        push_notifications.PushNotificationFormView.as_view(),
                                                        name="edit_push_notification",
                                                    ),
                                                ]
                                            ),
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path(
                    "language-tree/",
                    include(
                        [
                            path(
                                "",
                                language_tree.LanguageTreeView.as_view(),
                                name="language_tree",
                            ),
                            path(
                                "new/",
                                language_tree.LanguageTreeNodeFormView.as_view(),
                                name="new_language_tree_node",
                            ),
                            path(
                                "<int:language_tree_node_id>/",
                                include(
                                    [
                                        path(
                                            "edit/",
                                            language_tree.LanguageTreeNodeFormView.as_view(),
                                            name="edit_language_tree_node",
                                        ),
                                        path(
                                            "delete/",
                                            language_tree.delete_language_tree_node,
                                            name="delete_language_tree_node",
                                        ),
                                        # warning: the move url is also hardcoded in src/integreat_cms/static/js/tree_drag_and_drop.js
                                        path(
                                            "move/<int:target_id>/<slug:target_position>/",
                                            language_tree.move_language_tree_node,
                                            name="move_language_tree_node",
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path("media-library/", media.MediaListView.as_view(), name="media"),
                path(
                    "users/",
                    include(
                        [
                            path(
                                "",
                                users.RegionUserListView.as_view(),
                                name="region_users",
                            ),
                            path(
                                "new/",
                                users.RegionUserFormView.as_view(),
                                name="new_region_user",
                            ),
                            path(
                                "<int:user_id>/",
                                include(
                                    [
                                        path(
                                            "edit/",
                                            users.RegionUserFormView.as_view(),
                                            name="edit_region_user",
                                        ),
                                        path(
                                            "delete/",
                                            users.delete_region_user,
                                            name="delete_region_user",
                                        ),
                                        path(
                                            "resend-activation-link/",
                                            users.resend_activation_link_region,
                                            name="resend_activation_link_region",
                                        ),
                                    ]
                                ),
                            ),
                        ]
                    ),
                ),
                path("", include(user_settings_urlpatterns)),
            ]
        ),
    ),
]
