"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * Invoke a Lambda function as a Task
 *
 * OUTPUT: the output of this task is either the return value of Lambda's
 * Invoke call, or whatever the Lambda Function posted back using
 * `SendTaskSuccess/SendTaskFailure` in `waitForTaskToken` mode.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 * @deprecated Use `LambdaInvoke`
 */
class RunLambdaTask {
    constructor(lambdaFunction, props = {}) {
        this.lambdaFunction = lambdaFunction;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Lambda.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is missing in payload (pass Context.taskToken somewhere in payload)');
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('lambda', 'invoke', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    resources: [this.lambdaFunction.functionArn],
                    actions: ['lambda:InvokeFunction'],
                })],
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: { LambdaFunctionArn: this.lambdaFunction.functionArn },
            parameters: {
                FunctionName: this.lambdaFunction.functionName,
                Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                InvocationType: this.props.invocationType,
                ClientContext: this.props.clientContext,
                Qualifier: this.props.qualifier,
            },
        };
    }
}
exports.RunLambdaTask = RunLambdaTask;
/**
 * Invocation type of a Lambda
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke synchronously
     *
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke asynchronously
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * TValidate parameter values and verify that the user or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType = exports.InvocationType || (exports.InvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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