"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
/**
 * Verifies that a validation pattern is supported for a service integration
 *
 */
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE]: '',
    [aws_stepfunctions_1.IntegrationPattern.RUN_JOB]: '.sync',
    [aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${core_1.Aws.PARTITION}:states:::${service}:${api}` +
        (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
//# sourceMappingURL=data:application/json;base64,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