"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('create basic training job', () => {
    // WHEN
    const task = new sfn.Task(stack, 'TrainSagemaker', { task: new tasks.SagemakerTrainTask({
            trainingJobName: 'MyTrainJob',
            algorithmSpecification: {
                algorithmName: 'BlazingText',
            },
            inputDataConfig: [
                {
                    channelName: 'train',
                    dataSource: {
                        s3DataSource: {
                            s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucket', 'mybucket'), 'mytrainpath'),
                        },
                    },
                },
            ],
            outputDataConfig: {
                s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath'),
            },
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            AlgorithmSpecification: {
                AlgorithmName: 'BlazingText',
                TrainingInputMode: 'File',
            },
            InputDataConfig: [
                {
                    ChannelName: 'train',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytrainpath']],
                            },
                        },
                    },
                },
            ],
            OutputDataConfig: {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']],
                },
            },
            ResourceConfig: {
                InstanceCount: 1,
                InstanceType: 'ml.m4.xlarge',
                VolumeSizeInGB: 10,
            },
            RoleArn: { 'Fn::GetAtt': ['TrainSagemakerSagemakerRole89E8C593', 'Arn'] },
            StoppingCondition: {
                MaxRuntimeInSeconds: 3600,
            },
            TrainingJobName: 'MyTrainJob',
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'TrainSagemaker', { task: new tasks.SagemakerTrainTask({
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
                trainingJobName: 'MyTrainJob',
                algorithmSpecification: {
                    algorithmName: 'BlazingText',
                },
                inputDataConfig: [
                    {
                        channelName: 'train',
                        dataSource: {
                            s3DataSource: {
                                s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucket', 'mybucket'), 'mytrainpath'),
                            },
                        },
                    },
                ],
                outputDataConfig: {
                    s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath'),
                },
            }) });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call SageMaker./i);
});
test('create complex training job', () => {
    // WHEN
    const kmsKey = new kms.Key(stack, 'Key');
    const vpc = new ec2.Vpc(stack, 'VPC');
    const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc, description: 'My SG' });
    securityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'allow ssh access from the world');
    const role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
        ],
    });
    const trainTask = new tasks.SagemakerTrainTask({
        trainingJobName: 'MyTrainJob',
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        role,
        algorithmSpecification: {
            algorithmName: 'BlazingText',
            trainingInputMode: tasks.InputMode.FILE,
            metricDefinitions: [
                {
                    name: 'mymetric', regex: 'regex_pattern',
                },
            ],
        },
        hyperparameters: {
            lr: '0.1',
        },
        inputDataConfig: [
            {
                channelName: 'train',
                contentType: 'image/jpeg',
                compressionType: tasks.CompressionType.NONE,
                recordWrapperType: tasks.RecordWrapperType.RECORD_IO,
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucketA', 'mybucket'), 'mytrainpath'),
                    },
                },
            },
            {
                channelName: 'test',
                contentType: 'image/jpeg',
                compressionType: tasks.CompressionType.GZIP,
                recordWrapperType: tasks.RecordWrapperType.RECORD_IO,
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucketB', 'mybucket'), 'mytestpath'),
                    },
                },
            },
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath'),
            encryptionKey: kmsKey,
        },
        resourceConfig: {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
            volumeSizeInGB: 50,
            volumeEncryptionKey: kmsKey,
        },
        stoppingCondition: {
            maxRuntime: cdk.Duration.hours(1),
        },
        tags: {
            Project: 'MyProject',
        },
        vpcConfig: {
            vpc,
        },
    });
    trainTask.addSecurityGroup(securityGroup);
    const task = new sfn.Task(stack, 'TrainSagemaker', { task: trainTask });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            TrainingJobName: 'MyTrainJob',
            RoleArn: { 'Fn::GetAtt': ['Role1ABCC5F0', 'Arn'] },
            AlgorithmSpecification: {
                TrainingInputMode: 'File',
                AlgorithmName: 'BlazingText',
                MetricDefinitions: [
                    { Name: 'mymetric', Regex: 'regex_pattern' },
                ],
            },
            HyperParameters: {
                lr: '0.1',
            },
            InputDataConfig: [
                {
                    ChannelName: 'train',
                    CompressionType: 'None',
                    RecordWrapperType: 'RecordIO',
                    ContentType: 'image/jpeg',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytrainpath']],
                            },
                        },
                    },
                },
                {
                    ChannelName: 'test',
                    CompressionType: 'Gzip',
                    RecordWrapperType: 'RecordIO',
                    ContentType: 'image/jpeg',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytestpath']],
                            },
                        },
                    },
                },
            ],
            OutputDataConfig: {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']],
                },
                KmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            ResourceConfig: {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeSizeInGB: 50,
                VolumeKmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            StoppingCondition: {
                MaxRuntimeInSeconds: 3600,
            },
            Tags: [
                { Key: 'Project', Value: 'MyProject' },
            ],
            VpcConfig: {
                SecurityGroupIds: [
                    { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
                    { 'Fn::GetAtt': ['TrainSagemakerTrainJobSecurityGroup7C858EB9', 'GroupId'] },
                ],
                Subnets: [
                    { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                    { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                ],
            },
        },
    });
});
test('pass param to training job', () => {
    // WHEN
    const role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
        ],
    });
    const task = new sfn.Task(stack, 'TrainSagemaker', { task: new tasks.SagemakerTrainTask({
            trainingJobName: sfn.Data.stringAt('$.JobName'),
            role,
            algorithmSpecification: {
                algorithmName: 'BlazingText',
                trainingInputMode: tasks.InputMode.FILE,
            },
            inputDataConfig: [
                {
                    channelName: 'train',
                    dataSource: {
                        s3DataSource: {
                            s3DataType: tasks.S3DataType.S3_PREFIX,
                            s3Location: tasks.S3Location.fromJsonExpression('$.S3Bucket'),
                        },
                    },
                },
            ],
            outputDataConfig: {
                s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'Bucket', 'mybucket'), 'myoutputpath'),
            },
            resourceConfig: {
                instanceCount: 1,
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
                volumeSizeInGB: 50,
            },
            stoppingCondition: {
                maxRuntime: cdk.Duration.hours(1),
            },
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            'TrainingJobName.$': '$.JobName',
            'RoleArn': { 'Fn::GetAtt': ['Role1ABCC5F0', 'Arn'] },
            'AlgorithmSpecification': {
                TrainingInputMode: 'File',
                AlgorithmName: 'BlazingText',
            },
            'InputDataConfig': [
                {
                    ChannelName: 'train',
                    DataSource: {
                        S3DataSource: {
                            'S3DataType': 'S3Prefix',
                            'S3Uri.$': '$.S3Bucket',
                        },
                    },
                },
            ],
            'OutputDataConfig': {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']],
                },
            },
            'ResourceConfig': {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeSizeInGB: 50,
            },
            'StoppingCondition': {
                MaxRuntimeInSeconds: 3600,
            },
        },
    });
});
test('Cannot create a SageMaker train task with both algorithm name and image name missing', () => {
    expect(() => new tasks.SagemakerTrainTask({
        trainingJobName: 'myTrainJob',
        algorithmSpecification: {},
        inputDataConfig: [
            {
                channelName: 'train',
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromJsonExpression('$.S3Bucket'),
                    },
                },
            },
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'Bucket', 'mybucket'), 'myoutputpath/'),
        },
    }))
        .toThrowError(/Must define either an algorithm name or training image URI in the algorithm specification/);
});
//# sourceMappingURL=data:application/json;base64,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