"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
let role;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
        ],
    });
});
test('create basic transform job', () => {
    // WHEN
    const task = new sfn.Task(stack, 'TransformTask', { task: new tasks.SagemakerTransformTask({
            transformJobName: 'MyTransformJob',
            modelName: 'MyModelName',
            transformInput: {
                transformDataSource: {
                    s3DataSource: {
                        s3Uri: 's3://inputbucket/prefix',
                    },
                },
            },
            transformOutput: {
                s3OutputPath: 's3://outputbucket/prefix',
            },
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTransformJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            TransformJobName: 'MyTransformJob',
            ModelName: 'MyModelName',
            TransformInput: {
                DataSource: {
                    S3DataSource: {
                        S3Uri: 's3://inputbucket/prefix',
                        S3DataType: 'S3Prefix',
                    },
                },
            },
            TransformOutput: {
                S3OutputPath: 's3://outputbucket/prefix',
            },
            TransformResources: {
                InstanceCount: 1,
                InstanceType: 'ml.m4.xlarge',
            },
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'TransformTask', { task: new tasks.SagemakerTransformTask({
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
                transformJobName: 'MyTransformJob',
                modelName: 'MyModelName',
                transformInput: {
                    transformDataSource: {
                        s3DataSource: {
                            s3Uri: 's3://inputbucket/prefix',
                        },
                    },
                },
                transformOutput: {
                    s3OutputPath: 's3://outputbucket/prefix',
                },
            }) });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call SageMaker./i);
});
test('create complex transform job', () => {
    // WHEN
    const kmsKey = new kms.Key(stack, 'Key');
    const task = new sfn.Task(stack, 'TransformTask', { task: new tasks.SagemakerTransformTask({
            transformJobName: 'MyTransformJob',
            modelName: 'MyModelName',
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
            role,
            transformInput: {
                transformDataSource: {
                    s3DataSource: {
                        s3Uri: 's3://inputbucket/prefix',
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                    },
                },
            },
            transformOutput: {
                s3OutputPath: 's3://outputbucket/prefix',
                encryptionKey: kmsKey,
            },
            transformResources: {
                instanceCount: 1,
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
                volumeKmsKeyId: kmsKey,
            },
            tags: {
                Project: 'MyProject',
            },
            batchStrategy: tasks.BatchStrategy.MULTI_RECORD,
            environment: {
                SOMEVAR: 'myvalue',
            },
            maxConcurrentTransforms: 3,
            maxPayloadInMB: 100,
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTransformJob.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            TransformJobName: 'MyTransformJob',
            ModelName: 'MyModelName',
            TransformInput: {
                DataSource: {
                    S3DataSource: {
                        S3Uri: 's3://inputbucket/prefix',
                        S3DataType: 'S3Prefix',
                    },
                },
            },
            TransformOutput: {
                S3OutputPath: 's3://outputbucket/prefix',
                KmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            TransformResources: {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeKmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            Tags: [
                { Key: 'Project', Value: 'MyProject' },
            ],
            MaxConcurrentTransforms: 3,
            MaxPayloadInMB: 100,
            Environment: {
                SOMEVAR: 'myvalue',
            },
            BatchStrategy: 'MultiRecord',
        },
    });
});
test('pass param to transform job', () => {
    // WHEN
    const task = new sfn.Task(stack, 'TransformTask', { task: new tasks.SagemakerTransformTask({
            transformJobName: sfn.Data.stringAt('$.TransformJobName'),
            modelName: sfn.Data.stringAt('$.ModelName'),
            role,
            transformInput: {
                transformDataSource: {
                    s3DataSource: {
                        s3Uri: 's3://inputbucket/prefix',
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                    },
                },
            },
            transformOutput: {
                s3OutputPath: 's3://outputbucket/prefix',
            },
            transformResources: {
                instanceCount: 1,
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
            },
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTransformJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            'TransformJobName.$': '$.TransformJobName',
            'ModelName.$': '$.ModelName',
            'TransformInput': {
                DataSource: {
                    S3DataSource: {
                        S3Uri: 's3://inputbucket/prefix',
                        S3DataType: 'S3Prefix',
                    },
                },
            },
            'TransformOutput': {
                S3OutputPath: 's3://outputbucket/prefix',
            },
            'TransformResources': {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
            },
        },
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2FnZW1ha2VyLXRyYW5zZm9ybS1qb2IudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNhZ2VtYWtlci10cmFuc2Zvcm0tam9iLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxnQ0FBOEI7QUFDOUIsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsa0RBQWtEO0FBQ2xELHFDQUFxQztBQUNyQyxtQ0FBbUM7QUFFbkMsSUFBSSxLQUFnQixDQUFDO0FBQ3JCLElBQUksSUFBYyxDQUFDO0FBRW5CLFVBQVUsQ0FBQyxHQUFHLEVBQUU7SUFDZCxRQUFRO0lBQ1IsS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQ3hCLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUNqQyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUM7UUFDOUQsZUFBZSxFQUFFO1lBQ2YsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQywyQkFBMkIsQ0FBQztTQUN4RTtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDRCQUE0QixFQUFFLEdBQUcsRUFBRTtJQUN0QyxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUM7WUFDekYsZ0JBQWdCLEVBQUUsZ0JBQWdCO1lBQ2xDLFNBQVMsRUFBRSxhQUFhO1lBQ3hCLGNBQWMsRUFBRTtnQkFDZCxtQkFBbUIsRUFBRTtvQkFDbkIsWUFBWSxFQUFFO3dCQUNaLEtBQUssRUFBRSx5QkFBeUI7cUJBQ2pDO2lCQUNGO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsWUFBWSxFQUFFLDBCQUEwQjthQUN6QztTQUNGLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFTixPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELHdDQUF3QztpQkFDekM7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixnQkFBZ0IsRUFBRSxnQkFBZ0I7WUFDbEMsU0FBUyxFQUFFLGFBQWE7WUFDeEIsY0FBYyxFQUFFO2dCQUNkLFVBQVUsRUFBRTtvQkFDVixZQUFZLEVBQUU7d0JBQ1osS0FBSyxFQUFFLHlCQUF5Qjt3QkFDaEMsVUFBVSxFQUFFLFVBQVU7cUJBQ3ZCO2lCQUNGO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsWUFBWSxFQUFFLDBCQUEwQjthQUN6QztZQUNELGtCQUFrQixFQUFFO2dCQUNsQixhQUFhLEVBQUUsQ0FBQztnQkFDaEIsWUFBWSxFQUFFLGNBQWM7YUFDN0I7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLCtFQUErRSxFQUFFLEdBQUcsRUFBRTtJQUN6RixNQUFNLENBQUMsR0FBRyxFQUFFO1FBQ1YsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUM7Z0JBQzVFLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxtQkFBbUI7Z0JBQ3JFLGdCQUFnQixFQUFFLGdCQUFnQjtnQkFDbEMsU0FBUyxFQUFFLGFBQWE7Z0JBQ3hCLGNBQWMsRUFBRTtvQkFDZCxtQkFBbUIsRUFBRTt3QkFDbkIsWUFBWSxFQUFFOzRCQUNaLEtBQUssRUFBRSx5QkFBeUI7eUJBQ2pDO3FCQUNGO2lCQUNGO2dCQUNELGVBQWUsRUFBRTtvQkFDZixZQUFZLEVBQUUsMEJBQTBCO2lCQUN6QzthQUNGLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDUixDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsK0ZBQStGLENBQUMsQ0FBQztBQUM5RyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw4QkFBOEIsRUFBRSxHQUFHLEVBQUU7SUFDeEMsT0FBTztJQUNQLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDekMsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUM7WUFDekYsZ0JBQWdCLEVBQUUsZ0JBQWdCO1lBQ2xDLFNBQVMsRUFBRSxhQUFhO1lBQ3hCLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJO1lBQ3RELElBQUk7WUFDSixjQUFjLEVBQUU7Z0JBQ2QsbUJBQW1CLEVBQUU7b0JBQ25CLFlBQVksRUFBRTt3QkFDWixLQUFLLEVBQUUseUJBQXlCO3dCQUNoQyxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxTQUFTO3FCQUN2QztpQkFDRjthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLFlBQVksRUFBRSwwQkFBMEI7Z0JBQ3hDLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLGFBQWEsRUFBRSxDQUFDO2dCQUNoQixZQUFZLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUM7Z0JBQ2pGLGNBQWMsRUFBRSxNQUFNO2FBQ3ZCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLE9BQU8sRUFBRSxXQUFXO2FBQ3JCO1lBQ0QsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsWUFBWTtZQUMvQyxXQUFXLEVBQUU7Z0JBQ1gsT0FBTyxFQUFFLFNBQVM7YUFDbkI7WUFDRCx1QkFBdUIsRUFBRSxDQUFDO1lBQzFCLGNBQWMsRUFBRSxHQUFHO1NBQ3BCLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFTixPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELDZDQUE2QztpQkFDOUM7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixnQkFBZ0IsRUFBRSxnQkFBZ0I7WUFDbEMsU0FBUyxFQUFFLGFBQWE7WUFDeEIsY0FBYyxFQUFFO2dCQUNkLFVBQVUsRUFBRTtvQkFDVixZQUFZLEVBQUU7d0JBQ1osS0FBSyxFQUFFLHlCQUF5Qjt3QkFDaEMsVUFBVSxFQUFFLFVBQVU7cUJBQ3ZCO2lCQUNGO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsWUFBWSxFQUFFLDBCQUEwQjtnQkFDeEMsUUFBUSxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsYUFBYSxFQUFFLEtBQUssQ0FBRSxFQUFFO2FBQ3JEO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLGFBQWEsRUFBRSxDQUFDO2dCQUNoQixZQUFZLEVBQUUsZUFBZTtnQkFDN0IsY0FBYyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsYUFBYSxFQUFFLEtBQUssQ0FBRSxFQUFFO2FBQzNEO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLEVBQUUsR0FBRyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsV0FBVyxFQUFFO2FBQ3ZDO1lBQ0QsdUJBQXVCLEVBQUUsQ0FBQztZQUMxQixjQUFjLEVBQUUsR0FBRztZQUNuQixXQUFXLEVBQUU7Z0JBQ1gsT0FBTyxFQUFFLFNBQVM7YUFDbkI7WUFDRCxhQUFhLEVBQUUsYUFBYTtTQUM3QjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDZCQUE2QixFQUFFLEdBQUcsRUFBRTtJQUN2QyxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUM7WUFDekYsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsb0JBQW9CLENBQUM7WUFDekQsU0FBUyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQztZQUMzQyxJQUFJO1lBQ0osY0FBYyxFQUFFO2dCQUNkLG1CQUFtQixFQUFFO29CQUNuQixZQUFZLEVBQUU7d0JBQ1osS0FBSyxFQUFFLHlCQUF5Qjt3QkFDaEMsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsU0FBUztxQkFDdkM7aUJBQ0Y7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixZQUFZLEVBQUUsMEJBQTBCO2FBQ3pDO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLGFBQWEsRUFBRSxDQUFDO2dCQUNoQixZQUFZLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUM7YUFDbEY7U0FDRixDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBRU4sT0FBTztJQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ2hELElBQUksRUFBRSxNQUFNO1FBQ1osUUFBUSxFQUFFO1lBQ1IsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTjt3QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3FCQUN0QjtvQkFDRCx3Q0FBd0M7aUJBQ3pDO2FBQ0Y7U0FDRjtRQUNELEdBQUcsRUFBRSxJQUFJO1FBQ1QsVUFBVSxFQUFFO1lBQ1Ysb0JBQW9CLEVBQUUsb0JBQW9CO1lBQzFDLGFBQWEsRUFBRSxhQUFhO1lBQzVCLGdCQUFnQixFQUFFO2dCQUNoQixVQUFVLEVBQUU7b0JBQ1YsWUFBWSxFQUFFO3dCQUNaLEtBQUssRUFBRSx5QkFBeUI7d0JBQ2hDLFVBQVUsRUFBRSxVQUFVO3FCQUN2QjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLFlBQVksRUFBRSwwQkFBMEI7YUFDekM7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsYUFBYSxFQUFFLENBQUM7Z0JBQ2hCLFlBQVksRUFBRSxlQUFlO2FBQzlCO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMga21zIGZyb20gJ0Bhd3MtY2RrL2F3cy1rbXMnO1xuaW1wb3J0ICogYXMgc2ZuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIHRhc2tzIGZyb20gJy4uLy4uL2xpYic7XG5cbmxldCBzdGFjazogY2RrLlN0YWNrO1xubGV0IHJvbGU6IGlhbS5Sb2xlO1xuXG5iZWZvcmVFYWNoKCgpID0+IHtcbiAgLy8gR0lWRU5cbiAgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBbWF6b25TYWdlTWFrZXJGdWxsQWNjZXNzJyksXG4gICAgXSxcbiAgfSk7XG59KTtcblxudGVzdCgnY3JlYXRlIGJhc2ljIHRyYW5zZm9ybSBqb2InLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3QgdGFzayA9IG5ldyBzZm4uVGFzayhzdGFjaywgJ1RyYW5zZm9ybVRhc2snLCB7IHRhc2s6IG5ldyB0YXNrcy5TYWdlbWFrZXJUcmFuc2Zvcm1UYXNrKHtcbiAgICB0cmFuc2Zvcm1Kb2JOYW1lOiAnTXlUcmFuc2Zvcm1Kb2InLFxuICAgIG1vZGVsTmFtZTogJ015TW9kZWxOYW1lJyxcbiAgICB0cmFuc2Zvcm1JbnB1dDoge1xuICAgICAgdHJhbnNmb3JtRGF0YVNvdXJjZToge1xuICAgICAgICBzM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICBzM1VyaTogJ3MzOi8vaW5wdXRidWNrZXQvcHJlZml4JyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICB0cmFuc2Zvcm1PdXRwdXQ6IHtcbiAgICAgIHMzT3V0cHV0UGF0aDogJ3MzOi8vb3V0cHV0YnVja2V0L3ByZWZpeCcsXG4gICAgfSxcbiAgfSkgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjpzYWdlbWFrZXI6Y3JlYXRlVHJhbnNmb3JtSm9iJyxcbiAgICAgICAgXSxcbiAgICAgIF0sXG4gICAgfSxcbiAgICBFbmQ6IHRydWUsXG4gICAgUGFyYW1ldGVyczoge1xuICAgICAgVHJhbnNmb3JtSm9iTmFtZTogJ015VHJhbnNmb3JtSm9iJyxcbiAgICAgIE1vZGVsTmFtZTogJ015TW9kZWxOYW1lJyxcbiAgICAgIFRyYW5zZm9ybUlucHV0OiB7XG4gICAgICAgIERhdGFTb3VyY2U6IHtcbiAgICAgICAgICBTM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgIFMzVXJpOiAnczM6Ly9pbnB1dGJ1Y2tldC9wcmVmaXgnLFxuICAgICAgICAgICAgUzNEYXRhVHlwZTogJ1MzUHJlZml4JyxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIFRyYW5zZm9ybU91dHB1dDoge1xuICAgICAgICBTM091dHB1dFBhdGg6ICdzMzovL291dHB1dGJ1Y2tldC9wcmVmaXgnLFxuICAgICAgfSxcbiAgICAgIFRyYW5zZm9ybVJlc291cmNlczoge1xuICAgICAgICBJbnN0YW5jZUNvdW50OiAxLFxuICAgICAgICBJbnN0YW5jZVR5cGU6ICdtbC5tNC54bGFyZ2UnLFxuICAgICAgfSxcbiAgICB9LFxuICB9KTtcbn0pO1xuXG50ZXN0KCdUYXNrIHRocm93cyBpZiBXQUlUX0ZPUl9UQVNLX1RPS0VOIGlzIHN1cHBsaWVkIGFzIHNlcnZpY2UgaW50ZWdyYXRpb24gcGF0dGVybicsICgpID0+IHtcbiAgZXhwZWN0KCgpID0+IHtcbiAgICBuZXcgc2ZuLlRhc2soc3RhY2ssICdUcmFuc2Zvcm1UYXNrJywgeyB0YXNrOiBuZXcgdGFza3MuU2FnZW1ha2VyVHJhbnNmb3JtVGFzayh7XG4gICAgICBpbnRlZ3JhdGlvblBhdHRlcm46IHNmbi5TZXJ2aWNlSW50ZWdyYXRpb25QYXR0ZXJuLldBSVRfRk9SX1RBU0tfVE9LRU4sXG4gICAgICB0cmFuc2Zvcm1Kb2JOYW1lOiAnTXlUcmFuc2Zvcm1Kb2InLFxuICAgICAgbW9kZWxOYW1lOiAnTXlNb2RlbE5hbWUnLFxuICAgICAgdHJhbnNmb3JtSW5wdXQ6IHtcbiAgICAgICAgdHJhbnNmb3JtRGF0YVNvdXJjZToge1xuICAgICAgICAgIHMzRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgczNVcmk6ICdzMzovL2lucHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICB0cmFuc2Zvcm1PdXRwdXQ6IHtcbiAgICAgICAgczNPdXRwdXRQYXRoOiAnczM6Ly9vdXRwdXRidWNrZXQvcHJlZml4JyxcbiAgICAgIH0sXG4gICAgfSkgfSk7XG4gIH0pLnRvVGhyb3coL0ludmFsaWQgU2VydmljZSBJbnRlZ3JhdGlvbiBQYXR0ZXJuOiBXQUlUX0ZPUl9UQVNLX1RPS0VOIGlzIG5vdCBzdXBwb3J0ZWQgdG8gY2FsbCBTYWdlTWFrZXIuL2kpO1xufSk7XG5cbnRlc3QoJ2NyZWF0ZSBjb21wbGV4IHRyYW5zZm9ybSBqb2InLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3Qga21zS2V5ID0gbmV3IGttcy5LZXkoc3RhY2ssICdLZXknKTtcbiAgY29uc3QgdGFzayA9IG5ldyBzZm4uVGFzayhzdGFjaywgJ1RyYW5zZm9ybVRhc2snLCB7IHRhc2s6IG5ldyB0YXNrcy5TYWdlbWFrZXJUcmFuc2Zvcm1UYXNrKHtcbiAgICB0cmFuc2Zvcm1Kb2JOYW1lOiAnTXlUcmFuc2Zvcm1Kb2InLFxuICAgIG1vZGVsTmFtZTogJ015TW9kZWxOYW1lJyxcbiAgICBpbnRlZ3JhdGlvblBhdHRlcm46IHNmbi5TZXJ2aWNlSW50ZWdyYXRpb25QYXR0ZXJuLlNZTkMsXG4gICAgcm9sZSxcbiAgICB0cmFuc2Zvcm1JbnB1dDoge1xuICAgICAgdHJhbnNmb3JtRGF0YVNvdXJjZToge1xuICAgICAgICBzM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICBzM1VyaTogJ3MzOi8vaW5wdXRidWNrZXQvcHJlZml4JyxcbiAgICAgICAgICBzM0RhdGFUeXBlOiB0YXNrcy5TM0RhdGFUeXBlLlMzX1BSRUZJWCxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICB0cmFuc2Zvcm1PdXRwdXQ6IHtcbiAgICAgIHMzT3V0cHV0UGF0aDogJ3MzOi8vb3V0cHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICBlbmNyeXB0aW9uS2V5OiBrbXNLZXksXG4gICAgfSxcbiAgICB0cmFuc2Zvcm1SZXNvdXJjZXM6IHtcbiAgICAgIGluc3RhbmNlQ291bnQ6IDEsXG4gICAgICBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGUub2YoZWMyLkluc3RhbmNlQ2xhc3MuUDMsIGVjMi5JbnN0YW5jZVNpemUuWExBUkdFMiksXG4gICAgICB2b2x1bWVLbXNLZXlJZDoga21zS2V5LFxuICAgIH0sXG4gICAgdGFnczoge1xuICAgICAgUHJvamVjdDogJ015UHJvamVjdCcsXG4gICAgfSxcbiAgICBiYXRjaFN0cmF0ZWd5OiB0YXNrcy5CYXRjaFN0cmF0ZWd5Lk1VTFRJX1JFQ09SRCxcbiAgICBlbnZpcm9ubWVudDoge1xuICAgICAgU09NRVZBUjogJ215dmFsdWUnLFxuICAgIH0sXG4gICAgbWF4Q29uY3VycmVudFRyYW5zZm9ybXM6IDMsXG4gICAgbWF4UGF5bG9hZEluTUI6IDEwMCxcbiAgfSkgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjpzYWdlbWFrZXI6Y3JlYXRlVHJhbnNmb3JtSm9iLnN5bmMnLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICBUcmFuc2Zvcm1Kb2JOYW1lOiAnTXlUcmFuc2Zvcm1Kb2InLFxuICAgICAgTW9kZWxOYW1lOiAnTXlNb2RlbE5hbWUnLFxuICAgICAgVHJhbnNmb3JtSW5wdXQ6IHtcbiAgICAgICAgRGF0YVNvdXJjZToge1xuICAgICAgICAgIFMzRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgUzNVcmk6ICdzMzovL2lucHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICAgICAgICBTM0RhdGFUeXBlOiAnUzNQcmVmaXgnLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgVHJhbnNmb3JtT3V0cHV0OiB7XG4gICAgICAgIFMzT3V0cHV0UGF0aDogJ3MzOi8vb3V0cHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICAgIEttc0tleUlkOiB7ICdGbjo6R2V0QXR0JzogWyAnS2V5OTYxQjczRkQnLCAnQXJuJyBdIH0sXG4gICAgICB9LFxuICAgICAgVHJhbnNmb3JtUmVzb3VyY2VzOiB7XG4gICAgICAgIEluc3RhbmNlQ291bnQ6IDEsXG4gICAgICAgIEluc3RhbmNlVHlwZTogJ21sLnAzLjJ4bGFyZ2UnLFxuICAgICAgICBWb2x1bWVLbXNLZXlJZDogeyAnRm46OkdldEF0dCc6IFsgJ0tleTk2MUI3M0ZEJywgJ0FybicgXSB9LFxuICAgICAgfSxcbiAgICAgIFRhZ3M6IFtcbiAgICAgICAgeyBLZXk6ICdQcm9qZWN0JywgVmFsdWU6ICdNeVByb2plY3QnIH0sXG4gICAgICBdLFxuICAgICAgTWF4Q29uY3VycmVudFRyYW5zZm9ybXM6IDMsXG4gICAgICBNYXhQYXlsb2FkSW5NQjogMTAwLFxuICAgICAgRW52aXJvbm1lbnQ6IHtcbiAgICAgICAgU09NRVZBUjogJ215dmFsdWUnLFxuICAgICAgfSxcbiAgICAgIEJhdGNoU3RyYXRlZ3k6ICdNdWx0aVJlY29yZCcsXG4gICAgfSxcbiAgfSk7XG59KTtcblxudGVzdCgncGFzcyBwYXJhbSB0byB0cmFuc2Zvcm0gam9iJywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIGNvbnN0IHRhc2sgPSBuZXcgc2ZuLlRhc2soc3RhY2ssICdUcmFuc2Zvcm1UYXNrJywgeyB0YXNrOiBuZXcgdGFza3MuU2FnZW1ha2VyVHJhbnNmb3JtVGFzayh7XG4gICAgdHJhbnNmb3JtSm9iTmFtZTogc2ZuLkRhdGEuc3RyaW5nQXQoJyQuVHJhbnNmb3JtSm9iTmFtZScpLFxuICAgIG1vZGVsTmFtZTogc2ZuLkRhdGEuc3RyaW5nQXQoJyQuTW9kZWxOYW1lJyksXG4gICAgcm9sZSxcbiAgICB0cmFuc2Zvcm1JbnB1dDoge1xuICAgICAgdHJhbnNmb3JtRGF0YVNvdXJjZToge1xuICAgICAgICBzM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICBzM1VyaTogJ3MzOi8vaW5wdXRidWNrZXQvcHJlZml4JyxcbiAgICAgICAgICBzM0RhdGFUeXBlOiB0YXNrcy5TM0RhdGFUeXBlLlMzX1BSRUZJWCxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICB0cmFuc2Zvcm1PdXRwdXQ6IHtcbiAgICAgIHMzT3V0cHV0UGF0aDogJ3MzOi8vb3V0cHV0YnVja2V0L3ByZWZpeCcsXG4gICAgfSxcbiAgICB0cmFuc2Zvcm1SZXNvdXJjZXM6IHtcbiAgICAgIGluc3RhbmNlQ291bnQ6IDEsXG4gICAgICBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGUub2YoZWMyLkluc3RhbmNlQ2xhc3MuUDMsIGVjMi5JbnN0YW5jZVNpemUuWExBUkdFMiksXG4gICAgfSxcbiAgfSkgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjpzYWdlbWFrZXI6Y3JlYXRlVHJhbnNmb3JtSm9iJyxcbiAgICAgICAgXSxcbiAgICAgIF0sXG4gICAgfSxcbiAgICBFbmQ6IHRydWUsXG4gICAgUGFyYW1ldGVyczoge1xuICAgICAgJ1RyYW5zZm9ybUpvYk5hbWUuJCc6ICckLlRyYW5zZm9ybUpvYk5hbWUnLFxuICAgICAgJ01vZGVsTmFtZS4kJzogJyQuTW9kZWxOYW1lJyxcbiAgICAgICdUcmFuc2Zvcm1JbnB1dCc6IHtcbiAgICAgICAgRGF0YVNvdXJjZToge1xuICAgICAgICAgIFMzRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgUzNVcmk6ICdzMzovL2lucHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICAgICAgICBTM0RhdGFUeXBlOiAnUzNQcmVmaXgnLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgJ1RyYW5zZm9ybU91dHB1dCc6IHtcbiAgICAgICAgUzNPdXRwdXRQYXRoOiAnczM6Ly9vdXRwdXRidWNrZXQvcHJlZml4JyxcbiAgICAgIH0sXG4gICAgICAnVHJhbnNmb3JtUmVzb3VyY2VzJzoge1xuICAgICAgICBJbnN0YW5jZUNvdW50OiAxLFxuICAgICAgICBJbnN0YW5jZVR5cGU6ICdtbC5wMy4yeGxhcmdlJyxcbiAgICAgIH0sXG4gICAgfSxcbiAgfSk7XG59KTtcbiJdfQ==