import pytest
from aglfn import AGL

a = AGL()


def test_glyph_names():
    assert len(a.glyph_names()) == 3727


def test_all_glyph_names():
    assert len(a.all_glyphs()) == 1114112


info_to_test = [
    ('0', 48, '0030', ['zero']),
    ('1', 49, '0031', ['one']),
    ('2', 50, '0032', ['two']),
    ('3', 51, '0033', ['three']),
    ('4', 52, '0034', ['four']),
    ('5', 53, '0035', ['five']),
    ('6', 54, '0036', ['six']),
    ('7', 55, '0037', ['seven']),
    ('8', 56, '0038', ['eight']),
    ('9', 57, '0039', ['nine']),
    ('a', 97, '0061', ['a']),
    ('b', 98, '0062', ['b']),
    ('c', 99, '0063', ['c']),
    ('d', 100, '0064', ['d']),
    ('e', 101, '0065', ['e']),
    ('f', 102, '0066', ['f']),
    ('g', 103, '0067', ['g']),
    ('h', 104, '0068', ['h']),
    ('i', 105, '0069', ['i']),
    ('j', 106, '006A', ['j']),
    ('k', 107, '006B', ['k']),
    ('l', 108, '006C', ['l']),
    ('m', 109, '006D', ['m']),
    ('n', 110, '006E', ['n']),
    ('o', 111, '006F', ['o']),
    ('p', 112, '0070', ['p']),
    ('q', 113, '0071', ['q']),
    ('r', 114, '0072', ['r']),
    ('s', 115, '0073', ['s']),
    ('t', 116, '0074', ['t']),
    ('u', 117, '0075', ['u']),
    ('v', 118, '0076', ['v']),
    ('w', 119, '0077', ['w']),
    ('x', 120, '0078', ['x']),
    ('y', 121, '0079', ['y']),
    ('z', 122, '007A', ['z']),
    ('A', 65, '0041', ['A']),
    ('B', 66, '0042', ['B']),
    ('C', 67, '0043', ['C']),
    ('D', 68, '0044', ['D']),
    ('E', 69, '0045', ['E']),
    ('F', 70, '0046', ['F']),
    ('G', 71, '0047', ['G']),
    ('H', 72, '0048', ['H']),
    ('I', 73, '0049', ['I']),
    ('J', 74, '004A', ['J']),
    ('K', 75, '004B', ['K']),
    ('L', 76, '004C', ['L']),
    ('M', 77, '004D', ['M']),
    ('N', 78, '004E', ['N']),
    ('O', 79, '004F', ['O']),
    ('P', 80, '0050', ['P']),
    ('Q', 81, '0051', ['Q']),
    ('R', 82, '0052', ['R']),
    ('S', 83, '0053', ['S']),
    ('T', 84, '0054', ['T']),
    ('U', 85, '0055', ['U']),
    ('V', 86, '0056', ['V']),
    ('W', 87, '0057', ['W']),
    ('X', 88, '0058', ['X']),
    ('Y', 89, '0059', ['Y']),
    ('Z', 90, '005A', ['Z']),
    ('!', 33, '0021', ['exclam']),
    ('"', 34, '0022', ['quotedbl']),
    ('#', 35, '0023', ['numbersign']),
    ('$', 36, '0024', ['dollar']),
    ('%', 37, '0025', ['percent']),
    ('&', 38, '0026', ['ampersand']),
    ('\'', 39, '0027', ['quotesingle']),
    ('(', 40, '0028', ['parenleft']),
    (')', 41, '0029', ['parenright']),
    ('*', 42, '002A', ['asterisk']),
    ('+', 43, '002B', ['plus']),
    (',', 44, '002C', ['comma']),
    ('-', 45, '002D', ['hyphen']),
    ('.', 46, '002E', ['period']),
    ('/', 47, '002F', ['slash']),
    (':', 58, '003A', ['colon']),
    (';', 59, '003B', ['semicolon']),
    ('<', 60, '003C', ['less']),
    ('=', 61, '003D', ['equal']),
    ('>', 62, '003E', ['greater']),
    ('?', 63, '003F', ['question']),
    ('@', 64, '0040', ['at']),
    ('[', 91, '005B', ['bracketleft']),
    ('\\', 92, '005C', ['backslash']),
    (']', 93, '005D', ['bracketright']),
    ('^', 94, '005E', ['asciicircum']),
    ('_', 95, '005F', ['underscore']),
    ('`', 96, '0060', ['grave']),
    ('{', 123, '007B', ['braceleft']),
    ('|', 124, '007C', ['bar', 'verticalbar']),
    ('}', 125, '007D', ['braceright']),
    ('~', 126, '007E', ['asciitilde']),
    (' ', 32, '0020', ['space', 'spacehackarabic']),
    ('\t', 9, '0009', ['controlHT']),
    ('\n', 10, '000A', ['controlLF']),
    ('\r', 13, '000D', ['controlCR']),
    ('\x0b', 11, '000B', ['controlVT']),
    ('\x0c', 12, '000C', ['controlFF']),
]


@pytest.mark.parametrize('char, ucp, usv, agl', info_to_test)
def test_info_glyph(char, ucp, usv, agl):
    info = a.lookup(char)
    assert info.get('unicode_code_point') == ucp
    assert info.get('unicode_standard_value') == usv
    assert info.get('agl_glyph_names') == agl


values_to_test = [
    ('A', 'A'),
    ('Æ', 'AE'),
    ('Ǽ', 'AEacute'),
    ('Ǣ', 'AEmacron'),
    ('', 'AEsmall'),
    ('Á', 'Aacute'),
    ('', 'Aacutesmall'),
    ('Ă', 'Abreve'),
    ('Ắ', 'Abreveacute'),
    ('Ӑ', 'Abrevecyrillic'),
    ('Ặ', 'Abrevedotbelow'),
    ('Ằ', 'Abrevegrave'),
    ('Ẳ', 'Abrevehookabove'),
    ('Ẵ', 'Abrevetilde'),
    ('Ǎ', 'Acaron'),
    ('Ⓐ', 'Acircle'),
    ('Â', 'Acircumflex'),
    ('Ấ', 'Acircumflexacute'),
    ('Ậ', 'Acircumflexdotbelow'),
    ('Ầ', 'Acircumflexgrave'),
    ('Ẩ', 'Acircumflexhookabove'),
    ('', 'Acircumflexsmall'),
    ('Ẫ', 'Acircumflextilde'),
    ('', 'Acute'),
    ('', 'Acutesmall'),
    ('А', 'Acyrillic'),
    ('Ȁ', 'Adblgrave'),
    ('Ä', 'Adieresis'),
    ('Ӓ', 'Adieresiscyrillic'),
    ('Ǟ', 'Adieresismacron'),
    ('', 'Adieresissmall'),
    ('Ạ', 'Adotbelow'),
    ('Ǡ', 'Adotmacron'),
    ('À', 'Agrave'),
    ('', 'Agravesmall'),
    ('Ả', 'Ahookabove'),
    ('Ӕ', 'Aiecyrillic'),
    ('Ȃ', 'Ainvertedbreve'),
    ('Α', 'Alpha'),
    ('Ά', 'Alphatonos'),
    ('Ā', 'Amacron'),
    ('Ａ', 'Amonospace'),
    ('Ą', 'Aogonek'),
    ('Å', 'Aring'),
    ('Ǻ', 'Aringacute'),
    ('Ḁ', 'Aringbelow'),
    ('', 'Aringsmall'),
    ('', 'Asmall'),
    ('Ã', 'Atilde'),
    ('', 'Atildesmall'),
    ('Ա', 'Aybarmenian'),
    ('B', 'B'),
    ('Ⓑ', 'Bcircle'),
    ('Ḃ', 'Bdotaccent'),
    ('Ḅ', 'Bdotbelow'),
    ('Б', 'Becyrillic'),
    ('Բ', 'Benarmenian'),
    ('Β', 'Beta'),
    ('Ɓ', 'Bhook'),
    ('Ḇ', 'Blinebelow'),
    ('Ｂ', 'Bmonospace'),
    ('', 'Brevesmall'),
    ('', 'Bsmall'),
    ('Ƃ', 'Btopbar'),
    ('C', 'C'),
    ('Ծ', 'Caarmenian'),
    ('Ć', 'Cacute'),
    ('', 'Caron'),
    ('', 'Caronsmall'),
    ('Č', 'Ccaron'),
    ('Ç', 'Ccedilla'),
    ('Ḉ', 'Ccedillaacute'),
    ('', 'Ccedillasmall'),
    ('Ⓒ', 'Ccircle'),
    ('Ĉ', 'Ccircumflex'),
    ('Ċ', 'Cdot'),
    ('Ċ', 'Cdotaccent'),
    ('', 'Cedillasmall'),
    ('Չ', 'Chaarmenian'),
    ('Ҽ', 'Cheabkhasiancyrillic'),
    ('Ч', 'Checyrillic'),
    ('Ҿ', 'Chedescenderabkhasiancyrillic'),
    ('Ҷ', 'Chedescendercyrillic'),
    ('Ӵ', 'Chedieresiscyrillic'),
    ('Ճ', 'Cheharmenian'),
    ('Ӌ', 'Chekhakassiancyrillic'),
    ('Ҹ', 'Cheverticalstrokecyrillic'),
    ('Χ', 'Chi'),
    ('Ƈ', 'Chook'),
    ('', 'Circumflexsmall'),
    ('Ｃ', 'Cmonospace'),
    ('Ց', 'Coarmenian'),
    ('', 'Csmall'),
    ('D', 'D'),
    ('Ǳ', 'DZ'),
    ('Ǆ', 'DZcaron'),
    ('Դ', 'Daarmenian'),
    ('Ɖ', 'Dafrican'),
    ('Ď', 'Dcaron'),
    ('Ḑ', 'Dcedilla'),
    ('Ⓓ', 'Dcircle'),
    ('Ḓ', 'Dcircumflexbelow'),
    ('Đ', 'Dcroat'),
    ('Ḋ', 'Ddotaccent'),
    ('Ḍ', 'Ddotbelow'),
    ('Д', 'Decyrillic'),
    ('Ϯ', 'Deicoptic'),
    ('∆', 'Delta'),
    ('Δ', 'Deltagreek'),
    ('Ɗ', 'Dhook'),
    ('', 'Dieresis'),
    ('', 'DieresisAcute'),
    ('', 'DieresisGrave'),
    ('', 'Dieresissmall'),
    ('Ϝ', 'Digammagreek'),
    ('Ђ', 'Djecyrillic'),
    ('Ḏ', 'Dlinebelow'),
    ('Ｄ', 'Dmonospace'),
    ('', 'Dotaccentsmall'),
    ('Đ', 'Dslash'),
    ('', 'Dsmall'),
    ('Ƌ', 'Dtopbar'),
    ('ǲ', 'Dz'),
    ('ǅ', 'Dzcaron'),
    ('Ӡ', 'Dzeabkhasiancyrillic'),
    ('Ѕ', 'Dzecyrillic'),
    ('Џ', 'Dzhecyrillic'),
    ('E', 'E'),
    ('É', 'Eacute'),
    ('', 'Eacutesmall'),
    ('Ĕ', 'Ebreve'),
    ('Ě', 'Ecaron'),
    ('Ḝ', 'Ecedillabreve'),
    ('Ե', 'Echarmenian'),
    ('Ⓔ', 'Ecircle'),
    ('Ê', 'Ecircumflex'),
    ('Ế', 'Ecircumflexacute'),
    ('Ḙ', 'Ecircumflexbelow'),
    ('Ệ', 'Ecircumflexdotbelow'),
    ('Ề', 'Ecircumflexgrave'),
    ('Ể', 'Ecircumflexhookabove'),
    ('', 'Ecircumflexsmall'),
    ('Ễ', 'Ecircumflextilde'),
    ('Є', 'Ecyrillic'),
    ('Ȅ', 'Edblgrave'),
    ('Ë', 'Edieresis'),
    ('', 'Edieresissmall'),
    ('Ė', 'Edot'),
    ('Ė', 'Edotaccent'),
    ('Ẹ', 'Edotbelow'),
    ('Ф', 'Efcyrillic'),
    ('È', 'Egrave'),
    ('', 'Egravesmall'),
    ('Է', 'Eharmenian'),
    ('Ẻ', 'Ehookabove'),
    ('Ⅷ', 'Eightroman'),
    ('Ȇ', 'Einvertedbreve'),
    ('Ѥ', 'Eiotifiedcyrillic'),
    ('Л', 'Elcyrillic'),
    ('Ⅺ', 'Elevenroman'),
    ('Ē', 'Emacron'),
    ('Ḗ', 'Emacronacute'),
    ('Ḕ', 'Emacrongrave'),
    ('М', 'Emcyrillic'),
    ('Ｅ', 'Emonospace'),
    ('Н', 'Encyrillic'),
    ('Ң', 'Endescendercyrillic'),
    ('Ŋ', 'Eng'),
    ('Ҥ', 'Enghecyrillic'),
    ('Ӈ', 'Enhookcyrillic'),
    ('Ę', 'Eogonek'),
    ('Ɛ', 'Eopen'),
    ('Ε', 'Epsilon'),
    ('Έ', 'Epsilontonos'),
    ('Р', 'Ercyrillic'),
    ('Ǝ', 'Ereversed'),
    ('Э', 'Ereversedcyrillic'),
    ('С', 'Escyrillic'),
    ('Ҫ', 'Esdescendercyrillic'),
    ('Ʃ', 'Esh'),
    ('', 'Esmall'),
    ('Η', 'Eta'),
    ('Ը', 'Etarmenian'),
    ('Ή', 'Etatonos'),
    ('Ð', 'Eth'),
    ('', 'Ethsmall'),
    ('Ẽ', 'Etilde'),
    ('Ḛ', 'Etildebelow'),
    ('€', 'Euro'),
    ('Ʒ', 'Ezh'),
    ('Ǯ', 'Ezhcaron'),
    ('Ƹ', 'Ezhreversed'),
    ('F', 'F'),
    ('Ⓕ', 'Fcircle'),
    ('Ḟ', 'Fdotaccent'),
    ('Ֆ', 'Feharmenian'),
    ('Ϥ', 'Feicoptic'),
    ('Ƒ', 'Fhook'),
    ('Ѳ', 'Fitacyrillic'),
    ('Ⅴ', 'Fiveroman'),
    ('Ｆ', 'Fmonospace'),
    ('Ⅳ', 'Fourroman'),
    ('', 'Fsmall'),
    ('G', 'G'),
    ('㎇', 'GBsquare'),
    ('Ǵ', 'Gacute'),
    ('Γ', 'Gamma'),
    ('Ɣ', 'Gammaafrican'),
    ('Ϫ', 'Gangiacoptic'),
    ('Ğ', 'Gbreve'),
    ('Ǧ', 'Gcaron'),
    ('Ģ', 'Gcedilla'),
    ('Ⓖ', 'Gcircle'),
    ('Ĝ', 'Gcircumflex'),
    ('Ģ', 'Gcommaaccent'),
    ('Ġ', 'Gdot'),
    ('Ġ', 'Gdotaccent'),
    ('Г', 'Gecyrillic'),
    ('Ղ', 'Ghadarmenian'),
    ('Ҕ', 'Ghemiddlehookcyrillic'),
    ('Ғ', 'Ghestrokecyrillic'),
    ('Ґ', 'Gheupturncyrillic'),
    ('Ɠ', 'Ghook'),
    ('Գ', 'Gimarmenian'),
    ('Ѓ', 'Gjecyrillic'),
    ('Ḡ', 'Gmacron'),
    ('Ｇ', 'Gmonospace'),
    ('', 'Grave'),
    ('', 'Gravesmall'),
    ('', 'Gsmall'),
    ('ʛ', 'Gsmallhook'),
    ('Ǥ', 'Gstroke'),
    ('H', 'H'),
    ('●', 'H18533'),
    ('▪', 'H18543'),
    ('▫', 'H18551'),
    ('□', 'H22073'),
    ('㏋', 'HPsquare'),
    ('Ҩ', 'Haabkhasiancyrillic'),
    ('Ҳ', 'Hadescendercyrillic'),
    ('Ъ', 'Hardsigncyrillic'),
    ('Ħ', 'Hbar'),
    ('Ḫ', 'Hbrevebelow'),
    ('Ḩ', 'Hcedilla'),
    ('Ⓗ', 'Hcircle'),
    ('Ĥ', 'Hcircumflex'),
    ('Ḧ', 'Hdieresis'),
    ('Ḣ', 'Hdotaccent'),
    ('Ḥ', 'Hdotbelow'),
    ('Ｈ', 'Hmonospace'),
    ('Հ', 'Hoarmenian'),
    ('Ϩ', 'Horicoptic'),
    ('', 'Hsmall'),
    ('', 'Hungarumlaut'),
    ('', 'Hungarumlautsmall'),
    ('㎐', 'Hzsquare'),
    ('I', 'I'),
    ('Я', 'IAcyrillic'),
    ('Ĳ', 'IJ'),
    ('Ю', 'IUcyrillic'),
    ('Í', 'Iacute'),
    ('', 'Iacutesmall'),
    ('Ĭ', 'Ibreve'),
    ('Ǐ', 'Icaron'),
    ('Ⓘ', 'Icircle'),
    ('Î', 'Icircumflex'),
    ('', 'Icircumflexsmall'),
    ('І', 'Icyrillic'),
    ('Ȉ', 'Idblgrave'),
    ('Ï', 'Idieresis'),
    ('Ḯ', 'Idieresisacute'),
    ('Ӥ', 'Idieresiscyrillic'),
    ('', 'Idieresissmall'),
    ('İ', 'Idot'),
    ('İ', 'Idotaccent'),
    ('Ị', 'Idotbelow'),
    ('Ӗ', 'Iebrevecyrillic'),
    ('Е', 'Iecyrillic'),
    ('ℑ', 'Ifraktur'),
    ('Ì', 'Igrave'),
    ('', 'Igravesmall'),
    ('Ỉ', 'Ihookabove'),
    ('И', 'Iicyrillic'),
    ('Ȋ', 'Iinvertedbreve'),
    ('Й', 'Iishortcyrillic'),
    ('Ī', 'Imacron'),
    ('Ӣ', 'Imacroncyrillic'),
    ('Ｉ', 'Imonospace'),
    ('Ի', 'Iniarmenian'),
    ('Ё', 'Iocyrillic'),
    ('Į', 'Iogonek'),
    ('Ι', 'Iota'),
    ('Ɩ', 'Iotaafrican'),
    ('Ϊ', 'Iotadieresis'),
    ('Ί', 'Iotatonos'),
    ('', 'Ismall'),
    ('Ɨ', 'Istroke'),
    ('Ĩ', 'Itilde'),
    ('Ḭ', 'Itildebelow'),
    ('Ѵ', 'Izhitsacyrillic'),
    ('Ѷ', 'Izhitsadblgravecyrillic'),
    ('J', 'J'),
    ('Ձ', 'Jaarmenian'),
    ('Ⓙ', 'Jcircle'),
    ('Ĵ', 'Jcircumflex'),
    ('Ј', 'Jecyrillic'),
    ('Ջ', 'Jheharmenian'),
    ('Ｊ', 'Jmonospace'),
    ('', 'Jsmall'),
    ('K', 'K'),
    ('㎅', 'KBsquare'),
    ('㏍', 'KKsquare'),
    ('Ҡ', 'Kabashkircyrillic'),
    ('Ḱ', 'Kacute'),
    ('К', 'Kacyrillic'),
    ('Қ', 'Kadescendercyrillic'),
    ('Ӄ', 'Kahookcyrillic'),
    ('Κ', 'Kappa'),
    ('Ҟ', 'Kastrokecyrillic'),
    ('Ҝ', 'Kaverticalstrokecyrillic'),
    ('Ǩ', 'Kcaron'),
    ('Ķ', 'Kcedilla'),
    ('Ⓚ', 'Kcircle'),
    ('Ķ', 'Kcommaaccent'),
    ('Ḳ', 'Kdotbelow'),
    ('Ք', 'Keharmenian'),
    ('Կ', 'Kenarmenian'),
    ('Х', 'Khacyrillic'),
    ('Ϧ', 'Kheicoptic'),
    ('Ƙ', 'Khook'),
    ('Ќ', 'Kjecyrillic'),
    ('Ḵ', 'Klinebelow'),
    ('Ｋ', 'Kmonospace'),
    ('Ҁ', 'Koppacyrillic'),
    ('Ϟ', 'Koppagreek'),
    ('Ѯ', 'Ksicyrillic'),
    ('', 'Ksmall'),
    ('L', 'L'),
    ('Ǉ', 'LJ'),
    ('', 'LL'),
    ('Ĺ', 'Lacute'),
    ('Λ', 'Lambda'),
    ('Ľ', 'Lcaron'),
    ('Ļ', 'Lcedilla'),
    ('Ⓛ', 'Lcircle'),
    ('Ḽ', 'Lcircumflexbelow'),
    ('Ļ', 'Lcommaaccent'),
    ('Ŀ', 'Ldot'),
    ('Ŀ', 'Ldotaccent'),
    ('Ḷ', 'Ldotbelow'),
    ('Ḹ', 'Ldotbelowmacron'),
    ('Լ', 'Liwnarmenian'),
    ('ǈ', 'Lj'),
    ('Љ', 'Ljecyrillic'),
    ('Ḻ', 'Llinebelow'),
    ('Ｌ', 'Lmonospace'),
    ('Ł', 'Lslash'),
    ('', 'Lslashsmall'),
    ('', 'Lsmall'),
    ('M', 'M'),
    ('㎆', 'MBsquare'),
    ('', 'Macron'),
    ('', 'Macronsmall'),
    ('Ḿ', 'Macute'),
    ('Ⓜ', 'Mcircle'),
    ('Ṁ', 'Mdotaccent'),
    ('Ṃ', 'Mdotbelow'),
    ('Մ', 'Menarmenian'),
    ('Ｍ', 'Mmonospace'),
    ('', 'Msmall'),
    ('Ɯ', 'Mturned'),
    ('Μ', 'Mu'),
    ('N', 'N'),
    ('Ǌ', 'NJ'),
    ('Ń', 'Nacute'),
    ('Ň', 'Ncaron'),
    ('Ņ', 'Ncedilla'),
    ('Ⓝ', 'Ncircle'),
    ('Ṋ', 'Ncircumflexbelow'),
    ('Ņ', 'Ncommaaccent'),
    ('Ṅ', 'Ndotaccent'),
    ('Ṇ', 'Ndotbelow'),
    ('Ɲ', 'Nhookleft'),
    ('Ⅸ', 'Nineroman'),
    ('ǋ', 'Nj'),
    ('Њ', 'Njecyrillic'),
    ('Ṉ', 'Nlinebelow'),
    ('Ｎ', 'Nmonospace'),
    ('Ն', 'Nowarmenian'),
    ('', 'Nsmall'),
    ('Ñ', 'Ntilde'),
    ('', 'Ntildesmall'),
    ('Ν', 'Nu'),
    ('O', 'O'),
    ('Œ', 'OE'),
    ('', 'OEsmall'),
    ('Ó', 'Oacute'),
    ('', 'Oacutesmall'),
    ('Ө', 'Obarredcyrillic'),
    ('Ӫ', 'Obarreddieresiscyrillic'),
    ('Ŏ', 'Obreve'),
    ('Ǒ', 'Ocaron'),
    ('Ɵ', 'Ocenteredtilde'),
    ('Ⓞ', 'Ocircle'),
    ('Ô', 'Ocircumflex'),
    ('Ố', 'Ocircumflexacute'),
    ('Ộ', 'Ocircumflexdotbelow'),
    ('Ồ', 'Ocircumflexgrave'),
    ('Ổ', 'Ocircumflexhookabove'),
    ('', 'Ocircumflexsmall'),
    ('Ỗ', 'Ocircumflextilde'),
    ('О', 'Ocyrillic'),
    ('Ő', 'Odblacute'),
    ('Ȍ', 'Odblgrave'),
    ('Ö', 'Odieresis'),
    ('Ӧ', 'Odieresiscyrillic'),
    ('', 'Odieresissmall'),
    ('Ọ', 'Odotbelow'),
    ('', 'Ogoneksmall'),
    ('Ò', 'Ograve'),
    ('', 'Ogravesmall'),
    ('Օ', 'Oharmenian'),
    ('Ω', 'Ohm'),
    ('Ỏ', 'Ohookabove'),
    ('Ơ', 'Ohorn'),
    ('Ớ', 'Ohornacute'),
    ('Ợ', 'Ohorndotbelow'),
    ('Ờ', 'Ohorngrave'),
    ('Ở', 'Ohornhookabove'),
    ('Ỡ', 'Ohorntilde'),
    ('Ő', 'Ohungarumlaut'),
    ('Ƣ', 'Oi'),
    ('Ȏ', 'Oinvertedbreve'),
    ('Ō', 'Omacron'),
    ('Ṓ', 'Omacronacute'),
    ('Ṑ', 'Omacrongrave'),
    ('Ω', 'Omega'),
    ('Ѡ', 'Omegacyrillic'),
    ('Ω', 'Omegagreek'),
    ('Ѻ', 'Omegaroundcyrillic'),
    ('Ѽ', 'Omegatitlocyrillic'),
    ('Ώ', 'Omegatonos'),
    ('Ο', 'Omicron'),
    ('Ό', 'Omicrontonos'),
    ('Ｏ', 'Omonospace'),
    ('Ⅰ', 'Oneroman'),
    ('Ǫ', 'Oogonek'),
    ('Ǭ', 'Oogonekmacron'),
    ('Ɔ', 'Oopen'),
    ('Ø', 'Oslash'),
    ('Ǿ', 'Oslashacute'),
    ('', 'Oslashsmall'),
    ('', 'Osmall'),
    ('Ǿ', 'Ostrokeacute'),
    ('Ѿ', 'Otcyrillic'),
    ('Õ', 'Otilde'),
    ('Ṍ', 'Otildeacute'),
    ('Ṏ', 'Otildedieresis'),
    ('', 'Otildesmall'),
    ('P', 'P'),
    ('Ṕ', 'Pacute'),
    ('Ⓟ', 'Pcircle'),
    ('Ṗ', 'Pdotaccent'),
    ('П', 'Pecyrillic'),
    ('Պ', 'Peharmenian'),
    ('Ҧ', 'Pemiddlehookcyrillic'),
    ('Φ', 'Phi'),
    ('Ƥ', 'Phook'),
    ('Π', 'Pi'),
    ('Փ', 'Piwrarmenian'),
    ('Ｐ', 'Pmonospace'),
    ('Ψ', 'Psi'),
    ('Ѱ', 'Psicyrillic'),
    ('', 'Psmall'),
    ('Q', 'Q'),
    ('Ⓠ', 'Qcircle'),
    ('Ｑ', 'Qmonospace'),
    ('', 'Qsmall'),
    ('R', 'R'),
    ('Ռ', 'Raarmenian'),
    ('Ŕ', 'Racute'),
    ('Ř', 'Rcaron'),
    ('Ŗ', 'Rcedilla'),
    ('Ⓡ', 'Rcircle'),
    ('Ŗ', 'Rcommaaccent'),
    ('Ȑ', 'Rdblgrave'),
    ('Ṙ', 'Rdotaccent'),
    ('Ṛ', 'Rdotbelow'),
    ('Ṝ', 'Rdotbelowmacron'),
    ('Ր', 'Reharmenian'),
    ('ℜ', 'Rfraktur'),
    ('Ρ', 'Rho'),
    ('', 'Ringsmall'),
    ('Ȓ', 'Rinvertedbreve'),
    ('Ṟ', 'Rlinebelow'),
    ('Ｒ', 'Rmonospace'),
    ('', 'Rsmall'),
    ('ʁ', 'Rsmallinverted'),
    ('ʶ', 'Rsmallinvertedsuperior'),
    ('S', 'S'),
    ('┌', 'SF010000'),
    ('└', 'SF020000'),
    ('┐', 'SF030000'),
    ('┘', 'SF040000'),
    ('┼', 'SF050000'),
    ('┬', 'SF060000'),
    ('┴', 'SF070000'),
    ('├', 'SF080000'),
    ('┤', 'SF090000'),
    ('─', 'SF100000'),
    ('│', 'SF110000'),
    ('╡', 'SF190000'),
    ('╢', 'SF200000'),
    ('╖', 'SF210000'),
    ('╕', 'SF220000'),
    ('╣', 'SF230000'),
    ('║', 'SF240000'),
    ('╗', 'SF250000'),
    ('╝', 'SF260000'),
    ('╜', 'SF270000'),
    ('╛', 'SF280000'),
    ('╞', 'SF360000'),
    ('╟', 'SF370000'),
    ('╚', 'SF380000'),
    ('╔', 'SF390000'),
    ('╩', 'SF400000'),
    ('╦', 'SF410000'),
    ('╠', 'SF420000'),
    ('═', 'SF430000'),
    ('╬', 'SF440000'),
    ('╧', 'SF450000'),
    ('╨', 'SF460000'),
    ('╤', 'SF470000'),
    ('╥', 'SF480000'),
    ('╙', 'SF490000'),
    ('╘', 'SF500000'),
    ('╒', 'SF510000'),
    ('╓', 'SF520000'),
    ('╫', 'SF530000'),
    ('╪', 'SF540000'),
    ('Ś', 'Sacute'),
    ('Ṥ', 'Sacutedotaccent'),
    ('Ϡ', 'Sampigreek'),
    ('Š', 'Scaron'),
    ('Ṧ', 'Scarondotaccent'),
    ('', 'Scaronsmall'),
    ('Ş', 'Scedilla'),
    ('Ə', 'Schwa'),
    ('Ә', 'Schwacyrillic'),
    ('Ӛ', 'Schwadieresiscyrillic'),
    ('Ⓢ', 'Scircle'),
    ('Ŝ', 'Scircumflex'),
    ('Ș', 'Scommaaccent'),
    ('Ṡ', 'Sdotaccent'),
    ('Ṣ', 'Sdotbelow'),
    ('Ṩ', 'Sdotbelowdotaccent'),
    ('Ս', 'Seharmenian'),
    ('Ⅶ', 'Sevenroman'),
    ('Շ', 'Shaarmenian'),
    ('Ш', 'Shacyrillic'),
    ('Щ', 'Shchacyrillic'),
    ('Ϣ', 'Sheicoptic'),
    ('Һ', 'Shhacyrillic'),
    ('Ϭ', 'Shimacoptic'),
    ('Σ', 'Sigma'),
    ('Ⅵ', 'Sixroman'),
    ('Ｓ', 'Smonospace'),
    ('Ь', 'Softsigncyrillic'),
    ('', 'Ssmall'),
    ('Ϛ', 'Stigmagreek'),
    ('T', 'T'),
    ('Τ', 'Tau'),
    ('Ŧ', 'Tbar'),
    ('Ť', 'Tcaron'),
    ('Ţ', 'Tcedilla'),
    ('Ⓣ', 'Tcircle'),
    ('Ṱ', 'Tcircumflexbelow'),
    ('Ţ', 'Tcommaaccent'),
    ('Ṫ', 'Tdotaccent'),
    ('Ṭ', 'Tdotbelow'),
    ('Т', 'Tecyrillic'),
    ('Ҭ', 'Tedescendercyrillic'),
    ('Ⅹ', 'Tenroman'),
    ('Ҵ', 'Tetsecyrillic'),
    ('Θ', 'Theta'),
    ('Ƭ', 'Thook'),
    ('Þ', 'Thorn'),
    ('', 'Thornsmall'),
    ('Ⅲ', 'Threeroman'),
    ('', 'Tildesmall'),
    ('Տ', 'Tiwnarmenian'),
    ('Ṯ', 'Tlinebelow'),
    ('Ｔ', 'Tmonospace'),
    ('Թ', 'Toarmenian'),
    ('Ƽ', 'Tonefive'),
    ('Ƅ', 'Tonesix'),
    ('Ƨ', 'Tonetwo'),
    ('Ʈ', 'Tretroflexhook'),
    ('Ц', 'Tsecyrillic'),
    ('Ћ', 'Tshecyrillic'),
    ('', 'Tsmall'),
    ('Ⅻ', 'Twelveroman'),
    ('Ⅱ', 'Tworoman'),
    ('U', 'U'),
    ('Ú', 'Uacute'),
    ('', 'Uacutesmall'),
    ('Ŭ', 'Ubreve'),
    ('Ǔ', 'Ucaron'),
    ('Ⓤ', 'Ucircle'),
    ('Û', 'Ucircumflex'),
    ('Ṷ', 'Ucircumflexbelow'),
    ('', 'Ucircumflexsmall'),
    ('У', 'Ucyrillic'),
    ('Ű', 'Udblacute'),
    ('Ȕ', 'Udblgrave'),
    ('Ü', 'Udieresis'),
    ('Ǘ', 'Udieresisacute'),
    ('Ṳ', 'Udieresisbelow'),
    ('Ǚ', 'Udieresiscaron'),
    ('Ӱ', 'Udieresiscyrillic'),
    ('Ǜ', 'Udieresisgrave'),
    ('Ǖ', 'Udieresismacron'),
    ('', 'Udieresissmall'),
    ('Ụ', 'Udotbelow'),
    ('Ù', 'Ugrave'),
    ('', 'Ugravesmall'),
    ('Ủ', 'Uhookabove'),
    ('Ư', 'Uhorn'),
    ('Ứ', 'Uhornacute'),
    ('Ự', 'Uhorndotbelow'),
    ('Ừ', 'Uhorngrave'),
    ('Ử', 'Uhornhookabove'),
    ('Ữ', 'Uhorntilde'),
    ('Ű', 'Uhungarumlaut'),
    ('Ӳ', 'Uhungarumlautcyrillic'),
    ('Ȗ', 'Uinvertedbreve'),
    ('Ѹ', 'Ukcyrillic'),
    ('Ū', 'Umacron'),
    ('Ӯ', 'Umacroncyrillic'),
    ('Ṻ', 'Umacrondieresis'),
    ('Ｕ', 'Umonospace'),
    ('Ų', 'Uogonek'),
    ('Υ', 'Upsilon'),
    ('ϒ', 'Upsilon1'),
    ('ϓ', 'Upsilonacutehooksymbolgreek'),
    ('Ʊ', 'Upsilonafrican'),
    ('Ϋ', 'Upsilondieresis'),
    ('ϔ', 'Upsilondieresishooksymbolgreek'),
    ('ϒ', 'Upsilonhooksymbol'),
    ('Ύ', 'Upsilontonos'),
    ('Ů', 'Uring'),
    ('Ў', 'Ushortcyrillic'),
    ('', 'Usmall'),
    ('Ү', 'Ustraightcyrillic'),
    ('Ұ', 'Ustraightstrokecyrillic'),
    ('Ũ', 'Utilde'),
    ('Ṹ', 'Utildeacute'),
    ('Ṵ', 'Utildebelow'),
    ('V', 'V'),
    ('Ⓥ', 'Vcircle'),
    ('Ṿ', 'Vdotbelow'),
    ('В', 'Vecyrillic'),
    ('Վ', 'Vewarmenian'),
    ('Ʋ', 'Vhook'),
    ('Ｖ', 'Vmonospace'),
    ('Ո', 'Voarmenian'),
    ('', 'Vsmall'),
    ('Ṽ', 'Vtilde'),
    ('W', 'W'),
    ('Ẃ', 'Wacute'),
    ('Ⓦ', 'Wcircle'),
    ('Ŵ', 'Wcircumflex'),
    ('Ẅ', 'Wdieresis'),
    ('Ẇ', 'Wdotaccent'),
    ('Ẉ', 'Wdotbelow'),
    ('Ẁ', 'Wgrave'),
    ('Ｗ', 'Wmonospace'),
    ('', 'Wsmall'),
    ('X', 'X'),
    ('Ⓧ', 'Xcircle'),
    ('Ẍ', 'Xdieresis'),
    ('Ẋ', 'Xdotaccent'),
    ('Խ', 'Xeharmenian'),
    ('Ξ', 'Xi'),
    ('Ｘ', 'Xmonospace'),
    ('', 'Xsmall'),
    ('Y', 'Y'),
    ('Ý', 'Yacute'),
    ('', 'Yacutesmall'),
    ('Ѣ', 'Yatcyrillic'),
    ('Ⓨ', 'Ycircle'),
    ('Ŷ', 'Ycircumflex'),
    ('Ÿ', 'Ydieresis'),
    ('', 'Ydieresissmall'),
    ('Ẏ', 'Ydotaccent'),
    ('Ỵ', 'Ydotbelow'),
    ('Ы', 'Yericyrillic'),
    ('Ӹ', 'Yerudieresiscyrillic'),
    ('Ỳ', 'Ygrave'),
    ('Ƴ', 'Yhook'),
    ('Ỷ', 'Yhookabove'),
    ('Յ', 'Yiarmenian'),
    ('Ї', 'Yicyrillic'),
    ('Ւ', 'Yiwnarmenian'),
    ('Ｙ', 'Ymonospace'),
    ('', 'Ysmall'),
    ('Ỹ', 'Ytilde'),
    ('Ѫ', 'Yusbigcyrillic'),
    ('Ѭ', 'Yusbigiotifiedcyrillic'),
    ('Ѧ', 'Yuslittlecyrillic'),
    ('Ѩ', 'Yuslittleiotifiedcyrillic'),
    ('Z', 'Z'),
    ('Զ', 'Zaarmenian'),
    ('Ź', 'Zacute'),
    ('Ž', 'Zcaron'),
    ('', 'Zcaronsmall'),
    ('Ⓩ', 'Zcircle'),
    ('Ẑ', 'Zcircumflex'),
    ('Ż', 'Zdot'),
    ('Ż', 'Zdotaccent'),
    ('Ẓ', 'Zdotbelow'),
    ('З', 'Zecyrillic'),
    ('Ҙ', 'Zedescendercyrillic'),
    ('Ӟ', 'Zedieresiscyrillic'),
    ('Ζ', 'Zeta'),
    ('Ժ', 'Zhearmenian'),
    ('Ӂ', 'Zhebrevecyrillic'),
    ('Ж', 'Zhecyrillic'),
    ('Җ', 'Zhedescendercyrillic'),
    ('Ӝ', 'Zhedieresiscyrillic'),
    ('Ẕ', 'Zlinebelow'),
    ('Ｚ', 'Zmonospace'),
    ('', 'Zsmall'),
    ('Ƶ', 'Zstroke'),
    ('a', 'a'),
    ('আ', 'aabengali'),
    ('á', 'aacute'),
    ('आ', 'aadeva'),
    ('આ', 'aagujarati'),
    ('ਆ', 'aagurmukhi'),
    ('ਾ', 'aamatragurmukhi'),
    ('㌃', 'aarusquare'),
    ('া', 'aavowelsignbengali'),
    ('ा', 'aavowelsigndeva'),
    ('ા', 'aavowelsigngujarati'),
    ('՟', 'abbreviationmarkarmenian'),
    ('॰', 'abbreviationsigndeva'),
    ('অ', 'abengali'),
    ('ㄚ', 'abopomofo'),
    ('ă', 'abreve'),
    ('ắ', 'abreveacute'),
    ('ӑ', 'abrevecyrillic'),
    ('ặ', 'abrevedotbelow'),
    ('ằ', 'abrevegrave'),
    ('ẳ', 'abrevehookabove'),
    ('ẵ', 'abrevetilde'),
    ('ǎ', 'acaron'),
    ('ⓐ', 'acircle'),
    ('â', 'acircumflex'),
    ('ấ', 'acircumflexacute'),
    ('ậ', 'acircumflexdotbelow'),
    ('ầ', 'acircumflexgrave'),
    ('ẩ', 'acircumflexhookabove'),
    ('ẫ', 'acircumflextilde'),
    ('´', 'acute'),
    ('̗', 'acutebelowcmb'),
    ('́', 'acutecmb'),
    ('́', 'acutecomb'),
    ('॔', 'acutedeva'),
    ('ˏ', 'acutelowmod'),
    ('́', 'acutetonecmb'),
    ('а', 'acyrillic'),
    ('ȁ', 'adblgrave'),
    ('ੱ', 'addakgurmukhi'),
    ('अ', 'adeva'),
    ('ä', 'adieresis'),
    ('ӓ', 'adieresiscyrillic'),
    ('ǟ', 'adieresismacron'),
    ('ạ', 'adotbelow'),
    ('ǡ', 'adotmacron'),
    ('æ', 'ae'),
    ('ǽ', 'aeacute'),
    ('ㅐ', 'aekorean'),
    ('ǣ', 'aemacron'),
    ('―', 'afii00208'),
    ('₤', 'afii08941'),
    ('А', 'afii10017'),
    ('Б', 'afii10018'),
    ('В', 'afii10019'),
    ('Г', 'afii10020'),
    ('Д', 'afii10021'),
    ('Е', 'afii10022'),
    ('Ё', 'afii10023'),
    ('Ж', 'afii10024'),
    ('З', 'afii10025'),
    ('И', 'afii10026'),
    ('Й', 'afii10027'),
    ('К', 'afii10028'),
    ('Л', 'afii10029'),
    ('М', 'afii10030'),
    ('Н', 'afii10031'),
    ('О', 'afii10032'),
    ('П', 'afii10033'),
    ('Р', 'afii10034'),
    ('С', 'afii10035'),
    ('Т', 'afii10036'),
    ('У', 'afii10037'),
    ('Ф', 'afii10038'),
    ('Х', 'afii10039'),
    ('Ц', 'afii10040'),
    ('Ч', 'afii10041'),
    ('Ш', 'afii10042'),
    ('Щ', 'afii10043'),
    ('Ъ', 'afii10044'),
    ('Ы', 'afii10045'),
    ('Ь', 'afii10046'),
    ('Э', 'afii10047'),
    ('Ю', 'afii10048'),
    ('Я', 'afii10049'),
    ('Ґ', 'afii10050'),
    ('Ђ', 'afii10051'),
    ('Ѓ', 'afii10052'),
    ('Є', 'afii10053'),
    ('Ѕ', 'afii10054'),
    ('І', 'afii10055'),
    ('Ї', 'afii10056'),
    ('Ј', 'afii10057'),
    ('Љ', 'afii10058'),
    ('Њ', 'afii10059'),
    ('Ћ', 'afii10060'),
    ('Ќ', 'afii10061'),
    ('Ў', 'afii10062'),
    ('', 'afii10063'),
    ('', 'afii10064'),
    ('а', 'afii10065'),
    ('б', 'afii10066'),
    ('в', 'afii10067'),
    ('г', 'afii10068'),
    ('д', 'afii10069'),
    ('е', 'afii10070'),
    ('ё', 'afii10071'),
    ('ж', 'afii10072'),
    ('з', 'afii10073'),
    ('и', 'afii10074'),
    ('й', 'afii10075'),
    ('к', 'afii10076'),
    ('л', 'afii10077'),
    ('м', 'afii10078'),
    ('н', 'afii10079'),
    ('о', 'afii10080'),
    ('п', 'afii10081'),
    ('р', 'afii10082'),
    ('с', 'afii10083'),
    ('т', 'afii10084'),
    ('у', 'afii10085'),
    ('ф', 'afii10086'),
    ('х', 'afii10087'),
    ('ц', 'afii10088'),
    ('ч', 'afii10089'),
    ('ш', 'afii10090'),
    ('щ', 'afii10091'),
    ('ъ', 'afii10092'),
    ('ы', 'afii10093'),
    ('ь', 'afii10094'),
    ('э', 'afii10095'),
    ('ю', 'afii10096'),
    ('я', 'afii10097'),
    ('ґ', 'afii10098'),
    ('ђ', 'afii10099'),
    ('ѓ', 'afii10100'),
    ('є', 'afii10101'),
    ('ѕ', 'afii10102'),
    ('і', 'afii10103'),
    ('ї', 'afii10104'),
    ('ј', 'afii10105'),
    ('љ', 'afii10106'),
    ('њ', 'afii10107'),
    ('ћ', 'afii10108'),
    ('ќ', 'afii10109'),
    ('ў', 'afii10110'),
    ('Џ', 'afii10145'),
    ('Ѣ', 'afii10146'),
    ('Ѳ', 'afii10147'),
    ('Ѵ', 'afii10148'),
    ('', 'afii10192'),
    ('џ', 'afii10193'),
    ('ѣ', 'afii10194'),
    ('ѳ', 'afii10195'),
    ('ѵ', 'afii10196'),
    ('', 'afii10831'),
    ('', 'afii10832'),
    ('ә', 'afii10846'),
    # ('', 'afii299'),
    # ('', 'afii300'),
    # ('‍', 'afii301'),
    ('٪', 'afii57381'),
    ('،', 'afii57388'),
    ('٠', 'afii57392'),
    ('١', 'afii57393'),
    ('٢', 'afii57394'),
    ('٣', 'afii57395'),
    ('٤', 'afii57396'),
    ('٥', 'afii57397'),
    ('٦', 'afii57398'),
    ('٧', 'afii57399'),
    ('٨', 'afii57400'),
    ('٩', 'afii57401'),
    ('؛', 'afii57403'),
    ('؟', 'afii57407'),
    ('ء', 'afii57409'),
    ('آ', 'afii57410'),
    ('أ', 'afii57411'),
    ('ؤ', 'afii57412'),
    ('إ', 'afii57413'),
    ('ئ', 'afii57414'),
    ('ا', 'afii57415'),
    ('ب', 'afii57416'),
    ('ة', 'afii57417'),
    ('ت', 'afii57418'),
    ('ث', 'afii57419'),
    ('ج', 'afii57420'),
    ('ح', 'afii57421'),
    ('خ', 'afii57422'),
    ('د', 'afii57423'),
    ('ذ', 'afii57424'),
    ('ر', 'afii57425'),
    ('ز', 'afii57426'),
    ('س', 'afii57427'),
    ('ش', 'afii57428'),
    ('ص', 'afii57429'),
    ('ض', 'afii57430'),
    ('ط', 'afii57431'),
    ('ظ', 'afii57432'),
    ('ع', 'afii57433'),
    ('غ', 'afii57434'),
    ('ـ', 'afii57440'),
    ('ف', 'afii57441'),
    ('ق', 'afii57442'),
    ('ك', 'afii57443'),
    ('ل', 'afii57444'),
    ('م', 'afii57445'),
    ('ن', 'afii57446'),
    ('و', 'afii57448'),
    ('ى', 'afii57449'),
    ('ي', 'afii57450'),
    ('ً', 'afii57451'),
    ('ٌ', 'afii57452'),
    ('ٍ', 'afii57453'),
    ('َ', 'afii57454'),
    ('ُ', 'afii57455'),
    ('ِ', 'afii57456'),
    ('ّ', 'afii57457'),
    ('ْ', 'afii57458'),
    ('ه', 'afii57470'),
    ('ڤ', 'afii57505'),
    ('پ', 'afii57506'),
    ('چ', 'afii57507'),
    ('ژ', 'afii57508'),
    ('گ', 'afii57509'),
    ('ٹ', 'afii57511'),
    ('ڈ', 'afii57512'),
    ('ڑ', 'afii57513'),
    ('ں', 'afii57514'),
    ('ے', 'afii57519'),
    ('ە', 'afii57534'),
    ('₪', 'afii57636'),
    ('־', 'afii57645'),
    ('׃', 'afii57658'),
    ('א', 'afii57664'),
    ('ב', 'afii57665'),
    ('ג', 'afii57666'),
    ('ד', 'afii57667'),
    ('ה', 'afii57668'),
    ('ו', 'afii57669'),
    ('ז', 'afii57670'),
    ('ח', 'afii57671'),
    ('ט', 'afii57672'),
    ('י', 'afii57673'),
    ('ך', 'afii57674'),
    ('כ', 'afii57675'),
    ('ל', 'afii57676'),
    ('ם', 'afii57677'),
    ('מ', 'afii57678'),
    ('ן', 'afii57679'),
    ('נ', 'afii57680'),
    ('ס', 'afii57681'),
    ('ע', 'afii57682'),
    ('ף', 'afii57683'),
    ('פ', 'afii57684'),
    ('ץ', 'afii57685'),
    ('צ', 'afii57686'),
    ('ק', 'afii57687'),
    ('ר', 'afii57688'),
    ('ש', 'afii57689'),
    ('ת', 'afii57690'),
    ('שׁ', 'afii57694'),
    ('שׂ', 'afii57695'),
    ('וֹ', 'afii57700'),
    ('ײַ', 'afii57705'),
    ('װ', 'afii57716'),
    ('ױ', 'afii57717'),
    ('ײ', 'afii57718'),
    ('וּ', 'afii57723'),
    ('ִ', 'afii57793'),
    ('ֵ', 'afii57794'),
    ('ֶ', 'afii57795'),
    ('ֻ', 'afii57796'),
    ('ָ', 'afii57797'),
    ('ַ', 'afii57798'),
    ('ְ', 'afii57799'),
    ('ֲ', 'afii57800'),
    ('ֱ', 'afii57801'),
    ('ֳ', 'afii57802'),
    ('ׂ', 'afii57803'),
    ('ׁ', 'afii57804'),
    ('ֹ', 'afii57806'),
    ('ּ', 'afii57807'),
    ('ֽ', 'afii57839'),
    ('ֿ', 'afii57841'),
    ('׀', 'afii57842'),
    ('ʼ', 'afii57929'),
    ('℅', 'afii61248'),
    ('ℓ', 'afii61289'),
    ('№', 'afii61352'),
    # ('', 'afii61573'),
    # ('', 'afii61574'),
    # ('', 'afii61575'),
    # ('', 'afii61664'),
    ('٭', 'afii63167'),
    ('ʽ', 'afii64937'),
    ('à', 'agrave'),
    ('અ', 'agujarati'),
    ('ਅ', 'agurmukhi'),
    ('あ', 'ahiragana'),
    ('ả', 'ahookabove'),
    ('ঐ', 'aibengali'),
    ('ㄞ', 'aibopomofo'),
    ('ऐ', 'aideva'),
    ('ӕ', 'aiecyrillic'),
    ('ઐ', 'aigujarati'),
    ('ਐ', 'aigurmukhi'),
    ('ੈ', 'aimatragurmukhi'),
    ('ع', 'ainarabic'),
    ('ﻊ', 'ainfinalarabic'),
    ('ﻋ', 'aininitialarabic'),
    ('ﻌ', 'ainmedialarabic'),
    ('ȃ', 'ainvertedbreve'),
    ('ৈ', 'aivowelsignbengali'),
    ('ै', 'aivowelsigndeva'),
    ('ૈ', 'aivowelsigngujarati'),
    ('ア', 'akatakana'),
    ('ｱ', 'akatakanahalfwidth'),
    ('ㅏ', 'akorean'),
    ('א', 'alef'),
    ('ا', 'alefarabic'),
    ('אּ', 'alefdageshhebrew'),
    ('ﺎ', 'aleffinalarabic'),
    ('أ', 'alefhamzaabovearabic'),
    ('ﺄ', 'alefhamzaabovefinalarabic'),
    ('إ', 'alefhamzabelowarabic'),
    ('ﺈ', 'alefhamzabelowfinalarabic'),
    ('א', 'alefhebrew'),
    ('ﭏ', 'aleflamedhebrew'),
    ('آ', 'alefmaddaabovearabic'),
    ('ﺂ', 'alefmaddaabovefinalarabic'),
    ('ى', 'alefmaksuraarabic'),
    ('ﻰ', 'alefmaksurafinalarabic'),
    ('ﻳ', 'alefmaksurainitialarabic'),
    ('ﻴ', 'alefmaksuramedialarabic'),
    ('אַ', 'alefpatahhebrew'),
    ('אָ', 'alefqamatshebrew'),
    ('ℵ', 'aleph'),
    ('≌', 'allequal'),
    ('α', 'alpha'),
    ('ά', 'alphatonos'),
    ('ā', 'amacron'),
    ('ａ', 'amonospace'),
    ('&', 'ampersand'),
    ('＆', 'ampersandmonospace'),
    ('', 'ampersandsmall'),
    ('㏂', 'amsquare'),
    ('ㄢ', 'anbopomofo'),
    ('ㄤ', 'angbopomofo'),
    ('๚', 'angkhankhuthai'),
    ('∠', 'angle'),
    ('〈', 'anglebracketleft'),
    ('︿', 'anglebracketleftvertical'),
    ('〉', 'anglebracketright'),
    ('﹀', 'anglebracketrightvertical'),
    ('〈', 'angleleft'),
    ('〉', 'angleright'),
    ('Å', 'angstrom'),
    ('·', 'anoteleia'),
    ('॒', 'anudattadeva'),
    ('ং', 'anusvarabengali'),
    ('ं', 'anusvaradeva'),
    ('ં', 'anusvaragujarati'),
    ('ą', 'aogonek'),
    ('㌀', 'apaatosquare'),
    ('⒜', 'aparen'),
    ('՚', 'apostrophearmenian'),
    ('ʼ', 'apostrophemod'),
    ('', 'apple'),
    ('≐', 'approaches'),
    ('≈', 'approxequal'),
    ('≒', 'approxequalorimage'),
    ('≅', 'approximatelyequal'),
    ('ㆎ', 'araeaekorean'),
    ('ㆍ', 'araeakorean'),
    ('⌒', 'arc'),
    ('ẚ', 'arighthalfring'),
    ('å', 'aring'),
    ('ǻ', 'aringacute'),
    ('ḁ', 'aringbelow'),
    ('↔', 'arrowboth'),
    ('⇣', 'arrowdashdown'),
    ('⇠', 'arrowdashleft'),
    ('⇢', 'arrowdashright'),
    ('⇡', 'arrowdashup'),
    ('⇔', 'arrowdblboth'),
    ('⇓', 'arrowdbldown'),
    ('⇐', 'arrowdblleft'),
    ('⇒', 'arrowdblright'),
    ('⇑', 'arrowdblup'),
    ('↓', 'arrowdown'),
    ('↙', 'arrowdownleft'),
    ('↘', 'arrowdownright'),
    ('⇩', 'arrowdownwhite'),
    ('˅', 'arrowheaddownmod'),
    ('˂', 'arrowheadleftmod'),
    ('˃', 'arrowheadrightmod'),
    ('˄', 'arrowheadupmod'),
    ('', 'arrowhorizex'),
    ('←', 'arrowleft'),
    ('⇐', 'arrowleftdbl'),
    ('⇍', 'arrowleftdblstroke'),
    ('⇆', 'arrowleftoverright'),
    ('⇦', 'arrowleftwhite'),
    ('→', 'arrowright'),
    ('⇏', 'arrowrightdblstroke'),
    ('➞', 'arrowrightheavy'),
    ('⇄', 'arrowrightoverleft'),
    ('⇨', 'arrowrightwhite'),
    ('⇤', 'arrowtableft'),
    ('⇥', 'arrowtabright'),
    ('↑', 'arrowup'),
    ('↕', 'arrowupdn'),
    ('↨', 'arrowupdnbse'),
    ('↨', 'arrowupdownbase'),
    ('↖', 'arrowupleft'),
    ('⇅', 'arrowupleftofdown'),
    ('↗', 'arrowupright'),
    ('⇧', 'arrowupwhite'),
    ('', 'arrowvertex'),
    ('^', 'asciicircum'),
    ('＾', 'asciicircummonospace'),
    ('~', 'asciitilde'),
    ('～', 'asciitildemonospace'),
    ('ɑ', 'ascript'),
    ('ɒ', 'ascriptturned'),
    ('ぁ', 'asmallhiragana'),
    ('ァ', 'asmallkatakana'),
    ('ｧ', 'asmallkatakanahalfwidth'),
    ('*', 'asterisk'),
    ('٭', 'asteriskaltonearabic'),
    ('٭', 'asteriskarabic'),
    ('∗', 'asteriskmath'),
    ('＊', 'asteriskmonospace'),
    ('﹡', 'asterisksmall'),
    ('⁂', 'asterism'),
    ('', 'asuperior'),
    ('≃', 'asymptoticallyequal'),
    ('@', 'at'),
    ('ã', 'atilde'),
    ('＠', 'atmonospace'),
    ('﹫', 'atsmall'),
    ('ɐ', 'aturned'),
    ('ঔ', 'aubengali'),
    ('ㄠ', 'aubopomofo'),
    ('औ', 'audeva'),
    ('ઔ', 'augujarati'),
    ('ਔ', 'augurmukhi'),
    ('ৗ', 'aulengthmarkbengali'),
    ('ੌ', 'aumatragurmukhi'),
    ('ৌ', 'auvowelsignbengali'),
    ('ौ', 'auvowelsigndeva'),
    ('ૌ', 'auvowelsigngujarati'),
    ('ऽ', 'avagrahadeva'),
    ('ա', 'aybarmenian'),
    ('ע', 'ayin'),
    ('ﬠ', 'ayinaltonehebrew'),
    ('ע', 'ayinhebrew'),
    ('b', 'b'),
    ('ব', 'babengali'),
    ('\\', 'backslash'),
    ('＼', 'backslashmonospace'),
    ('ब', 'badeva'),
    ('બ', 'bagujarati'),
    ('ਬ', 'bagurmukhi'),
    ('ば', 'bahiragana'),
    ('฿', 'bahtthai'),
    ('バ', 'bakatakana'),
    ('|', 'bar'),
    ('｜', 'barmonospace'),
    ('ㄅ', 'bbopomofo'),
    ('ⓑ', 'bcircle'),
    ('ḃ', 'bdotaccent'),
    ('ḅ', 'bdotbelow'),
    ('♬', 'beamedsixteenthnotes'),
    ('∵', 'because'),
    ('б', 'becyrillic'),
    ('ب', 'beharabic'),
    ('ﺐ', 'behfinalarabic'),
    ('ﺑ', 'behinitialarabic'),
    ('べ', 'behiragana'),
    ('ﺒ', 'behmedialarabic'),
    ('ﲟ', 'behmeeminitialarabic'),
    ('ﰈ', 'behmeemisolatedarabic'),
    ('ﱭ', 'behnoonfinalarabic'),
    ('ベ', 'bekatakana'),
    ('բ', 'benarmenian'),
    ('ב', 'bet'),
    ('β', 'beta'),
    ('ϐ', 'betasymbolgreek'),
    ('בּ', 'betdagesh'),
    ('בּ', 'betdageshhebrew'),
    ('ב', 'bethebrew'),
    ('בֿ', 'betrafehebrew'),
    ('ভ', 'bhabengali'),
    ('भ', 'bhadeva'),
    ('ભ', 'bhagujarati'),
    ('ਭ', 'bhagurmukhi'),
    ('ɓ', 'bhook'),
    ('び', 'bihiragana'),
    ('ビ', 'bikatakana'),
    ('ʘ', 'bilabialclick'),
    ('ਂ', 'bindigurmukhi'),
    ('㌱', 'birusquare'),
    ('●', 'blackcircle'),
    ('◆', 'blackdiamond'),
    ('▼', 'blackdownpointingtriangle'),
    ('◄', 'blackleftpointingpointer'),
    ('◀', 'blackleftpointingtriangle'),
    ('【', 'blacklenticularbracketleft'),
    ('︻', 'blacklenticularbracketleftvertical'),
    ('】', 'blacklenticularbracketright'),
    ('︼', 'blacklenticularbracketrightvertical'),
    ('◣', 'blacklowerlefttriangle'),
    ('◢', 'blacklowerrighttriangle'),
    ('▬', 'blackrectangle'),
    ('►', 'blackrightpointingpointer'),
    ('▶', 'blackrightpointingtriangle'),
    ('▪', 'blacksmallsquare'),
    ('☻', 'blacksmilingface'),
    ('■', 'blacksquare'),
    ('★', 'blackstar'),
    ('◤', 'blackupperlefttriangle'),
    ('◥', 'blackupperrighttriangle'),
    ('▴', 'blackuppointingsmalltriangle'),
    ('▲', 'blackuppointingtriangle'),
    ('␣', 'blank'),
    ('ḇ', 'blinebelow'),
    ('█', 'block'),
    ('ｂ', 'bmonospace'),
    ('บ', 'bobaimaithai'),
    ('ぼ', 'bohiragana'),
    ('ボ', 'bokatakana'),
    ('⒝', 'bparen'),
    ('㏃', 'bqsquare'),
    ('', 'braceex'),
    ('{', 'braceleft'),
    ('', 'braceleftbt'),
    ('', 'braceleftmid'),
    ('｛', 'braceleftmonospace'),
    ('﹛', 'braceleftsmall'),
    ('', 'bracelefttp'),
    ('︷', 'braceleftvertical'),
    ('}', 'braceright'),
    ('', 'bracerightbt'),
    ('', 'bracerightmid'),
    ('｝', 'bracerightmonospace'),
    ('﹜', 'bracerightsmall'),
    ('', 'bracerighttp'),
    ('︸', 'bracerightvertical'),
    ('[', 'bracketleft'),
    ('', 'bracketleftbt'),
    ('', 'bracketleftex'),
    ('［', 'bracketleftmonospace'),
    ('', 'bracketlefttp'),
    (']', 'bracketright'),
    ('', 'bracketrightbt'),
    ('', 'bracketrightex'),
    ('］', 'bracketrightmonospace'),
    ('', 'bracketrighttp'),
    ('˘', 'breve'),
    ('̮', 'brevebelowcmb'),
    ('̆', 'brevecmb'),
    ('̯', 'breveinvertedbelowcmb'),
    ('̑', 'breveinvertedcmb'),
    ('͡', 'breveinverteddoublecmb'),
    ('̪', 'bridgebelowcmb'),
    ('̺', 'bridgeinvertedbelowcmb'),
    ('¦', 'brokenbar'),
    ('ƀ', 'bstroke'),
    ('', 'bsuperior'),
    ('ƃ', 'btopbar'),
    ('ぶ', 'buhiragana'),
    ('ブ', 'bukatakana'),
    ('•', 'bullet'),
    ('◘', 'bulletinverse'),
    ('∙', 'bulletoperator'),
    ('◎', 'bullseye'),
    ('c', 'c'),
    ('ծ', 'caarmenian'),
    ('চ', 'cabengali'),
    ('ć', 'cacute'),
    ('च', 'cadeva'),
    ('ચ', 'cagujarati'),
    ('ਚ', 'cagurmukhi'),
    ('㎈', 'calsquare'),
    ('ঁ', 'candrabindubengali'),
    ('̐', 'candrabinducmb'),
    ('ँ', 'candrabindudeva'),
    ('ઁ', 'candrabindugujarati'),
    ('⇪', 'capslock'),
    ('℅', 'careof'),
    ('ˇ', 'caron'),
    ('̬', 'caronbelowcmb'),
    ('̌', 'caroncmb'),
    ('↵', 'carriagereturn'),
    ('ㄘ', 'cbopomofo'),
    ('č', 'ccaron'),
    ('ç', 'ccedilla'),
    ('ḉ', 'ccedillaacute'),
    ('ⓒ', 'ccircle'),
    ('ĉ', 'ccircumflex'),
    ('ɕ', 'ccurl'),
    ('ċ', 'cdot'),
    ('ċ', 'cdotaccent'),
    ('㏅', 'cdsquare'),
    ('¸', 'cedilla'),
    ('̧', 'cedillacmb'),
    ('¢', 'cent'),
    ('℃', 'centigrade'),
    ('', 'centinferior'),
    ('￠', 'centmonospace'),
    ('', 'centoldstyle'),
    ('', 'centsuperior'),
    ('չ', 'chaarmenian'),
    ('ছ', 'chabengali'),
    ('छ', 'chadeva'),
    ('છ', 'chagujarati'),
    ('ਛ', 'chagurmukhi'),
    ('ㄔ', 'chbopomofo'),
    ('ҽ', 'cheabkhasiancyrillic'),
    ('✓', 'checkmark'),
    ('ч', 'checyrillic'),
    ('ҿ', 'chedescenderabkhasiancyrillic'),
    ('ҷ', 'chedescendercyrillic'),
    ('ӵ', 'chedieresiscyrillic'),
    ('ճ', 'cheharmenian'),
    ('ӌ', 'chekhakassiancyrillic'),
    ('ҹ', 'cheverticalstrokecyrillic'),
    ('χ', 'chi'),
    ('㉷', 'chieuchacirclekorean'),
    ('㈗', 'chieuchaparenkorean'),
    ('㉩', 'chieuchcirclekorean'),
    ('ㅊ', 'chieuchkorean'),
    ('㈉', 'chieuchparenkorean'),
    ('ช', 'chochangthai'),
    ('จ', 'chochanthai'),
    ('ฉ', 'chochingthai'),
    ('ฌ', 'chochoethai'),
    ('ƈ', 'chook'),
    ('㉶', 'cieucacirclekorean'),
    ('㈖', 'cieucaparenkorean'),
    ('㉨', 'cieuccirclekorean'),
    ('ㅈ', 'cieuckorean'),
    ('㈈', 'cieucparenkorean'),
    ('㈜', 'cieucuparenkorean'),
    ('○', 'circle'),
    ('⊗', 'circlemultiply'),
    ('⊙', 'circleot'),
    ('⊕', 'circleplus'),
    ('〶', 'circlepostalmark'),
    ('◐', 'circlewithlefthalfblack'),
    ('◑', 'circlewithrighthalfblack'),
    ('ˆ', 'circumflex'),
    ('̭', 'circumflexbelowcmb'),
    ('̂', 'circumflexcmb'),
    ('⌧', 'clear'),
    ('ǂ', 'clickalveolar'),
    ('ǀ', 'clickdental'),
    ('ǁ', 'clicklateral'),
    ('ǃ', 'clickretroflex'),
    ('♣', 'club'),
    ('♣', 'clubsuitblack'),
    ('♧', 'clubsuitwhite'),
    ('㎤', 'cmcubedsquare'),
    ('ｃ', 'cmonospace'),
    ('㎠', 'cmsquaredsquare'),
    ('ց', 'coarmenian'),
    (':', 'colon'),
    ('₡', 'colonmonetary'),
    ('：', 'colonmonospace'),
    ('₡', 'colonsign'),
    ('﹕', 'colonsmall'),
    ('ˑ', 'colontriangularhalfmod'),
    ('ː', 'colontriangularmod'),
    (',', 'comma'),
    ('̓', 'commaabovecmb'),
    ('̕', 'commaaboverightcmb'),
    ('', 'commaaccent'),
    ('،', 'commaarabic'),
    ('՝', 'commaarmenian'),
    ('', 'commainferior'),
    ('，', 'commamonospace'),
    ('̔', 'commareversedabovecmb'),
    ('ʽ', 'commareversedmod'),
    ('﹐', 'commasmall'),
    ('', 'commasuperior'),
    ('̒', 'commaturnedabovecmb'),
    ('ʻ', 'commaturnedmod'),
    ('☼', 'compass'),
    ('≅', 'congruent'),
    ('∮', 'contourintegral'),
    ('⌃', 'control'),
    ('©', 'copyright'),
    ('', 'copyrightsans'),
    ('', 'copyrightserif'),
    ('「', 'cornerbracketleft'),
    ('｢', 'cornerbracketlefthalfwidth'),
    ('﹁', 'cornerbracketleftvertical'),
    ('」', 'cornerbracketright'),
    ('｣', 'cornerbracketrighthalfwidth'),
    ('﹂', 'cornerbracketrightvertical'),
    ('㍿', 'corporationsquare'),
    ('㏇', 'cosquare'),
    ('㏆', 'coverkgsquare'),
    ('⒞', 'cparen'),
    ('₢', 'cruzeiro'),
    ('ʗ', 'cstretched'),
    ('⋏', 'curlyand'),
    ('⋎', 'curlyor'),
    ('¤', 'currency'),
    ('', 'cyrBreve'),
    ('', 'cyrFlex'),
    ('', 'cyrbreve'),
    ('', 'cyrflex'),
    ('d', 'd'),
    ('դ', 'daarmenian'),
    ('দ', 'dabengali'),
    ('ض', 'dadarabic'),
    ('द', 'dadeva'),
    ('ﺾ', 'dadfinalarabic'),
    ('ﺿ', 'dadinitialarabic'),
    ('ﻀ', 'dadmedialarabic'),
    ('ּ', 'dagesh'),
    ('ּ', 'dageshhebrew'),
    ('†', 'dagger'),
    ('‡', 'daggerdbl'),
    ('દ', 'dagujarati'),
    ('ਦ', 'dagurmukhi'),
    ('だ', 'dahiragana'),
    ('ダ', 'dakatakana'),
    ('د', 'dalarabic'),
    ('ד', 'dalet'),
    ('דּ', 'daletdagesh'),
    ('דּ', 'daletdageshhebrew'),
]


@pytest.mark.parametrize('value, expected', values_to_test)
def test_glyph_name(value, expected):
    assert expected in a.char_to_glyph_names(value)


