"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProwlerAudit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// import { Stack, Duration, RemovalPolicy, CustomResource } from 'aws-cdk-lib';
// eslint-disable-next-line no-duplicate-imports
// import { aws_iam as iam, aws_logs as logs, aws_s3 as s3, aws_codebuild as codebuild, aws_lambda as lambda, custom_resources as cr } from 'aws-cdk-lib';
// import { Construct } from 'constructs';
const codebuild = require("@aws-cdk/aws-codebuild");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
// eslint-disable-next-line import/no-extraneous-dependencies
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const statement = require("cdk-iam-floyd");
/**
 * Creates a CodeBuild project to audit an AWS account with Prowler and stores the html report in a S3 bucket.
 *
 * This will run onece at the beginning and on a schedule afterwards. Partial contribution from https://github.com/stevecjones
 *
 * @stability stable
 */
class ProwlerAudit extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(parent, id, props) {
        var _b, _c, _d, _e, _f;
        super(parent, id);
        // defaults
        this.serviceName = (props === null || props === void 0 ? void 0 : props.serviceName) ? props.serviceName : 'prowler';
        this.logsRetentionInDays = (props === null || props === void 0 ? void 0 : props.logsRetentionInDays) ? props.logsRetentionInDays : logs.RetentionDays.THREE_DAYS;
        this.enableScheduler = (props === null || props === void 0 ? void 0 : props.enableScheduler) ? props.enableScheduler : false;
        this.prowlerScheduler = (props === null || props === void 0 ? void 0 : props.prowlerScheduler) ? props.prowlerScheduler : 'cron(0 22 * * ? *)';
        this.prowlerOptions = (props === null || props === void 0 ? void 0 : props.prowlerOptions) ? props.prowlerOptions : '-M text,junit-xml,html,csv,json';
        this.prowlerVersion = (props === null || props === void 0 ? void 0 : props.prowlerVersion) ? props.prowlerVersion : '2.5.0';
        const reportBucket = (_b = props === null || props === void 0 ? void 0 : props.reportBucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ReportBucket', {
            //bucketName: `${'123456'}-prowler-reports`,
            autoDeleteObjects: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const reportGroup = new codebuild.ReportGroup(this, 'reportGroup', { /**reportGroupName: 'testReportGroup', */ removalPolicy: core_1.RemovalPolicy.DESTROY });
        const preBuildCommands = [];
        if (!!(props === null || props === void 0 ? void 0 : props.allowlist)) {
            const prowlerFilename = 'allowlist.txt';
            if (props.allowlist.isZipArchive) {
                preBuildCommands.push(`aws s3 cp ${props.allowlist.s3ObjectUrl} .`);
                preBuildCommands.push(`unzip ${props.allowlist.s3ObjectKey} -d prowler`);
            }
            else {
                preBuildCommands.push(`aws s3 cp ${props.allowlist.s3ObjectUrl} prowler/${prowlerFilename}`);
            }
            this.prowlerOptions = this.prowlerOptions + ` -w ${prowlerFilename}`;
        }
        const prowlerBuild = this.codebuildProject = new codebuild.Project(this, 'prowlerBuild', {
            description: 'Run Prowler assessment',
            timeout: core_1.Duration.hours(5),
            environment: {
                environmentVariables: {
                    BUCKET_REPORT: { value: reportBucket.bucketName || '' },
                    BUCKET_PREFIX: { value: (_c = props === null || props === void 0 ? void 0 : props.reportBucketPrefix) !== null && _c !== void 0 ? _c : '' },
                    ADDITIONAL_S3_ARGS: { value: (_d = props === null || props === void 0 ? void 0 : props.additionalS3CopyArgs) !== null && _d !== void 0 ? _d : '' },
                    PROWLER_OPTIONS: { value: this.prowlerOptions || '' },
                },
                buildImage: codebuild.LinuxBuildImage.fromCodeBuildImageId('aws/codebuild/amazonlinux2-x86_64-standard:3.0'),
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        'runtime-versions': {
                            python: 3.8,
                        },
                        'commands': [
                            'echo "Installing Prowler and dependencies..."',
                            'pip3 install detect-secrets',
                            'yum -y install jq',
                            'curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip"',
                            'unzip awscliv2.zip',
                            './aws/install',
                            `git clone -b ${this.prowlerVersion} https://github.com/toniblyx/prowler`,
                        ],
                    },
                    pre_build: {
                        commands: preBuildCommands,
                    },
                    build: {
                        commands: [
                            `echo "Running Prowler as ./prowler ${this.prowlerOptions} && echo OK || echo FAILED"`,
                            'cd prowler',
                            `./prowler ${this.prowlerOptions} && echo OK || echo FAILED`,
                        ],
                    },
                    post_build: {
                        commands: [
                            'echo "Uploading reports to S3..." ',
                            'aws s3 cp --sse AES256 output/ s3://$BUCKET_REPORT/$BUCKET_PREFIX --recursive $ADDITIONAL_S3_ARGS',
                            'echo "Done!"',
                        ],
                    },
                },
                reports: {
                    [reportGroup.reportGroupName]: {
                        'files': ['**/*'],
                        'base-directory': 'prowler/junit-reports',
                        'file-format': 'JunitXml',
                    },
                },
            }),
        });
        if (!!(props === null || props === void 0 ? void 0 : props.allowlist)) {
            props.allowlist.bucket.grantRead(prowlerBuild);
        }
        (_e = prowlerBuild.role) === null || _e === void 0 ? void 0 : _e.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('SecurityAudit'));
        (_f = prowlerBuild.role) === null || _f === void 0 ? void 0 : _f.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('job-function/ViewOnlyAccess'));
        // prowlerBuild.addToRolePolicy(new statement.Dax().allow().to());
        prowlerBuild.addToRolePolicy(new statement.Ds().allow().toListAuthorizedApplications());
        prowlerBuild.addToRolePolicy(new statement.Ec2().allow().toGetEbsEncryptionByDefault());
        prowlerBuild.addToRolePolicy(new statement.Ecr().allow().toDescribeImageScanFindings().toDescribeImages().toDescribeRegistry());
        prowlerBuild.addToRolePolicy(new statement.Tag().allow().toGetTagKeys());
        prowlerBuild.addToRolePolicy(new statement.Lambda().allow().toGetFunction());
        prowlerBuild.addToRolePolicy(new statement.Glue().allow().toSearchTables().toGetConnections());
        prowlerBuild.addToRolePolicy(new statement.Apigateway().allow().toGET());
        prowlerBuild.addToRolePolicy(new iam.PolicyStatement({ actions: ['support:Describe*'], resources: ['*'] }));
        reportBucket.grantPut(prowlerBuild);
        const myRole = new iam.Role(this, 'MyRole', { assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com') });
        const prowlerStartBuildLambda = new lambda.Function(this, 'Lambda', {
            runtime: lambda.Runtime.PYTHON_3_6,
            timeout: core_1.Duration.seconds(120),
            handler: 'index.lambda_handler',
            code: lambda.Code.fromInline(`import boto3
import cfnresponse
from botocore.exceptions import ClientError
def lambda_handler(event,context):
  props = event['ResourceProperties']
  codebuild_client = boto3.client('codebuild')
  if (event['RequestType'] == 'Create' or event['RequestType'] == 'Update'):
    try:
        response = codebuild_client.start_build(projectName=props['Build'])
        print(response)
        print("Respond: SUCCESS")
        cfnresponse.send(event, context, cfnresponse.SUCCESS, {})
    except Exception as ex:
        print(ex.response['Error']['Message'])
        cfnresponse.send(event, context, cfnresponse.FAILED, ex.response)
      `),
        });
        prowlerStartBuildLambda.addToRolePolicy(new statement.Codebuild().allow().toStartBuild()); // onResource project ...
        const myProvider = new cr.Provider(this, 'MyProvider', {
            onEventHandler: prowlerStartBuildLambda,
            logRetention: this.logsRetentionInDays,
            role: myRole,
        });
        new core_1.CustomResource(this, 'Resource1', {
            serviceToken: myProvider.serviceToken,
            properties: {
                Build: prowlerBuild.projectName,
                RERUN_PROWLER: Boolean(this.node.tryGetContext('reRunProwler')) ? Date.now().toString() : '',
            },
        });
        if (this.enableScheduler) {
            const prowlerSchedulerLambda = new lambda.Function(this, 'ScheduleLambda', {
                runtime: lambda.Runtime.PYTHON_3_6,
                timeout: core_1.Duration.seconds(120),
                handler: 'index.lambda_handler',
                environment: {
                    buildName: prowlerBuild.projectName,
                },
                code: lambda.Code.fromInline(`import boto3
        import os
        def lambda_handler(event,context):
          codebuild_client = boto3.client('codebuild')
          print("Running Prowler scheduled!: " + os.environ['buildName'])
          project_name = os.environ['buildName']
          response = codebuild_client.start_build(projectName=project_name)
          print(response)
          print("Respond: SUCCESS")
        `),
            });
            new events.Rule(this, 'Rule', {
                description: 'A schedule for the Lambda function that triggers Prowler in CodeBuild.',
                targets: [new targets.LambdaFunction(prowlerSchedulerLambda)],
                schedule: events.Schedule.expression(this.prowlerScheduler || ''),
            });
        }
    }
}
exports.ProwlerAudit = ProwlerAudit;
_a = JSII_RTTI_SYMBOL_1;
ProwlerAudit[_a] = { fqn: "cdk-prowler.ProwlerAudit", version: "1.124.0" };
//# sourceMappingURL=data:application/json;base64,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