#
# A MeqTrees script to rotate or de-rotate the effects of Rotation
# Measure due to the Ionosphere from synhesis telescope data in 
# Measurement Set format

#% $Id$
#
#
# Copyright (C) 2002-2019
# The MeqTree Foundation &
# ASTRON (Netherlands Foundation for Research in Astronomy)
# P.O.Box 2, 7990 AA Dwingeloo, The Netherlands
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see <http://www.gnu.org/licenses/>,
# or write to the Free Software Foundation, Inc.,
# 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

#  (c) 2019.                             (c) 2019.
#  National Research Council             Conseil national de recherches
#  Ottawa, Canada, K1A 0R6               Ottawa, Canada, K1A 0R6
#
#  This software is free software;       Ce logiciel est libre, vous
#  you can redistribute it and/or        pouvez le redistribuer et/ou le
#  modify it under the terms of          modifier selon les termes de la
#  the GNU General Public License        Licence Publique Generale GNU
#  as published by the Free              publiee par la Free Software
#  Software Foundation; either           Foundation (version 3 ou bien
#  version 2 of the License, or          toute autre version ulterieure
#  (at your option) any later            choisie par vous).
#  version.
#
#  This software is distributed in       Ce logiciel est distribue car
#  the hope that it will be              potentiellement utile, mais
#  useful, but WITHOUT ANY               SANS AUCUNE GARANTIE, ni
#  WARRANTY; without even the            explicite ni implicite, y
#  implied warranty of                   compris les garanties de
#  MERCHANTABILITY or FITNESS FOR        commercialisation ou
#  A PARTICULAR PURPOSE.  See the        d'adaptation dans un but
#  GNU General Public License for        specifique. Reportez-vous a la
#  more details.                         Licence Publique Generale GNU
#                                        pour plus de details.
#
#  You should have received a copy       Vous devez avoir recu une copie
#  of the GNU General Public             de la Licence Publique Generale
#  License along with this               GNU en meme temps que ce
#  software; if not, contact the         logiciel ; si ce n'est pas le
#  Free Software Foundation, Inc.        cas, communiquez avec la Free
#  at http://www.fsf.org.                Software Foundation, Inc. au
#                                                http://www.fsf.org.
#
#  email:                                courriel:
#  business@hia-iha.nrc-cnrc.gc.ca       business@hia-iha.nrc-cnrc.gc.ca
#
#  National Research Council             Conseil national de recherches
#      Canada                               Canada
#  Herzberg Institute of Astrophysics    Institut Herzberg d'astrophysique
#  5071 West Saanich Rd.                 5071 West Saanich Rd.
#  Victoria BC V9E 2E7                   Victoria BC V9E 2E7
#  CANADA                                        CANADA

#
# standard stuff 
from Timba.TDL import *
from Timba.Meq import meq
from Timba import pynode
from Timba import mequtils
from Timba.array import *
import os
import random

import Meow

from Meow import Bookmarks,Context
import Meow.StdTrees

from string import split, strip
import Kittens.utils


Settings.forest_state.cache_policy = 100

# MS options first
mssel = Context.mssel = Meow.MSUtils.MSSelector(has_input=True,read_flags=True,has_output=True,tile_sizes=[10,100,200]);
# MS compile-time options
TDLCompileOptions(*mssel.compile_options());

# The script will read in RM files generated by either Maaijke Mevius' RMextract
# module or the ALBUS ionosphere module. The RM in these files is the ionosphere
# RM as seen  by the ground-based observer. So this script will rotate data
# in the opposite direction

# get name of RMextract or ALBUS file of time variable ionosphere RMs
# the file should have the extension 'iono' at the end
TDLCompileOption("ionosphere_filename","Ionosphere RM file",TDLFileSelect("*iono"), doc= """
<P> ASCII text file produced by RMextract or ALBUS ionosphere programs. The file name should end with 'iono'.</P> 
 """)

# if the user wants to corrupt model data with the actual ionosphere effects 
# then switch this flag to True  (corrupt_RM = True)
TDLCompileOption("corrupt_RM","Corrupt model visibilities with ionosphere RM data",False,doc="""
  <P>If enabled, this option can be used to corrupt model data so as to agree with actual observed data. Otherwise actual data will be de-rotated to remove ionospheric RM effects from the data.</P>
  """);

# MS run-time options
TDLRuntimeMenu("Data selection & flag handling",*mssel.runtime_options());

def _define_forest(ns):
  # setup contexts from MS
  mssel.setup_observation_context(ns);
  array = Meow.Context.array;
  observation = Meow.Context.observation;

# pass in the name of the ionosphere data file to the PYNode that will get the
# ionosphere rotation angles as a function of time and frequency
  pa = ns.rm_angle << Meq.PyNode(class_name="PyGetRMAngle",module_name="PYRMAngle", albus_filename=ionosphere_filename)

  cospa = ns << Meq.Cos(pa);
  sinpa = ns << Meq.Sin(pa);
  # set up REVERSE rotation matrix to take observed raw uncorrected data
  # and rotate back to reference frame above the ionosphere
  if not Context.observation.circular():
    if corrupt_RM:
      ns.P << Meq.Matrix22(cospa,Meq.Negate(sinpa),sinpa,cospa);
    else:
      ns.P << Meq.Matrix22(cospa,sinpa,Meq.Negate(sinpa),cospa);
  else:
    if corrupt_RM:
      ns.P << Meq.Matrix22(cospa + sinpa*1j,0.0,0.0,cospa - sinpa*1j);
    else:
      ns.P << Meq.Matrix22(cospa - sinpa*1j,0.0,0.0,cospa + sinpa*1j);
  ns.Pt<< Meq.ConjTranspose(ns.P);

  # make spigot nodes
  spigots = spigots0 = array.spigots(corr=mssel.get_corr_index());
  
  for p,q in array.ifrs():
    ns.output(p,q) << Meq.MatrixMultiply(ns.P,spigots(p,q),ns.Pt);
  outputs = ns.output

  Bookmarks.make_node_folder("Input visibilities by baseline",
    [ spigots(p,q) for p,q in array.ifrs() ],sorted=True,ncol=2,nrow=2);

  Bookmarks.make_node_folder("Output visibilities by baseline",
    [ outputs(p,q) for p,q in array.ifrs() ],sorted=True,ncol=2,nrow=2);
 
  # ...and an inspector for them
  # how to I get this to work properly with VisDataMux so outputs are written
  # out to the MS ??
# Meow.StdTrees.vis_inspector(ns.inspector('output'),outputs,vells_label=Context.correlations,
#                             bookmark="Inspect output visibilities");
# ns.inspectors << Meq.ReqMux(ns.inspector('output'));
# ns.VisDataMux << Meq.VisDataMux(post=ns.inspectors);


  # make sinks and vdm
  Meow.StdTrees.make_sinks(ns,outputs,spigots=spigots)


def _tdl_job_Process_MS (mqs,parent,**kw):
  req = mssel.create_io_request();
  mqs.execute('VisDataMux',req,wait=False);

