"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutoScaler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
class AutoScaler extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const launchTemplate = this.getLT(props.templateProps, props.asgName);
        this.loadBalancerProperties = this.getTG(props.networkProps, props.templateProps.vpc.vpcName, props.appName);
        new aws_autoscaling_1.CfnAutoScalingGroup(this, props.asgName, {
            maxSize: props.maxSize,
            minSize: props.minSize,
            autoScalingGroupName: props.asgName,
            launchTemplate: {
                version: launchTemplate.versionNumber,
                launchTemplateId: launchTemplate.launchTemplateId,
                launchTemplateName: launchTemplate.launchTemplateName,
            },
            targetGroupArns: this.loadBalancerProperties.map((lb) => { return lb.targetGroupArn; }),
            tags: props.tags,
            availabilityZones: this.getZones(props.subnets),
            vpcZoneIdentifier: props.subnets,
            healthCheckGracePeriod: 300,
        });
    }
    getVPC(props) {
        const stackVPC = aws_ec2_1.Vpc.fromLookup(this, props.vpcName, {
            isDefault: false,
            vpcId: props.vpcName,
        });
        return stackVPC;
    }
    getRole(props, asgName) {
        var _b;
        if (props.type == 'existing') {
            const role = aws_iam_1.Role.fromRoleArn(this, asgName + '-stackRole', props.roleArn);
            return role;
        }
        else {
            const role = new aws_iam_1.Role(this, asgName + '-stackRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                roleName: asgName + '-role',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/service-role/AmazonEC2SpotFleetTaggingRole',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/ReadOnlyAccess',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/AmazonEC2FullAccess',
            });
            role.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: ['*'],
                actions: [
                    'iam:ListUsers',
                    'iam:GetGroup',
                ],
            }));
            role.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: ['*'],
                actions: [
                    'ec2:DescribeTags',
                ],
            }));
            role.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: ['*'],
                actions: [
                    'iam:ListUsers',
                    'iam:GetGroup',
                ],
                sid: 'VisualEditor0',
            }));
            (_b = props.additionalPolicies) === null || _b === void 0 ? void 0 : _b.forEach(policyDoc => {
                role.addToPolicy(aws_iam_1.PolicyStatement.fromJson(policyDoc));
            });
            return role;
        }
    }
    getSG(props, vpc, asgName) {
        var _b, _c, _d;
        if (props.type == 'existing') {
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(this, 'stack-sg', props.sgGroupId);
            return securityGroup;
        }
        else {
            const sgProps = {
                securityGroupName: (_b = props.securityGroupName) !== null && _b !== void 0 ? _b : 'stack-sg-group',
                vpc: vpc,
                allowAllOutbound: (_c = props.allowAllOutbound) !== null && _c !== void 0 ? _c : true,
                disableInlineRules: (_d = props.disableInlineRules) !== null && _d !== void 0 ? _d : false,
            };
            const securityGroup = new aws_ec2_1.SecurityGroup(this, asgName + '-stack-sg', sgProps);
            props.ingressRules.forEach(ingress => {
                var _b;
                const ingressSG = aws_ec2_1.SecurityGroup.fromSecurityGroupId(this, ingress.sourceSG + ingress.port.toString(), ingress.sourceSG);
                securityGroup.connections.allowFrom(ingressSG, aws_ec2_1.Port.tcp(ingress.port), (_b = ingress.description) !== null && _b !== void 0 ? _b : 'Application port');
            });
            return securityGroup;
        }
    }
    getBD(props) {
        const bd = {
            deviceName: props.name,
            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(props.size, {
                volumeType: props.type,
                deleteOnTermination: true,
            }),
        };
        return bd;
    }
    getLT(props, asgName) {
        if (props.type == 'existing') {
            const launchTemplate = aws_ec2_1.LaunchTemplate.fromLaunchTemplateAttributes(this, props.templateName, props.existingAttributes);
            return launchTemplate;
        }
        else {
            const launchTemplate = new aws_ec2_1.LaunchTemplate(this, props.templateName, {
                launchTemplateName: props.templateName,
                instanceType: new aws_ec2_1.InstanceType(props.instanceType),
                machineImage: aws_ec2_1.MachineImage.lookup({
                    name: props.amiImageId,
                }),
                securityGroup: this.getSG(props.securityGroup, this.getVPC(props.vpc), asgName),
                role: this.getRole(props.role, asgName),
                detailedMonitoring: false,
                blockDevices: [this.getBD(props.blockDevice)],
                keyName: props.sshKey,
            });
            return launchTemplate;
        }
    }
    getTG(props, vpcId, appName) {
        let lbProps = [];
        props.forEach(t => {
            var _b;
            const tg = new aws_elasticloadbalancingv2_1.CfnTargetGroup(this, appName + t.port.toString(), {
                name: appName + ((_b = t.port) === null || _b === void 0 ? void 0 : _b.toString()),
                healthCheckEnabled: true,
                healthCheckPath: t.healthCheckPath,
                ...((t.protocol == 'GRPC') ? { protocol: 'HTTP' } : { protocol: t.protocol }),
                ...((t.protocol == 'GRPC') ? { protocolVersion: 'GRPC' } : {}),
                healthCheckTimeoutSeconds: 30,
                healthCheckPort: String(t.port),
                port: t.port,
                vpcId: vpcId,
            });
            lbProps.push({
                appName: appName,
                hostHeader: t.host,
                lbArn: t.lbArn,
                sslEnabled: t.sslEnabled,
                targetGroupArn: tg.ref,
            });
        });
        return lbProps;
    }
    getZones(subnets) {
        var availabilityZones = [];
        subnets.forEach(subnet => {
            const net = aws_ec2_1.Subnet.fromSubnetAttributes(this, subnet, {
                availabilityZone: 'dummy',
                subnetId: subnet,
            });
            availabilityZones.push(net.availabilityZone);
        });
        return availabilityZones;
    }
}
exports.AutoScaler = AutoScaler;
_a = JSII_RTTI_SYMBOL_1;
AutoScaler[_a] = { fqn: "@smallcase/aws-cdk-microservice.AutoScaler", version: "0.0.15" };
//# sourceMappingURL=data:application/json;base64,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