"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2020.8.11"
__checksum__ = "c169caad6f07cdb937d8ebb82d41e2387a62e2f48b12bc1a8a8d4c83f8a17d19"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'chrome', b'dev', b'foo', b'gle', b'gmail', b'google', b'hangout', b'insurance', b'meet', b'new', b'page', b'play', b'search', b'youtube'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 5), None, (16, 57), (73, 26), (99, 12), None, (111, 19), (130, 22), (152, 7), (159, 20), (179, 18), None, (197, 22), (219, 45), (264, 7), (271, 9), (280, 36), (316, 10), (326, 10), (336, 21), None, (357, 50), (407, 8), (415, 9), (424, 19), (443, 13), (456, 14), (470, 14), None, None, (484, 29), (513, 16), (529, 35), (564, 14), (578, 24), (602, 9), None, (611, 25), (636, 20), (656, 8), (664, 13), (677, 10), None, (687, 17), (704, 6), (710, 26), (736, 5), (741, 5), (746, 10), (756, 10), (766, 11), (777, 12), (789, 27), None, (816, 11), (827, 11), (838, 7), (845, 29), (874, 18), (892, 27), (919, 46), (965, 25), (990, 16), (1006, 8), (1014, 5), (1019, 22), (1041, 18), None, (1059, 36), (1095, 15), (1110, 8), (1118, 5), None, (1123, 5), (1128, 16), (1144, 14), (1158, 18), None, (1176, 14), (1190, 18), (1208, 48), (1256, 19), (1275, 5), (1280, 46), (1326, 14), (1340, 14), (1354, 20), None, (1374, 10), (1384, 13), (1397, 10), (1407, 19), None, (1426, 13), (1439, 19), (1458, 5), (1463, 4), (1467, 22), (1489, 10), (1499, 7), (1506, 14), (1520, 21), (1541, 11), (1552, 10), (1562, 12), (1574, 32), None, (1606, 10), (1616, 14), (1630, 12), (1642, 45), (1687, 15), None, (1702, 11), (1713, 23), (1736, 21), (1757, 26), (1783, 6), (1789, 6), (1795, 7), (1802, 5), (1807, 20), (1827, 23), (1850, 24), (1874, 13), (1887, 15), (1902, 19), (1921, 6), (1927, 61), (1988, 44), (2032, 12), (2044, 23), (2067, 16), (2083, 38), (2121, 6), (2127, 12), (2139, 44), (2183, 6), (2189, 41), (2230, 13), (2243, 23), (2266, 30), (2296, 16), (2312, 8), (2320, 15), (2335, 12), (2347, 19), (2366, 21), (2387, 15), None, (2402, 35), (2437, 21), (2458, 17), (2475, 19), (2494, 26), (2520, 5), (2525, 37), (2562, 30), (2592, 16), (2608, 10), (2618, 17), (2635, 23), (2658, 14), (2672, 17), (2689, 8), (2697, 4), (2701, 7), (2708, 29), (2737, 6), (2743, 18), (2761, 27), (2788, 20), (2808, 17), (2825, 19), (2844, 12), (2856, 40), (2896, 40), (2936, 12), (2948, 48), (2996, 25), (3021, 12), None, (3033, 8), (3041, 20), (3061, 19), (3080, 6), (3086, 23), None, (3109, 23), (3132, 33), (3165, 14), (3179, 12), (3191, 27), None, (3218, 26), (3244, 31), (3275, 50), (3325, 15), (3340, 20), (3360, 15), (3375, 21), (3396, 32), (3428, 24), (3452, 20), (3472, 13), (3485, 60), (3545, 19), (3564, 9), (3573, 12), (3585, 12), (3597, 11), (3608, 10), (3618, 48), (3666, 32), None, (3698, 25), (3723, 12), None, (3735, 8), (3743, 8), (3751, 7), None, (3758, 25), (3783, 17), None, (3800, 21), (3821, 35), (3856, 12), (3868, 10), (3878, 36), (3914, 20), (3934, 22), (3956, 23), (3979, 19), (3998, 12), (4010, 5), (4015, 30), (4045, 24), (4069, 14), (4083, 14), (4097, 47), (4144, 46), None, None, (4190, 51), (4241, 42), None, (4283, 14), None, (4297, 15), (4312, 8), (4320, 21), (4341, 6), (4347, 16), (4363, 17)], [(4380, 6191), (10571, 6575), (17146, 7025), (24171, 5967), (30138, 6287), (36425, 5998), (42423, 6895), (49318, 6224), (55542, 6777), (62319, 6036), (68355, 7050), (75405, 6386), (81791, 6724), (88515, 7066), (95581, 6479), (102060, 6443), (108503, 6983), (115486, 5955), (121441, 6282), (127723, 6517), (134240, 6839), (141079, 6516), (147595, 6819), (154414, 6084), (160498, 6265), (166763, 6588), (173351, 6532), (179883, 6830), (186713, 6311), (193024, 6579), (199603, 6824), (206427, 6497), (212924, 6384), (219308, 7016), (226324, 6090), (232414, 6817), (239231, 6208), (245439, 6981), (252420, 6712), (259132, 6940), (266072, 7448), (273520, 6350), (279870, 6270), (286140, 6124), (292264, 6346), (298610, 6073), (304683, 6275), (310958, 7005), (317963, 6367), (324330, 5712), (330042, 6359), (336401, 6567), (342968, 6480), (349448, 6723), (356171, 6802), (362973, 6744), (369717, 6771), (376488, 5899), (382387, 6812), (389199, 5796), (394995, 6608), (401603, 6319), (407922, 6312), (414234, 6810), (421044, 6658), (427702, 6526), (434228, 6194), (440422, 7039), (447461, 6690), (454151, 6806), (460957, 6473), (467430, 6398), (473828, 5683), (479511, 6937), (486448, 7046), (493494, 7000), (500494, 6126), (506620, 7148), (513768, 7024), (520792, 6017), (526809, 6729), (533538, 5689), (539227, 6352), (545579, 6565), (552144, 6449), (558593, 6402), (564995, 6573), (571568, 6651), (578219, 6743), (584962, 6419), (591381, 7167), (598548, 6022), (604570, 6211), (610781, 6505), (617286, 6531), (623817, 7114), (630931, 6810), (637741, 6419), (644160, 6233), (650393, 6075), (656468, 6214), (662682, 6717), (669399, 6103), (675502, 6448), (681950, 6130), (688080, 6822), (694902, 6614), (701516, 7034), (708550, 7986), (716536, 7117), (723653, 6823), (730476, 6453), (736929, 6291), (743220, 6659), (749879, 6897), (756776, 6494), (763270, 6281), (769551, 6358), (775909, 6348), (782257, 6940), (789197, 6717), (795914, 6850), (802764, 7048), (809812, 6811), (816623, 7729), (824352, 6389), (830741, 5769), (836510, 6895), (843405, 6594), (849999, 7904), (857903, 6959), (864862, 6114), (870976, 6739), (877715, 6820), (884535, 6306), (890841, 6690), (897531, 6145), (903676, 6768), (910444, 6414), (916858, 6508), (923366, 6509), (929875, 7203), (937078, 6213), (943291, 6216), (949507, 6565), (956072, 6566), (962638, 6410), (969048, 6804), (975852, 6143), (981995, 7125), (989120, 6616), (995736, 6679), (1002415, 6823), (1009238, 6443), (1015681, 6540), (1022221, 6444), (1028665, 6283), (1034948, 6346), (1041294, 6224), (1047518, 5757), (1053275, 6152), (1059427, 6548), (1065975, 7168), (1073143, 6091), (1079234, 6638), (1085872, 6884), (1092756, 6423), (1099179, 6072), (1105251, 6832), (1112083, 6485), (1118568, 5923), (1124491, 6493), (1130984, 7727), (1138711, 6017), (1144728, 6195), (1150923, 6644), (1157567, 6226), (1163793, 6662), (1170455, 6316), (1176771, 5953), (1182724, 7358), (1190082, 6689), (1196771, 6370), (1203141, 6952), (1210093, 7295), (1217388, 7242), (1224630, 6151), (1230781, 6941), (1237722, 6297), (1244019, 6597), (1250616, 6713), (1257329, 6072), (1263401, 6777), (1270178, 6933), (1277111, 6553), (1283664, 6581), (1290245, 6357), (1296602, 6506), (1303108, 6759), (1309867, 6303), (1316170, 6656), (1322826, 5886), (1328712, 7109), (1335821, 6876), (1342697, 6620), (1349317, 6880), (1356197, 5706), (1361903, 6700), (1368603, 6434), (1375037, 6702), (1381739, 6760), (1388499, 7116), (1395615, 6549), (1402164, 6680), (1408844, 6746), (1415590, 6317), (1421907, 6461), (1428368, 6465), (1434833, 6467), (1441300, 6333), (1447633, 6438), (1454071, 5914), (1459985, 7494), (1467479, 6656), (1474135, 6264), (1480399, 6676), (1487075, 6611), (1493686, 5799), (1499485, 6658), (1506143, 6531), (1512674, 7611), (1520285, 6489), (1526774, 5933), (1532707, 7014), (1539721, 6350), (1546071, 7045), (1553116, 6096), (1559212, 6219), (1565431, 5825), (1571256, 6635), (1577891, 6492), (1584383, 6815), (1591198, 6159), (1597357, 6545), (1603902, 6482), (1610384, 6981), (1617365, 6317), (1623682, 5769), (1629451, 6522), (1635973, 6176), (1642149, 6675), (1648824, 6840), (1655664, 7069), (1662733, 6298), (1669031, 6265), (1675296, 6650)], [(1681946, 703), (1682649, 625), (1683274, 649), (1683923, 663), (1684586, 523), (1685109, 633), (1685742, 644), (1686386, 836), (1687222, 640), (1687862, 627), (1688489, 536), (1689025, 544), (1689569, 758), (1690327, 866), (1691193, 987), (1692180, 714), (1692894, 1224), (1694118, 606), (1694724, 875), (1695599, 673), (1696272, 745), (1697017, 733), (1697750, 802), (1698552, 731), (1699283, 703), (1699986, 631), (1700617, 955), (1701572, 1087), (1702659, 791), (1703450, 682), (1704132, 922), (1705054, 787), (1705841, 557), (1706398, 691), (1707089, 732), (1707821, 743), (1708564, 619), (1709183, 688), (1709871, 692), (1710563, 1005), (1711568, 683), (1712251, 812), (1713063, 698), (1713761, 689), (1714450, 728), (1715178, 362), (1715540, 786), (1716326, 857), (1717183, 721), (1717904, 542), (1718446, 805), (1719251, 656), (1719907, 763), (1720670, 952), (1721622, 932), (1722554, 509), (1723063, 661), (1723724, 511), (1724235, 578), (1724813, 724), (1725537, 734), (1726271, 776), (1727047, 1008), (1728055, 861), (1728916, 628), (1729544, 692), (1730236, 767), (1731003, 444), (1731447, 561), (1732008, 518), (1732526, 692), (1733218, 850), (1734068, 523), (1734591, 725), (1735316, 634), (1735950, 671), (1736621, 552), (1737173, 696), (1737869, 768), (1738637, 428), (1739065, 711), (1739776, 629), (1740405, 828), (1741233, 623), (1741856, 607), (1742463, 346), (1742809, 597), (1743406, 725), (1744131, 756), (1744887, 688), (1745575, 816), (1746391, 1074), (1747465, 814), (1748279, 773), (1749052, 674), (1749726, 436), (1750162, 971), (1751133, 881), (1752014, 580), (1752594, 615), (1753209, 683), (1753892, 843), (1754735, 839), (1755574, 552), (1756126, 632), (1756758, 740), (1757498, 395), (1757893, 479), (1758372, 954), (1759326, 880), (1760206, 792), (1760998, 774), (1761772, 610), (1762382, 771), (1763153, 659), (1763812, 699), (1764511, 709), (1765220, 433), (1765653, 652), (1766305, 649), (1766954, 914), (1767868, 653), (1768521, 789), (1769310, 404), (1769714, 703), (1770417, 739), (1771156, 835), (1771991, 908), (1772899, 781), (1773680, 904), (1774584, 774), (1775358, 524), (1775882, 757), (1776639, 583), (1777222, 758), (1777980, 732), (1778712, 661), (1779373, 682), (1780055, 622), (1780677, 656), (1781333, 594), (1781927, 691), (1782618, 715), (1783333, 637), (1783970, 455), (1784425, 587), (1785012, 656), (1785668, 573), (1786241, 717), (1786958, 594), (1787552, 761), (1788313, 532), (1788845, 490), (1789335, 656), (1789991, 596), (1790587, 658), (1791245, 639), (1791884, 796), (1792680, 594), (1793274, 609), (1793883, 857), (1794740, 841), (1795581, 543), (1796124, 695), (1796819, 835), (1797654, 632), (1798286, 659), (1798945, 480), (1799425, 609), (1800034, 633), (1800667, 737), (1801404, 598), (1802002, 932), (1802934, 695), (1803629, 794), (1804423, 721), (1805144, 668), (1805812, 586), (1806398, 646), (1807044, 699), (1807743, 1277), (1809020, 514), (1809534, 624), (1810158, 608), (1810766, 962), (1811728, 754), (1812482, 745), (1813227, 546), (1813773, 587), (1814360, 808), (1815168, 553), (1815721, 569), (1816290, 847), (1817137, 650), (1817787, 883), (1818670, 791), (1819461, 668), (1820129, 686), (1820815, 822), (1821637, 620), (1822257, 889), (1823146, 631), (1823777, 762), (1824539, 558), (1825097, 742), (1825839, 449), (1826288, 775), (1827063, 779), (1827842, 648), (1828490, 916), (1829406, 775), (1830181, 790), (1830971, 903), (1831874, 1077), (1832951, 815), (1833766, 618), (1834384, 854), (1835238, 702), (1835940, 483), (1836423, 443), (1836866, 692), (1837558, 762), (1838320, 406), (1838726, 988), (1839714, 488), (1840202, 758), (1840960, 863), (1841823, 712), (1842535, 769), (1843304, 647), (1843951, 788), (1844739, 715), (1845454, 784), (1846238, 608), (1846846, 566), (1847412, 408), (1847820, 651), (1848471, 437), (1848908, 767), (1849675, 834), (1850509, 764), (1851273, 718), (1851991, 621), (1852612, 568), (1853180, 848), (1854028, 474), (1854502, 590), (1855092, 780), (1855872, 494), (1856366, 847), (1857213, 2086), (1859299, 548), (1859847, 627), (1860474, 890), (1861364, 885), (1862249, 510)], [(1862759, 48), None, (1862807, 35), (1862842, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (1862884, 42), None, (1862926, 25), (1862951, 16), None, (1862967, 18), None, None, None, None, (1862985, 26), None, None, None, None, (1863011, 21), (1863032, 25), None, None, (1863057, 26), None, None, None, None, (1863083, 44), (1863127, 21), (1863148, 23), None, None, None, None, (1863171, 48), None, None, None, None, None, (1863219, 31), None, None, None, None, (1863250, 42), None, (1863292, 22), None, (1863314, 21), None, (1863335, 26), (1863361, 42), None, None, (1863403, 77), None, None, None, None, None, (1863480, 21), (1863501, 21), None, None, (1863522, 34), (1863556, 42), None, None, None, (1863598, 25), None, None, (1863623, 21), None, None, None, None, None, (1863644, 24), (1863668, 21), None, None, (1863689, 26), None, (1863715, 18), None, (1863733, 54), None, None, None, None, None, None, (1863787, 26), None, (1863813, 19), None, (1863832, 20), None, None, (1863852, 42), (1863894, 42), (1863936, 17), None, (1863953, 26), None, (1863979, 26), None, None, None, (1864005, 26), (1864031, 20), (1864051, 26), None, (1864077, 42), (1864119, 63), None, None, None, (1864182, 40), (1864222, 48), None, None, None, (1864270, 47), None, None, None, None, None, None, None, (1864317, 42), None, (1864359, 55), None, (1864414, 9), None, (1864423, 21), (1864444, 42), None, None, (1864486, 42), (1864528, 82), None, None, (1864610, 42), None, None, None, None, None, None, None, None, None, (1864652, 42), (1864694, 21), (1864715, 21), None, (1864736, 42), (1864778, 25), None, None, (1864803, 21), (1864824, 42), None, None, (1864866, 21), (1864887, 19), (1864906, 26), None, None, None, (1864932, 21), None, None, (1864953, 38), None, (1864991, 22), (1865013, 21), (1865034, 21), None, None, (1865055, 63), None, (1865118, 21), (1865139, 42), None, (1865181, 17), None, None, None, None, (1865198, 21), (1865219, 21), None, None, (1865240, 21), None, None, (1865261, 21), None, (1865282, 26), None, (1865308, 50), None, None, None, (1865358, 50), (1865408, 26), (1865434, 21), (1865455, 21), (1865476, 19), None, (1865495, 35), (1865530, 26), (1865556, 23), (1865579, 21), (1865600, 42), None, None, None, None, None, None, (1865642, 21), None, None, None, (1865663, 21), None, None, (1865684, 90), None, (1865774, 239), (1866013, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path), "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
