# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['santoku',
 'santoku.aws',
 'santoku.aws.test',
 'santoku.google',
 'santoku.google.test',
 'santoku.salesforce',
 'santoku.salesforce.test',
 'santoku.slack',
 'santoku.slack.test',
 'santoku.sql',
 'santoku.sql.test',
 'santoku.utils',
 'santoku.utils.test']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.16.50,<2.0.0',
 'jsonschema>=3.2.0,<4.0.0',
 'mysql-connector-python>=8.0.5,<9.0.0',
 'pandas-gbq>=0.17.1,<0.18.0',
 'pandas>=1.2.4,<2.0.0',
 'requests>=2.27.1,<3.0.0',
 'slackclient>=2.5.0,<3.0.0',
 'tldextract>=3.1.0,<4.0.0']

setup_kwargs = {
    'name': 'santoku',
    'version': '220411.35',
    'description': 'Custom Python wrapper around many third party APIs, including AWS, BigQuery, Slack and Salesforce.',
    'long_description': '[![PyPI version](https://badge.fury.io/py/santoku.svg)](https://badge.fury.io/py/santoku)\n\n[![Install deps, Test & Release](https://github.com/wiris/santoku/actions/workflows/cd.yml/badge.svg)](https://github.com/wiris/santoku/actions/workflows/cd.yml)\n\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n# What is Santoku?\n\nSantoku is a toolkit written in Python for interacting with AWS, Google Cloud platform, Salesforce and Slack.\n\nThe purpose of Santoku is to have the interactions with all the external services collected in a single package. The package contains wrappers around the respective APIs and high level methods for the most common patterns in order to simplify the interaction with those services, whether by being shorter to type, more descriptive, more specific to our needs or simply easier to read for developers.\n\n## Quickstart\n\n### Installation\n\nIf you have the wheel, you can install it with:\n\n```bash\npip install --upgrade --force-reinstall dist/santoku-*.whl\n```\n\nRun the following command to install it from PyPI:\n\n```bash\npip install santoku\n```\n\nOr use the following to install it via Poetry\n\n```bash\npoetry add santoku\n```\n\n### How To Use It\n\nYou can use the package as follows:\n\n```python\nfrom santoku.slack.slack_bot_handler import SlackBotHandler\n\nslack_bot = SlackBotHandler.from_aws_secrets_manager("your_secret")\n```\n\n## Content\n\nThe package `santoku` contains several subpackages: `aws`, `google`, `salesforce`, `slack`, `sql`. Each subpackage provides connection to different external services and are formed by a collection of modules, where each module consists of handlers for more specific services. Each handler class has unit tests to ensure the correct behaviour of the methods of these classes.\n\n### AWS\n\nAWS (Amazon Web Services) is a cloud computing platform that provides a set of primitive abstract technical infrastructure and distributed computing building blocks and tools.\n\nThe connection to AWS has been done through the AWS SDK, which in Python is called [boto3](https://github.com/boto/boto3). We provide wrappers of the `boto3` SDK to make easy the operations to interact with different services.\n\nThe use of this subpackage requires having AWS credentials somewhere. We provide flexibility to either keep credentials in AWS credentials/configuration file, set environment variables, or to pass them directly as arguments in the initializer of each handler class. More info on AWS configurations and credentials [here](https://docs.aws.amazon.com/cli/latest/userguide/cli-chap-configure.html).\n\nThe unit tests in this subpackage implement mocks to the AWS services and do not pretended to access or modify the environment of your real account. In order to have safer unit tests for the AWS subpackage, we use moto, a mocking library for most AWS services, which allows our methods to interact with a fully mocked version of the AWS environment via decorators while not needing an actual connection to the internet or a test AWS account.\n\n#### Amazon S3\n\nAmazon Simple Storage Service (Amazon S3) is an object storage service that offers scalability, data availability, security, and performance.\n\nWe provide methods to easily list and delete objects inside buckets; read and write content within S3 objects; upload a dataframe into csv or parket format to a specific location; generate and upload an Amazon Quicksight manifest in S3 in order to create analysis in Amazon Quicksight, and so on.\n\nAn object can be uploaded to S3 with the following:\n\n```python\nfrom santoku.aws.s3_handler import S3Handler\n\ns3_handler = S3Handler()\ns3_handler.put_object(bucket="your_bucket_name", object_key="your_object_key", content="Your object content.")\n```\n\n#### AWS Secrets Manager\n\nAWS Secrets Manager protects secrets needed to access applications, services, and IT resources. The service allows rotating, managing, and retrieving credentials, keys, and other secrets.\n\nWe provide methods to get the content of a previously created secret.\n\n```python\nfrom santoku.aws.secrets_manager_handler import SecretsManagerHandler\n\nsecrets_manager = SecretsManagerHandler()\nsecret_content = secrets_manager.get_secret_value(secret_name="your_secret_name")\n```\n\nWe use this service as our default credential manager. Most classes that require some form of authentication in santoku are provided with alternative class methods that retrieve the credentials directly from Secrets Manager. For example, instead of directly providing credentials to the BigQuery handling class, we simply provide it with the name of the secret where they are stored:\n\n```python\nfrom santoku.google.bigquery import BigQueryHandler\n\nbigquery_handler = BigQueryHandler(\n    type="your_type",\n    project_id="your_project_id"\n    private_key_id="your_private_key_id"\n    private_key="your_private_key"\n    client_email="your_client_email"\n    client_id="your_client_id"\n    auth_uri="your_auth_uri"\n    token_uri="your_token_uri"\n    auth_provider_x509_cert_url="your_auth_provider_x509_cert_url"\n    client_x509_cert_url="your_client_x509_cert_url"\n)\n```\n\nor\n\n```python\nbigquery_handler = BigQueryHandler.from_aws_secrets_manager(\n    secret_name="your_secret_name"\n)\n```\n\n#### Amazon Simple Queue Service\n\nAmazon Simple Queue Service (SQS) is a fully managed message queuing service that supports programmatic sending of messages via web service applications as a way to communicate over the Internet.\n\nWe provide methods to receive, delete, and send single or a batch of messages.\n\n```python\nfrom santoku.aws.sqs_handler import SQSHandler\n\nsqs_handler = SQSHandler()\nentries = [\n    {\n        "Id": "Id1",\n        "MessageBody": "Your message 1",\n    },\n    {\n        "Id": "Id2",\n        "MessageBody": "Your message 1",\n    }\n]\nsqs_handler.send_message_batch(queue_name="your_queue_name", entries=entries)\n```\n\n### Google Cloud Platform\n\nGoogle Cloud Platform a suite of cloud computing services provided by Google that runs on the same Cloud infrastructure that Google uses internally for its end-user products.\n\nThe connection to Google Cloud Platform has been done using the [google-cloud-core](https://googleapis.dev/python/google-api-core/latest/index.html) package.\n\nThe use of this subpackage requires having [Google Cloud Platform credentials](https://cloud.google.com/docs/authentication/production#obtaining_and_providing_service_account_credentials_manually) (in this case, a service account for programmatic access), these can be passed as arguments in the initializer of the handler class directly, or you can store them in AWS Secrets Manager and retrieve them during the initialization using the class method instead.\n\nWe provide a handler that allows doing queries on BigQuery services:\n\n```python\nquery_results = bigquery_handler.get_query_results(query="SELECT * FROM `your_table`")\n```\n\n### Salesforce\n\nSalesforce is a Customer Relationship Management (CRM) platform that gives to the marketing, sales, commerce, and service depertments a single, shared view of every customer.\n\nThe connection to Salesforce has been done using the [Salesforce REST API](https://developer.salesforce.com/docs/atlas.en-us.api_rest.meta/api_rest/quickstart.htm).\n\nThe use of this subpackage requires having Salesforce credentials, these can be passed as arguments in the initializer of the handler class directly, or you can store them in AWS Secrets Manager and retrieve them during the initialization using the class method instead.\n\nThis subpackage provide methods to insert/modify/delete salesforce object records. You can perform operations by doing HTTP requests directly or using methods with higher level of abstraction, which are easier to handle. The lasts ones are just wrappers of the HTTP request method. To obtain records you can perform queries using SOQL.\n\nThe unit tests require valid Salesforce credentials to be executed. The tests are implemented in the way that no new data will remain in the account and no existent data will be modified. However, having Salesforce credentials for sandbox use is recommended.\n\nYou can use the package to perform a request as follows.\n\n```python\nfrom santoku.salesforce.objects_handler import ObjectsHandler\n\nobjects_handler = ObjectsHandler(\n    auth_url="your_auth_url",\n    username="your_username",\n    password="your_password",\n    client_id="your_client_id",\n    client_secret="your_client_secret",\n)\ncontact_payload = {"FirstName": "Alice", "LastName": "Ackerman", "Email": "alice@example.com"}\n\nobjects_handler.do_request(method="POST", path="sobjects/Contact", payload=contact_payload)\n```\n\nor insert a record with\n\n```python\nobjects_handler.insert_record(sobject="Contact", payload=contact_payload)\n```\n\nFinally, you can do a SOQL query with:\n\n```python\nrecords = objects_handler.do_query_with_SOQL("SELECT Id, Name from Contact")\n```\n\n### Slack\n\nSlack is a proprietary business communication platform. A Slack Bot is a nifty way to run code and automate tasks. In Slack, a bot is controlled programmatically via a bot user token that can access one or more of Slack’s APIs.\n\nThe connection to Slack has been done using the [Slack Web API](https://slack.dev/python-slackclient/basic_usage.html).\n\nThe use of this subpackage requires having Slack API Token of a Slack Bot, which can be passed as argument in the initializer of the handler class directly, or you can store it in AWS Secrets Manager and retrieve it during the initialization using the class method instead.\n\nThis subpackage provide methods to send messages to a channel. A message can be sent with:\n\n```python\nfrom santoku.slack.slack_bot_handler import SlackBotHandler\n\nslack_bot_handler = SlackBotHandler(api_token="your_api_token")\nslack_bot_handler.send_message(channel="your_chanel_name", message="Your message.")\n```\n\n### SQL\n\nSQL (Structured Query Language) is a domain-specific language designed for managing data held in a relational database management system (RDBMS). The purpose of this subpackage is to provide connection to different RDBMSs.\n\n#### MySQL\n\nMySQL is an open-source RDBMS. The connection to MySQL has been done using the [MySQL Connector for python](https://dev.mysql.com/doc/connector-python/en/).\n\nThe use of this subpackage requires having MySQL authentication parameters, which can be passed as argument in the initializer of the handler class directly, or you can store it in AWS Secrets Manager and retrieve it during the initialization using the class method instead.\n\nThis subpackage provides methods to do queries and retrieve the results in different forms.\n\n```python\nfrom santoku.sql.mysql_handler import MySQLHandler\n\nmysql_handler = MySQLHandler(user="your_user", password="your_password", host="your_host", database="your_database")\nmysql_handler.get_query_results(query="SELECT * FROM your_table")\n```\n\n## Development\n\n### Project\n\nWe use Poetry to handle this project. Poetry is a tool for dependency management and packaging in Python. It allows you to declare the libraries your project depends on and it will manage (install/update) them for you. More detais in [their documentation](https://python-poetry.org/docs/basic-usage/).\n\nPoetry is already included in the development environment.\n\n#### Dependencies\n\nIf you want to add dependencies to your project, you can specify them in the `tool.poetry.dependencies` section of the `pyproject.toml` file. Also, instead of modifying the `pyproject.toml` file by hand, you can use the add command:\n\n```bash\npoetry add <package_name>\n```\n\nPoetry will automatically find a suitable version constraint and install the package and subdependencies.\n\n**Note:** Remember to commit changes of `poetry.lock` and `pyproject.toml` files after adding a new dependency.\n\n**Note:** You can find more details on how to handle versions of packages [here](https://python-poetry.org/docs/dependency-specification/).\n\n### Environment\n\nWe provide a development environment that uses the Visual Studio Code Remote - Containers extension. This extension lets you use a Docker container in order to have a consistent and easily reproducible development environment.\n\nThe files needed to build the container are located in the `.devcontainer` directory:\n\n* `devcontainer.json` contains a set of configurations, tells VSCode how to access the container and which extensions it should install.\n* `Dockerfile` defines instructions for the building of the container image.\n\nMore info [here](https://code.visualstudio.com/docs/remote/containers-tutorial).\n\n#### Environment Variables\n\nThe containerized environment will automatically set as environment variables the your variables stored in a `.env` file at the top level of the repository. For example, this is required for certain tests that require credentials, which are (of course) not versioned. Be aware that:\n\n* The Docker image building process will **fail** if you do not include a `.env` file at the top level of the repository.\n* If you change the contents of the `.env` file you will need to rebuild the container for the changes to take effect within the environment.\n\n#### Sharing Git credentials with your container\n\nThe containerized environment will automatically forward your local SSH agent if one is running. More info [here](https://code.visualstudio.com/docs/remote/containers#_using-ssh-keys). It works for Windows and Linux.\n\n### Creating a PR\n\nCreate a pull request (PR) to propose and collaborate on changes to the project. These changes MUST BE proposed in a branch, which ensures that the main branch only contains finished and approved work.\n\nBe sure to run tests locally before commiting your changes.\n\n#### Running tests\n\nThe tests are implemented with pytest and there are unit tests for each of the handler modules. Tests in the `aws` subpackage implement mocks to S3 and do not require real credentials, however, the remaining tests in other subpackages do.\n\nTo run the tests just execute `pytest` if you are already inside Poetry virtual environment or `poetry run pytest`.\n\nMoreover, when a PR is created a GitHub Actions CI pipeline (see [`.github/workflows/ci.yml`](./.github/workflows/ci.yml)) is executed. This pipeline is in charge of running tests.\n\n### Release\n\nWheel is automatically created and uploaded to PyPI by the GitHub Actions CD pipeline (see [`.github/workflows/cd.yml`](./.github/workflows/cd.yml)) when the PR is merged in main branch.\n\n\n## Why Santoku?\n\nFrom Wikipedia:\n\n```text\nThe Santoku bōchō (Japanese: 三徳包丁; "three virtues" or "three uses") or Bunka bōchō (文化包丁) is a general-purpose kitchen knife originating in Japan. Its blade is typically between 13 and 20 cm (5 and 8 in) long, and has a flat edge and a sheepsfoot blade that curves down an angle approaching 60 degrees at the point. The term Santoku may refer to the wide variety of ingredients that the knife can handle: meat, fish and vegetables, or to the tasks it can perform: slicing, chopping and dicing, either interpretation indicating a multi-use, general-purpose kitchen knife.\n```\n',
    'author': 'Data Science Team at WIRIS',
    'author_email': 'info@wiris.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/wiris/santoku',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<3.10',
}


setup(**setup_kwargs)
