"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Archiver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The default backup intervall for the git repositories is "every week".
 */
const DEFAULT_SCHEDULE = aws_cdk_lib_1.aws_events.Schedule.expression('cron(0 0 ? * 1 *)');
class Archiver extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.kmsKey = this.createKey();
        this.logGroup = this.createLogGroup();
        this.topic = new aws_cdk_lib_1.aws_sns.Topic(this, 'notifications', {
            displayName: 'archiver-notifications',
            masterKey: this.kmsKey,
        });
        this.bucket = this.createArchiveBucket();
        this.createS3Notifications();
        this.createProjects();
        this.createCfnOutputs();
    }
    /**
     * Set up the S3-related event notifcations.
     *
     * @private
     * @memberof Archiver
     */
    createS3Notifications() {
        if (this.props.notificationEvents) {
            this.props.notificationEvents.forEach((event) => {
                this.bucket.addEventNotification(event, new aws_cdk_lib_1.aws_s3_notifications.SnsDestination(this.topic));
            });
        }
    }
    createCfnOutputs() {
        new aws_cdk_lib_1.CfnOutput(this, 's3-bucket-arn', {
            description: 'ARN of the S3 bucket storing the repositories.',
            value: this.bucket.bucketArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'log-group-arn', {
            description: 'ARN of the Cloudwatch Log group storing the code build logs.',
            value: this.logGroup.logGroupArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'kms-key', {
            description: 'ARN of the KMS key used to encrypt the Cloudwatch logs and the SNS topic.',
            value: this.kmsKey.keyArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'sns-topic-arn', {
            description: 'ARN of the SNS topic.',
            value: this.topic.topicArn,
        });
    }
    createLogGroup() {
        const loggroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'loggroup', {
            encryptionKey: this.kmsKey,
            retention: this.props.retentionDays
                ? this.props.retentionDays
                : aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        loggroup.node.addDependency(this.kmsKey);
        return loggroup;
    }
    /**
     *Create the S3 bucket that will later store the repositories.
     *
     * @private
     * @return {*}
     * @memberof Archiver
     */
    createArchiveBucket() {
        return new aws_cdk_lib_1.aws_s3.Bucket(this, 'destination', {
            blockPublicAccess: aws_cdk_lib_1.aws_s3.BlockPublicAccess.BLOCK_ALL,
            encryption: aws_cdk_lib_1.aws_s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
            lifecycleRules: [
                {
                    expiration: aws_cdk_lib_1.Duration.days(360),
                    transitions: [
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: aws_cdk_lib_1.Duration.days(30),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.GLACIER,
                            transitionAfter: aws_cdk_lib_1.Duration.days(90),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.DEEP_ARCHIVE,
                            transitionAfter: aws_cdk_lib_1.Duration.days(180),
                        },
                    ],
                    noncurrentVersionTransitions: [
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: aws_cdk_lib_1.Duration.days(30),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.GLACIER,
                            transitionAfter: aws_cdk_lib_1.Duration.days(90),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.DEEP_ARCHIVE,
                            transitionAfter: aws_cdk_lib_1.Duration.days(180),
                        },
                    ],
                },
            ],
            publicReadAccess: false,
            versioned: true,
        });
    }
    createKey() {
        const key = new aws_cdk_lib_1.aws_kms.Key(this, 'loggroupKey', {
            description: 'Azure DevOps git repository archiver',
            enableKeyRotation: true,
            pendingWindow: aws_cdk_lib_1.Duration.days(7),
            keyUsage: aws_cdk_lib_1.aws_kms.KeyUsage.ENCRYPT_DECRYPT,
            keySpec: aws_cdk_lib_1.aws_kms.KeySpec.SYMMETRIC_DEFAULT,
        });
        key.grantEncryptDecrypt(new aws_cdk_lib_1.aws_iam.ServicePrincipal(`logs.${aws_cdk_lib_1.Stack.of(this).region}.amazonaws.com`));
        if (this.props.notificationEvents) {
            key.grantEncryptDecrypt(new aws_cdk_lib_1.aws_iam.ServicePrincipal('s3.amazonaws.com'));
        }
        return key;
    }
    /**
     * Creates for each backup configuration a separate CodeBuild project.
     *
     * @private
     * @memberof Archiver
     */
    createProjects() {
        this.props.backupConfigurations.forEach((configuration) => {
            const project = this.createProject(configuration);
            this.bucket.grantReadWrite(project);
            this.createCloudwatchRule(project, configuration);
        });
    }
    createCloudwatchRule(project, element) {
        new aws_cdk_lib_1.aws_events.Rule(this, 'ScheduleRule-' + element.organizationName + '-' + element.projectName, {
            enabled: true,
            schedule: element.schedule ? element.schedule : DEFAULT_SCHEDULE,
            targets: [new aws_cdk_lib_1.aws_events_targets.CodeBuildProject(project)],
            description: 'Trigger for backing up Azure DevOps git repositories of organization ' +
                element.organizationName +
                ' and project ' +
                element.projectName +
                '.',
        });
    }
    /**
     * Create a CodeBuild project.
     *
     * @private
     * @param {BackupConfiguration} element
     * @return {*}
     * @memberof Archiver
     */
    createProject(element) {
        return new aws_cdk_lib_1.aws_codebuild.Project(this, 'archiver-' + element.organizationName + '-' + element.projectName, {
            timeout: aws_cdk_lib_1.Duration.hours(5),
            projectName: 'AzureDevOpsGitBackup' +
                '-' +
                element.organizationName +
                '-' +
                element.projectName,
            description: 'Backup Azure DevOps git repositories to an S3 bucket.',
            checkSecretsInPlainTextEnvVariables: true,
            concurrentBuildLimit: 90,
            environment: {
                environmentVariables: {
                    TOKEN: {
                        value: element.secretArn + ':pat',
                        type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    },
                    ORGANIZATION: { value: element.organizationName },
                    PROJECT: { value: element.projectName },
                },
                buildImage: aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.STANDARD_6_0,
            },
            logging: {
                cloudWatch: {
                    enabled: true,
                    logGroup: this.logGroup,
                },
            },
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject({
                version: 0.2,
                phases: {
                    build: {
                        commands: this.createCommands(element.repositoryNames),
                    },
                },
            }),
        });
    }
    createCommands(repositoryNames) {
        const output = [];
        repositoryNames.forEach((element) => {
            output.push('git clone --mirror "https://${TOKEN}@dev.azure.com/${ORGANIZATION}/${PROJECT}/_git/' + element + '"', 'tar czf ' + element + '.tgz ' + element + '.git', 'aws s3 cp ./' + element + '.tgz ' +
                this.bucket.s3UrlForObject() +
                '/${ORGANIZATION}/${PROJECT}/' + element + '.tgz', 'rm -f ' + element + '.tgz');
        });
        return output;
    }
}
exports.Archiver = Archiver;
_a = JSII_RTTI_SYMBOL_1;
Archiver[_a] = { fqn: "azure-devops-repository-archiver.Archiver", version: "1.5.5" };
//# sourceMappingURL=data:application/json;base64,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