"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Scheduler = exports.BaseOS = void 0;
const cdk = require("@aws-cdk/core");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const info_1 = require("./info");
var BaseOS;
(function (BaseOS) {
    BaseOS["CENTOS_7"] = "centos7";
    BaseOS["RHEL_7"] = "rhel7";
    BaseOS["AMZN2"] = "amazonlinux2";
})(BaseOS = exports.BaseOS || (exports.BaseOS = {}));
class Scheduler extends cdk.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const urlsuffix = cdk.Stack.of(this).urlSuffix;
        const stack = cdk.Stack.of(this);
        const region = cdk.Stack.of(this).region;
        const s3InstallBucket = props.s3InstallBucket;
        const s3InstallFolder = props.s3InstallFolder;
        const baseOs = props.baseOs ? props.baseOs.toString() : BaseOS.AMZN2.toString();
        const clusterId = props.network.clusterId;
        const socaVersion = info_1.SocaInfo.Data.Version;
        const ldapUserName = props.ldapUserName;
        const ldapUserPassword = props.ldapUserPassword;
        const socaInstallAmi = props.customAmi ? ec2.MachineImage.genericLinux({
            [region]: props.customAmi,
        }) : ec2.MachineImage.genericLinux({
            [region]: info_1.RegionMap[region][baseOs],
        });
        const socaInstallAmiId = socaInstallAmi.getImage(this).imageId;
        // const bootscript: string = fs.readFileSync(path.join(__dirname, '../assets/user-data'), 'utf-8');
        const userData = ec2.UserData.forLinux();
        // userData.addCommands(bootscript)
        userData.addCommands(`
export PATH=$PATH:/usr/local/bin
# Deactivate shell to make sure users won't access the cluster if it's not ready
echo '
************* SOCA FIRST TIME CONFIGURATION *************
Hold on, cluster is not ready yet.
Please wait ~30 minutes as SOCA is being installed.
Once cluster is ready to use, this message will be replaced automatically and you will be able to SSH.
*********************************************************' > /etc/nologin

if [ "${baseOs}" == "amazonlinux2" ] || [ "${baseOs}" == "rhel7" ];
  then
    usermod --shell /usr/sbin/nologin ec2-user
fi

if [ "${baseOs}" == "centos7" ];
  then
    usermod --shell /usr/sbin/nologin centos
fi

# Install awscli
if [ "${baseOs}" == "centos7" ] || [ "${baseOs}" == "rhel7" ];
then
  EASY_INSTALL=$(which easy_install-2.7)
  $EASY_INSTALL pip
  PIP=$(which pip2.7)
  $PIP install awscli
fi

# Disable automatic motd update if using ALI
if [ "${baseOs}" == "amazonlinux2" ];
then
  /usr/sbin/update-motd --disable
  rm /etc/cron.d/update-motd
  rm -f /etc/update-motd.d/*
fi

AWS=$(which aws)
echo export "SOCA_BASE_OS=${baseOs}" >> /etc/environment
echo export "SOCA_CONFIGURATION=${clusterId}" >> /etc/environment
echo export "AWS_DEFAULT_REGION=${region} " >> /etc/environment
echo export "SOCA_INSTALL_BUCKET=${s3InstallBucket}" >> /etc/environment
echo export "SOCA_INSTALL_BUCKET_FOLDER=${s3InstallFolder}" >> /etc/environment
echo export "SOCA_VERSION=${socaVersion}" >> /etc/environment
echo export "SOCA_INSTALL_AMI=${socaInstallAmiId}" >> /etc/environment
source /etc/environment

# Tag EBS disks manually as CFN ASG does not support it
AWS_AVAIL_ZONE=$(curl http://169.254.169.254/latest/meta-data/placement/availability-zone)
AWS_REGION=$(echo \"$AWS_AVAIL_ZONE\" | sed "s/[a-z]$//")
AWS_INSTANCE_ID=$(curl http://169.254.169.254/latest/meta-data/instance-id)
EBS_IDS=$(aws ec2 describe-volumes --filters Name=attachment.instance-id,Values="$AWS_INSTANCE_ID" --region $AWS_REGION --query "Volumes[*].[VolumeId]" --out text | tr "\n" " ")
$AWS ec2 create-tags --resources $EBS_IDS --region $AWS_REGION --tags Key=Name,Value="${clusterId} Root Disk" "Key=soca:ClusterId,Value=${clusterId}"

# Tag Network Adapter for the Scheduler
ENI_IDS=$(aws ec2 describe-network-interfaces --filters Name=attachment.instance-id,Values="$AWS_INSTANCE_ID" --region $AWS_REGION --query "NetworkInterfaces[*].[NetworkInterfaceId]" --out text | tr "\n" " ")
$AWS ec2 create-tags --resources $ENI_IDS --region $AWS_REGION --tags Key=Name,Value="${clusterId} Scheduler Network Adapter" "Key=soca:ClusterId,Value=${clusterId}"

echo "@reboot /bin/aws s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/SchedulerPostReboot.sh /root && /bin/bash /root/SchedulerPostReboot.sh ${s3InstallBucket} ${s3InstallFolder} ${ldapUserName} '${ldapUserPassword}' >> /root/PostRebootConfig.log 2>&1" | crontab -
AWS=$(which aws)
$AWS s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/config.cfg /root/
$AWS s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/requirements.txt /root/
$AWS s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/Scheduler.sh /root/
/bin/bash /root/Scheduler.sh ${props.storage.efsDataDns} ${props.storage.efsAppsDns} >> /root/Scheduler.sh.log 2>&1
`);
        const scheduler = new ec2.Instance(this, 'Scheduler', {
            vpc: props.network.vpc,
            instanceType: (_a = props.instanceType) !== null && _a !== void 0 ? _a : new ec2.InstanceType('m5.xlarge'),
            machineImage: socaInstallAmi,
            userData,
            securityGroup: props.schedulerSecurityGroup,
        });
        scheduler.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'pricing:GetProducts',
                'budgets:ViewBudget',
                'ec2:DescribeInstances',
                'ec2:DescribeSubnets',
                'ec2:DescribeSecurityGroups',
                'ec2:DescribeImages',
                'ec2:DescribeInstanceAttribute',
                'ec2:DescribeInstanceTypes',
                'ec2:DescribeReservedInstances',
                'ec2:DescribeSpotInstanceRequests',
                'ec2:DescribeVpcClassicLink',
                'ec2:DescribeVolumes',
                'ec2:DescribePlacementGroups',
                'ec2:DescribeKeyPairs',
                'ec2:DescribeLaunchTemplates',
                'ec2:DescribeLaunchTemplateVersions',
                'ec2:DescribeNetworkInterfaces',
                'ec2:DescribeSpotFleetRequests',
                'ec2:DescribeSpotFleetInstances',
                'fsx:DescribeFileSystems',
                'autoscaling:DescribeAutoScalingGroups',
                'autoscaling:DescribeScalingActivities',
                'autoscaling:DescribeLaunchConfigurations',
                'elasticloadbalancing:DescribeRules',
                'elasticloadbalancing:DescribeListeners',
                'elasticloadbalancing:DescribeTargetGroups',
                'savingsplans:DescribeSavingsPlans',
                'servicequotas:ListServiceQuotas',
            ],
            resources: ['*'],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
                'autoscaling:DeleteAutoScalingGroup',
                'autoscaling:CreateAutoScalingGroup',
                'autoscaling:DetachInstances',
                'ec2:DeleteLaunchTemplate',
                'ec2:CreateLaunchTemplate',
                'fsx:CreateDataRepositoryTask',
            ],
            resources: ['*'],
            conditions: {
                StringLikeIfExists: {
                    'autoscaling:LaunchConfigurationName': props.network.clusterId,
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:CreateTags',
            ],
            resources: [
                stack.formatArn({ service: 'ec2', resource: 'volume' }),
                stack.formatArn({ service: 'ec2', resource: 'network-interface' }),
                stack.formatArn({ service: 'ec2', resource: 'instance' }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'cloudformation:CreateStack',
                'cloudformation:DeleteStack',
                'cloudformation:DescribeStacks',
            ],
            resources: ['*'],
            conditions: {
                'ForAllValues:StringEquals': {
                    'cloudformation:TemplateURL': `https://s3.${urlsuffix}/${s3InstallBucket}/${s3InstallFolder}/templates/ComputeNode.template`,
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:RunInstances',
                'ec2:TerminateInstances',
                'ec2:CreatePlacementGroup',
                'ec2:DeletePlacementGroup',
            ],
            resources: [
                ...['subnet', 'key-pair', 'instance', 'launch-template', 'volume', 'security-group', 'placement-group', 'network-interface'].map(resource => stack.formatArn({ service: 'ec2', resource })).concat(['snapshot', 'image'].map(resource => stack.formatArn({ service: 'ec2', resource, account: '' }))),
            ],
            conditions: {
                'ForAllValues:ArnEqualsIfExists': {
                    'ec2:Vpc': stack.formatArn({ service: 'ec2', resource: 'vpc', resourceName: props.network.vpc.vpcId }),
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'lambda:InvokeFunction',
            ],
            resources: [
                stack.formatArn({
                    service: 'lambda',
                    resource: 'function',
                    resourceName: `${props.network.clusterId}-Metrics`,
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'fsx:CreateFileSystem',
            ],
            resources: [
                stack.formatArn({
                    service: 'fsx',
                    resource: 'file-system',
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'fsx:DeleteFileSystem',
            ],
            resources: [
                stack.formatArn({
                    service: 'fsx',
                    resource: 'file-system',
                }),
            ],
            conditions: {
                StringLike: {
                    'aws:ResourceTag/soca:ClusterId': props.network.clusterId,
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'iam:CreateServiceLinkedRole',
                'iam:AttachRolePolicy',
                'iam:PutRolePolicy',
            ],
            resources: [
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/s3.data-source.lustre.fsx.amazonaws.com/*',
                }),
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/autoscaling.amazonaws.com/*',
                }),
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/spotfleet.amazonaws.com/*',
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ses:SendEmail',
            ],
            resources: [
                stack.formatArn({
                    service: 'ses',
                    resource: 'identity',
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:CreatePlacementGroup',
                'ec2:DeletePlacementGroup',
                'ec2:RequestSpotFleet',
                'ec2:ModifySpotFleetRequest',
                'ec2:CancelSpotFleetRequests',
            ],
            resources: ['*'],
            conditions: {
                'ForAllValues:ArnEqualsIfExists': {
                    'ec2:Vpc': stack.formatArn({
                        service: 'ec2',
                        resource: 'vpc',
                        resourceName: props.network.vpc.vpcId,
                    }),
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject',
                's3:ListBucket',
                's3:PutObject',
            ],
            resources: [
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: props.s3InstallBucket,
                }),
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: `${props.s3InstallBucket}/*`,
                }),
            ],
        }));
        const eip = new ec2.CfnEIP(this, 'EIPScheduler', {
            instanceId: scheduler.instanceId,
            domain: props.network.vpc.vpcId,
        });
        new cdk.CfnOutput(this, 'SchedulerEIP', { value: eip.ref });
    }
}
exports.Scheduler = Scheduler;
//# sourceMappingURL=data:application/json;base64,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