"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessClamscan = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const path = require("path");
const aws_ec2_1 = require("monocdk/aws-ec2");
const aws_efs_1 = require("monocdk/aws-efs");
const aws_events_1 = require("monocdk/aws-events");
const aws_events_targets_1 = require("monocdk/aws-events-targets");
const aws_iam_1 = require("monocdk/aws-iam");
const aws_lambda_1 = require("monocdk/aws-lambda");
const aws_lambda_destinations_1 = require("monocdk/aws-lambda-destinations");
const aws_lambda_event_sources_1 = require("monocdk/aws-lambda-event-sources");
const aws_s3_1 = require("monocdk/aws-s3");
const aws_sqs_1 = require("monocdk/aws-sqs");
const monocdk_1 = require("monocdk");
const monocdk_nag_1 = require("monocdk-nag");
/**
  An [aws-cdk](https://github.com/aws/aws-cdk) construct that uses [ClamAV®](https://www.clamav.net/).
  to scan objects in Amazon S3 for viruses. The construct provides a flexible interface for a system
  to act based on the results of a ClamAV virus scan.

  The construct creates a Lambda function with EFS integration to support larger files.
  A VPC with isolated subnets, a S3 Gateway endpoint will also be created.

  Additionally creates an twice-daily job to download the latest ClamAV definition files to the
  Virus Definitions S3 Bucket by utilizing an EventBridge rule and a Lambda function and
  publishes CloudWatch Metrics to the 'serverless-clamscan' namespace.

  __Important O&M__:
  When ClamAV publishes updates to the scanner you will see “Your ClamAV installation is OUTDATED” in your scan results.
  While the construct creates a system to keep the database definitions up to date, you must update the scanner to
  detect all the latest Viruses.

  Update the docker images of the Lambda functions with the latest version of ClamAV by re-running `cdk deploy`.

  Successful Scan Event format
  ```json
  {
     "source": "serverless-clamscan",
     "input_bucket": <input_bucket_name>,
     "input_key": <object_key>,
     "status": <"CLEAN"|"INFECTED"|"N/A">,
     "message": <scan_summary>,
   }
  ```

  Note: The Virus Definitions bucket policy will likely cause a deletion error if you choose to delete
  the stack associated in the construct. However since the bucket itself gets deleted, you can delete
  the stack again to resolve the error.
 */
class ServerlessClamscan extends monocdk_1.Construct {
    /**
     * Creates a ServerlessClamscan construct.
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param props A `ServerlessClamscanProps` interface.
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this._efsRootPath = '/lambda';
        this._efsMountPath = `/mnt${this._efsRootPath}`;
        this._efsDefsPath = 'virus_database/';
        if (!props.onResult) {
            this.resultBus = new aws_events_1.EventBus(this, 'ScanResultBus');
            this.resultDest = new aws_lambda_destinations_1.EventBridgeDestination(this.resultBus);
            this.infectedRule = new aws_events_1.Rule(this, 'InfectedRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked INFECTED',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['INFECTED'],
                        },
                    },
                },
            });
            this.cleanRule = new aws_events_1.Rule(this, 'CleanRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked CLEAN',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['CLEAN'],
                        },
                    },
                },
            });
        }
        else {
            this.resultDest = props.onResult;
        }
        if (!props.onError) {
            this.errorDeadLetterQueue = new aws_sqs_1.Queue(this, 'ScanErrorDeadLetterQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            });
            this.errorDeadLetterQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorDeadLetterQueue.queueArn],
            }));
            this.errorQueue = new aws_sqs_1.Queue(this, 'ScanErrorQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
                deadLetterQueue: {
                    maxReceiveCount: 3,
                    queue: this.errorDeadLetterQueue,
                },
            });
            this.errorQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorQueue.queueArn],
            }));
            this.errorDest = new aws_lambda_destinations_1.SqsDestination(this.errorQueue);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this.errorDeadLetterQueue, [
                { id: 'AwsSolutions-SQS3', reason: 'This queue is a DLQ.' },
            ]);
        }
        else {
            this.errorDest = props.onError;
        }
        const vpc = new aws_ec2_1.Vpc(this, 'ScanVPC', {
            subnetConfiguration: [
                {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
                    name: 'Isolated',
                },
            ],
        });
        vpc.addFlowLog('FlowLogs');
        this._s3Gw = vpc.addGatewayEndpoint('S3Endpoint', {
            service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
        });
        const fileSystem = new aws_efs_1.FileSystem(this, 'ScanFileSystem', {
            vpc: vpc,
            encrypted: props.efsEncryption === false ? false : true,
            lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_7_DAYS,
            performanceMode: aws_efs_1.PerformanceMode.GENERAL_PURPOSE,
            removalPolicy: monocdk_1.RemovalPolicy.DESTROY,
            securityGroup: new aws_ec2_1.SecurityGroup(this, 'ScanFileSystemSecurityGroup', {
                vpc: vpc,
                allowAllOutbound: false,
            }),
        });
        const lambda_ap = fileSystem.addAccessPoint('ScanLambdaAP', {
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '755',
            },
            posixUser: {
                gid: '1000',
                uid: '1000',
            },
            path: this._efsRootPath,
        });
        const logs_bucket = (_b = props.defsBucketAccessLogsConfig) === null || _b === void 0 ? void 0 : _b.logsBucket;
        const logs_bucket_prefix = (_c = props.defsBucketAccessLogsConfig) === null || _c === void 0 ? void 0 : _c.logsPrefix;
        if (logs_bucket === true || logs_bucket === undefined) {
            this.defsAccessLogsBucket = new aws_s3_1.Bucket(this, 'VirusDefsAccessLogsBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: monocdk_1.RemovalPolicy.RETAIN,
                serverAccessLogsPrefix: 'access-logs-bucket-logs',
                blockPublicAccess: {
                    blockPublicAcls: true,
                    blockPublicPolicy: true,
                    ignorePublicAcls: true,
                    restrictPublicBuckets: true,
                },
            });
            this.defsAccessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:*'],
                resources: [
                    this.defsAccessLogsBucket.arnForObjects('*'),
                    this.defsAccessLogsBucket.bucketArn,
                ],
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: {
                    Bool: {
                        'aws:SecureTransport': false,
                    },
                },
            }));
        }
        else if (logs_bucket != false) {
            this.defsAccessLogsBucket = logs_bucket;
        }
        const defs_bucket = new aws_s3_1.Bucket(this, 'VirusDefsBucket', {
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            removalPolicy: monocdk_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            serverAccessLogsBucket: this.defsAccessLogsBucket,
            serverAccessLogsPrefix: logs_bucket === false ? undefined : logs_bucket_prefix,
            blockPublicAccess: {
                blockPublicAcls: true,
                blockPublicPolicy: true,
                ignorePublicAcls: true,
                restrictPublicBuckets: true,
            },
        });
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:*'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                Bool: {
                    'aws:SecureTransport': false,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                StringEquals: {
                    'aws:SourceVpce': this._s3Gw.vpcEndpointId,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:PutBucketPolicy', 's3:DeleteBucketPolicy'],
            resources: [defs_bucket.bucketArn],
            notPrincipals: [new aws_iam_1.AccountRootPrincipal()],
        }));
        this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
        }));
        this._scanFunction = new aws_lambda_1.DockerImageFunction(this, 'ServerlessClamscan', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/scan'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            onSuccess: this.resultDest,
            onFailure: this.errorDest,
            filesystem: aws_lambda_1.FileSystem.fromEfsAccessPoint(lambda_ap, this._efsMountPath),
            vpc: vpc,
            vpcSubnets: { subnets: vpc.isolatedSubnets },
            allowAllOutbound: false,
            timeout: monocdk_1.Duration.minutes(15),
            memorySize: 10240,
            reservedConcurrentExecutions: props.reservedConcurrency,
            environment: {
                EFS_MOUNT_PATH: this._efsMountPath,
                EFS_DEF_PATH: this._efsDefsPath,
                DEFS_URL: defs_bucket.virtualHostedUrlForObject(),
                POWERTOOLS_METRICS_NAMESPACE: 'serverless-clamscan',
                POWERTOOLS_SERVICE_NAME: 'virus-scan',
            },
        });
        if (this._scanFunction.role) {
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this._scanFunction.role, [
                { id: 'AwsSolutions-IAM4', reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch. The AWSLambdaVPCAccessExecutionRole is required for functions with VPC access to manage elastic network interfaces.' },
            ]);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this._scanFunction.role, [{
                    id: 'AwsSolutions-IAM5',
                    reason: 'The EFS mount point permissions are controlled through a condition which limit the scope of the * resources.',
                }], true);
        }
        this._scanFunction.connections.allowToAnyIpv4(aws_ec2_1.Port.tcp(443), 'Allow outbound HTTPS traffic for S3 access.');
        defs_bucket.grantRead(this._scanFunction);
        const download_defs = new aws_lambda_1.DockerImageFunction(this, 'DownloadDefs', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/download_defs'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            timeout: monocdk_1.Duration.minutes(5),
            memorySize: 1024,
            environment: {
                DEFS_BUCKET: defs_bucket.bucketName,
                POWERTOOLS_SERVICE_NAME: 'freshclam-update',
            },
        });
        const stack = monocdk_1.Stack.of(this);
        if (download_defs.role) {
            const download_defs_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${download_defs.role.roleName}/${download_defs.functionName}`;
            const download_defs_assumed_principal = new aws_iam_1.ArnPrincipal(download_defs_role);
            defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:PutObject*'],
                resources: [defs_bucket.arnForObjects('*')],
                notPrincipals: [download_defs.role, download_defs_assumed_principal],
            }));
            defs_bucket.grantReadWrite(download_defs);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(download_defs.role, [{
                    id: 'AwsSolutions-IAM4',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                }]);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(download_defs.role, [{
                    id: 'AwsSolutions-IAM5',
                    reason: 'The function is allowed to perform operations on all prefixes in the specified bucket.',
                }], true);
        }
        new aws_events_1.Rule(this, 'VirusDefsUpdateRule', {
            schedule: aws_events_1.Schedule.rate(monocdk_1.Duration.hours(12)),
            targets: [new aws_events_targets_1.LambdaFunction(download_defs)],
        });
        const init_defs_cr = new aws_lambda_1.Function(this, 'InitDefs', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_8,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../assets/lambda/code/initialize_defs_cr')),
            handler: 'lambda.lambda_handler',
            timeout: monocdk_1.Duration.minutes(5),
        });
        download_defs.grantInvoke(init_defs_cr);
        if (init_defs_cr.role) {
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(init_defs_cr.role, [{
                    id: 'AwsSolutions-IAM4',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                }]);
        }
        new monocdk_1.CustomResource(this, 'InitDefsCr', {
            serviceToken: init_defs_cr.functionArn,
            properties: {
                FnName: download_defs.functionName,
            },
        });
        if (props.buckets) {
            props.buckets.forEach((bucket) => {
                this.addSourceBucket(bucket);
            });
        }
    }
    /**
     * Sets the specified S3 Bucket as a s3:ObjectCreate* for the ClamAV function.
       Grants the ClamAV function permissions to get and tag objects.
       Adds a bucket policy to disallow GetObject operations on files that are tagged 'IN PROGRESS', 'INFECTED', or 'ERROR'.
     * @param bucket The bucket to add the scanning bucket policy and s3:ObjectCreate* trigger to.
     */
    addSourceBucket(bucket) {
        this._scanFunction.addEventSource(new aws_lambda_event_sources_1.S3EventSource(bucket, { events: [aws_s3_1.EventType.OBJECT_CREATED] }));
        bucket.grantRead(this._scanFunction);
        this._scanFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
            resources: [bucket.arnForObjects('*')],
        }));
        if (this._scanFunction.role) {
            const stack = monocdk_1.Stack.of(this);
            const scan_assumed_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${this._scanFunction.role.roleName}/${this._scanFunction.functionName}`;
            const scan_assumed_principal = new aws_iam_1.ArnPrincipal(scan_assumed_role);
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                resources: [bucket.bucketArn, bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
                resources: [bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            // Need the assumed role for the not Principal Action with Lambda
            bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:GetObject'],
                resources: [bucket.arnForObjects('*')],
                notPrincipals: [this._scanFunction.role, scan_assumed_principal],
                conditions: {
                    StringEquals: {
                        's3:ExistingObjectTag/scan-status': [
                            'IN PROGRESS',
                            'INFECTED',
                            'ERROR',
                        ],
                    },
                },
            }));
        }
    }
}
exports.ServerlessClamscan = ServerlessClamscan;
_a = JSII_RTTI_SYMBOL_1;
ServerlessClamscan[_a] = { fqn: "monocdk-serverless-clamscan.ServerlessClamscan", version: "1.1.21" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxxRUFBcUU7QUFDckUsc0NBQXNDO0FBRXRDLDZCQUE2QjtBQUM3Qiw2Q0FPeUI7QUFDekIsNkNBQStFO0FBQy9FLG1EQUE4RDtBQUM5RCxtRUFBNEQ7QUFDNUQsNkNBTXlCO0FBQ3pCLG1EQVE0QjtBQUM1Qiw2RUFHeUM7QUFDekMsK0VBQWlFO0FBQ2pFLDJDQUFxRTtBQUNyRSw2Q0FBeUQ7QUFDekQscUNBTWlCO0FBQ2pCLDZDQUE4QztBQThDOUM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQWlDRztBQUNILE1BQWEsa0JBQW1CLFNBQVEsbUJBQVM7SUErQy9DOzs7OztPQUtHO0lBQ0gsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE4Qjs7UUFDdEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQVhYLGlCQUFZLEdBQUcsU0FBUyxDQUFDO1FBQ3pCLGtCQUFhLEdBQUcsT0FBTyxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7UUFDM0MsaUJBQVksR0FBRyxpQkFBaUIsQ0FBQztRQVd2QyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNuQixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUkscUJBQVEsQ0FBQyxJQUFJLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDckQsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLGdEQUFzQixDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUM3RCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksaUJBQUksQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO2dCQUNqRCxRQUFRLEVBQUUsSUFBSSxDQUFDLFNBQVM7Z0JBQ3hCLFdBQVcsRUFBRSwwQ0FBMEM7Z0JBQ3ZELFlBQVksRUFBRTtvQkFDWixNQUFNLEVBQUU7d0JBQ04sZUFBZSxFQUFFOzRCQUNmLE1BQU0sRUFBRSxDQUFDLHFCQUFxQixDQUFDOzRCQUMvQixNQUFNLEVBQUUsQ0FBQyxVQUFVLENBQUM7eUJBQ3JCO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLGlCQUFJLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtnQkFDM0MsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTO2dCQUN4QixXQUFXLEVBQUUsdUNBQXVDO2dCQUNwRCxZQUFZLEVBQUU7b0JBQ1osTUFBTSxFQUFFO3dCQUNOLGVBQWUsRUFBRTs0QkFDZixNQUFNLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQzs0QkFDL0IsTUFBTSxFQUFFLENBQUMsT0FBTyxDQUFDO3lCQUNsQjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQztTQUNKO2FBQU07WUFDTCxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7U0FDbEM7UUFFRCxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRTtZQUNsQixJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxlQUFLLENBQUMsSUFBSSxFQUFFLDBCQUEwQixFQUFFO2dCQUN0RSxVQUFVLEVBQUUseUJBQWUsQ0FBQyxXQUFXO2FBQ3hDLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLHlCQUFlLENBQUM7Z0JBQ2hFLE9BQU8sRUFBRSxDQUFDLE9BQU8sQ0FBQztnQkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsSUFBSTtnQkFDbkIsVUFBVSxFQUFFLENBQUMsSUFBSSxzQkFBWSxFQUFFLENBQUM7Z0JBQ2hDLFVBQVUsRUFBRSxFQUFFLElBQUksRUFBRSxFQUFFLHFCQUFxQixFQUFFLEtBQUssRUFBRSxFQUFFO2dCQUN0RCxTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsUUFBUSxDQUFDO2FBQ2hELENBQUMsQ0FBQyxDQUFDO1lBQ0osSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLGVBQUssQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7Z0JBQ2xELFVBQVUsRUFBRSx5QkFBZSxDQUFDLFdBQVc7Z0JBQ3ZDLGVBQWUsRUFBRTtvQkFDZixlQUFlLEVBQUUsQ0FBQztvQkFDbEIsS0FBSyxFQUFFLElBQUksQ0FBQyxvQkFBb0I7aUJBQ2pDO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLHlCQUFlLENBQUM7Z0JBQ3RELE9BQU8sRUFBRSxDQUFDLE9BQU8sQ0FBQztnQkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsSUFBSTtnQkFDbkIsVUFBVSxFQUFFLENBQUMsSUFBSSxzQkFBWSxFQUFFLENBQUM7Z0JBQ2hDLFVBQVUsRUFBRSxFQUFFLElBQUksRUFBRSxFQUFFLHFCQUFxQixFQUFFLEtBQUssRUFBRSxFQUFFO2dCQUN0RCxTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQzthQUN0QyxDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSx3Q0FBYyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUNyRCw2QkFBZSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxvQkFBb0IsRUFBRTtnQkFDakUsRUFBRSxFQUFFLEVBQUUsbUJBQW1CLEVBQUUsTUFBTSxFQUFFLHNCQUFzQixFQUFFO2FBQzVELENBQUMsQ0FBQztTQUNKO2FBQU07WUFDTCxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7U0FDaEM7UUFFRCxNQUFNLEdBQUcsR0FBRyxJQUFJLGFBQUcsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ25DLG1CQUFtQixFQUFFO2dCQUNuQjtvQkFDRSxVQUFVLEVBQUUsb0JBQVUsQ0FBQyxnQkFBZ0I7b0JBQ3ZDLElBQUksRUFBRSxVQUFVO2lCQUNqQjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsR0FBRyxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUUzQixJQUFJLENBQUMsS0FBSyxHQUFHLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxZQUFZLEVBQUU7WUFDaEQsT0FBTyxFQUFFLHNDQUE0QixDQUFDLEVBQUU7U0FDekMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxVQUFVLEdBQUcsSUFBSSxvQkFBVSxDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtZQUN4RCxHQUFHLEVBQUUsR0FBRztZQUNSLFNBQVMsRUFBRSxLQUFLLENBQUMsYUFBYSxLQUFLLEtBQUssQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJO1lBQ3ZELGVBQWUsRUFBRSx5QkFBZSxDQUFDLFlBQVk7WUFDN0MsZUFBZSxFQUFFLHlCQUFlLENBQUMsZUFBZTtZQUNoRCxhQUFhLEVBQUUsdUJBQWEsQ0FBQyxPQUFPO1lBQ3BDLGFBQWEsRUFBRSxJQUFJLHVCQUFhLENBQUMsSUFBSSxFQUFFLDZCQUE2QixFQUFFO2dCQUNwRSxHQUFHLEVBQUUsR0FBRztnQkFDUixnQkFBZ0IsRUFBRSxLQUFLO2FBQ3hCLENBQUM7U0FDSCxDQUFDLENBQUM7UUFFSCxNQUFNLFNBQVMsR0FBRyxVQUFVLENBQUMsY0FBYyxDQUFDLGNBQWMsRUFBRTtZQUMxRCxTQUFTLEVBQUU7Z0JBQ1QsUUFBUSxFQUFFLE1BQU07Z0JBQ2hCLFFBQVEsRUFBRSxNQUFNO2dCQUNoQixXQUFXLEVBQUUsS0FBSzthQUNuQjtZQUNELFNBQVMsRUFBRTtnQkFDVCxHQUFHLEVBQUUsTUFBTTtnQkFDWCxHQUFHLEVBQUUsTUFBTTthQUNaO1lBQ0QsSUFBSSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ3hCLENBQUMsQ0FBQztRQUVILE1BQU0sV0FBVyxTQUFHLEtBQUssQ0FBQywwQkFBMEIsMENBQUUsVUFBVSxDQUFDO1FBQ2pFLE1BQU0sa0JBQWtCLFNBQUcsS0FBSyxDQUFDLDBCQUEwQiwwQ0FBRSxVQUFVLENBQUM7UUFDeEUsSUFBSSxXQUFXLEtBQUssSUFBSSxJQUFJLFdBQVcsS0FBSyxTQUFTLEVBQUU7WUFDckQsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksZUFBTSxDQUNwQyxJQUFJLEVBQ0osMkJBQTJCLEVBQzNCO2dCQUNFLFVBQVUsRUFBRSx5QkFBZ0IsQ0FBQyxVQUFVO2dCQUN2QyxhQUFhLEVBQUUsdUJBQWEsQ0FBQyxNQUFNO2dCQUNuQyxzQkFBc0IsRUFBRSx5QkFBeUI7Z0JBQ2pELGlCQUFpQixFQUFFO29CQUNqQixlQUFlLEVBQUUsSUFBSTtvQkFDckIsaUJBQWlCLEVBQUUsSUFBSTtvQkFDdkIsZ0JBQWdCLEVBQUUsSUFBSTtvQkFDdEIscUJBQXFCLEVBQUUsSUFBSTtpQkFDNUI7YUFDRixDQUNGLENBQUM7WUFDRixJQUFJLENBQUMsb0JBQW9CLENBQUMsbUJBQW1CLENBQzNDLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsSUFBSTtnQkFDbkIsT0FBTyxFQUFFLENBQUMsTUFBTSxDQUFDO2dCQUNqQixTQUFTLEVBQUU7b0JBQ1QsSUFBSSxDQUFDLG9CQUFvQixDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUM7b0JBQzVDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTO2lCQUNwQztnQkFDRCxVQUFVLEVBQUUsQ0FBQyxJQUFJLHNCQUFZLEVBQUUsQ0FBQztnQkFDaEMsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRTt3QkFDSixxQkFBcUIsRUFBRSxLQUFLO3FCQUM3QjtpQkFDRjthQUNGLENBQUMsQ0FDSCxDQUFDO1NBQ0g7YUFBTSxJQUFJLFdBQVcsSUFBSSxLQUFLLEVBQUU7WUFDL0IsSUFBSSxDQUFDLG9CQUFvQixHQUFHLFdBQVcsQ0FBQztTQUN6QztRQUVELE1BQU0sV0FBVyxHQUFHLElBQUksZUFBTSxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtZQUN0RCxVQUFVLEVBQUUseUJBQWdCLENBQUMsVUFBVTtZQUN2QyxhQUFhLEVBQUUsdUJBQWEsQ0FBQyxPQUFPO1lBQ3BDLGlCQUFpQixFQUFFLElBQUk7WUFDdkIsc0JBQXNCLEVBQUUsSUFBSSxDQUFDLG9CQUFvQjtZQUNqRCxzQkFBc0IsRUFDcEIsV0FBVyxLQUFLLEtBQUssQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxrQkFBa0I7WUFDeEQsaUJBQWlCLEVBQUU7Z0JBQ2pCLGVBQWUsRUFBRSxJQUFJO2dCQUNyQixpQkFBaUIsRUFBRSxJQUFJO2dCQUN2QixnQkFBZ0IsRUFBRSxJQUFJO2dCQUN0QixxQkFBcUIsRUFBRSxJQUFJO2FBQzVCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsV0FBVyxDQUFDLG1CQUFtQixDQUM3QixJQUFJLHlCQUFlLENBQUM7WUFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsSUFBSTtZQUNuQixPQUFPLEVBQUUsQ0FBQyxNQUFNLENBQUM7WUFDakIsU0FBUyxFQUFFLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsRUFBRSxXQUFXLENBQUMsU0FBUyxDQUFDO1lBQ2xFLFVBQVUsRUFBRSxDQUFDLElBQUksc0JBQVksRUFBRSxDQUFDO1lBQ2hDLFVBQVUsRUFBRTtnQkFDVixJQUFJLEVBQUU7b0JBQ0oscUJBQXFCLEVBQUUsS0FBSztpQkFDN0I7YUFDRjtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBQ0YsV0FBVyxDQUFDLG1CQUFtQixDQUM3QixJQUFJLHlCQUFlLENBQUM7WUFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSztZQUNwQixPQUFPLEVBQUUsQ0FBQyxjQUFjLEVBQUUsZUFBZSxDQUFDO1lBQzFDLFNBQVMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLEVBQUUsV0FBVyxDQUFDLFNBQVMsQ0FBQztZQUNsRSxVQUFVLEVBQUUsQ0FBQyxJQUFJLHNCQUFZLEVBQUUsQ0FBQztZQUNoQyxVQUFVLEVBQUU7Z0JBQ1YsWUFBWSxFQUFFO29CQUNaLGdCQUFnQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYTtpQkFDM0M7YUFDRjtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBQ0YsV0FBVyxDQUFDLG1CQUFtQixDQUM3QixJQUFJLHlCQUFlLENBQUM7WUFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsSUFBSTtZQUNuQixPQUFPLEVBQUUsQ0FBQyxvQkFBb0IsRUFBRSx1QkFBdUIsQ0FBQztZQUN4RCxTQUFTLEVBQUUsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDO1lBQ2xDLGFBQWEsRUFBRSxDQUFDLElBQUksOEJBQW9CLEVBQUUsQ0FBQztTQUM1QyxDQUFDLENBQ0gsQ0FBQztRQUNGLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUNwQixJQUFJLHlCQUFlLENBQUM7WUFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSztZQUNwQixPQUFPLEVBQUUsQ0FBQyxjQUFjLEVBQUUsZUFBZSxDQUFDO1lBQzFDLFNBQVMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLEVBQUUsV0FBVyxDQUFDLFNBQVMsQ0FBQztZQUNsRSxVQUFVLEVBQUUsQ0FBQyxJQUFJLHNCQUFZLEVBQUUsQ0FBQztTQUNqQyxDQUFDLENBQ0gsQ0FBQztRQUVGLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxnQ0FBbUIsQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLEVBQUU7WUFDdkUsSUFBSSxFQUFFLDRCQUFlLENBQUMsY0FBYyxDQUNsQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSw0QkFBNEIsQ0FBQyxFQUNsRDtnQkFDRSxTQUFTLEVBQUU7b0JBQ1Qsc0RBQXNEO29CQUN0RCxVQUFVLEVBQUUsSUFBSSxJQUFJLEVBQUUsQ0FBQyxZQUFZLEVBQUU7aUJBQ3RDO2dCQUNELFNBQVMsRUFBRSxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFO2FBQ2pDLENBQ0Y7WUFDRCxTQUFTLEVBQUUsSUFBSSxDQUFDLFVBQVU7WUFDMUIsU0FBUyxFQUFFLElBQUksQ0FBQyxTQUFTO1lBQ3pCLFVBQVUsRUFBRSx1QkFBZ0IsQ0FBQyxrQkFBa0IsQ0FDN0MsU0FBUyxFQUNULElBQUksQ0FBQyxhQUFhLENBQ25CO1lBQ0QsR0FBRyxFQUFFLEdBQUc7WUFDUixVQUFVLEVBQUUsRUFBRSxPQUFPLEVBQUUsR0FBRyxDQUFDLGVBQWUsRUFBRTtZQUM1QyxnQkFBZ0IsRUFBRSxLQUFLO1lBQ3ZCLE9BQU8sRUFBRSxrQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDN0IsVUFBVSxFQUFFLEtBQUs7WUFDakIsNEJBQTRCLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUN2RCxXQUFXLEVBQUU7Z0JBQ1gsY0FBYyxFQUFFLElBQUksQ0FBQyxhQUFhO2dCQUNsQyxZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7Z0JBQy9CLFFBQVEsRUFBRSxXQUFXLENBQUMseUJBQXlCLEVBQUU7Z0JBQ2pELDRCQUE0QixFQUFFLHFCQUFxQjtnQkFDbkQsdUJBQXVCLEVBQUUsWUFBWTthQUN0QztTQUNGLENBQUMsQ0FBQztRQUNILElBQUksSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUU7WUFDM0IsNkJBQWUsQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRTtnQkFDL0QsRUFBRSxFQUFFLEVBQUUsbUJBQW1CLEVBQUUsTUFBTSxFQUFFLHVOQUF1TixFQUFFO2FBQzdQLENBQUMsQ0FBQztZQUNILDZCQUFlLENBQUMsdUJBQXVCLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsQ0FBQztvQkFDaEUsRUFBRSxFQUFFLG1CQUFtQjtvQkFDdkIsTUFBTSxFQUNKLDhHQUE4RztpQkFDakgsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1NBQ1g7UUFDRCxJQUFJLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQzNDLGNBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQ2IsNkNBQTZDLENBQzlDLENBQUM7UUFDRixXQUFXLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUUxQyxNQUFNLGFBQWEsR0FBRyxJQUFJLGdDQUFtQixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDbEUsSUFBSSxFQUFFLDRCQUFlLENBQUMsY0FBYyxDQUNsQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxxQ0FBcUMsQ0FBQyxFQUMzRDtnQkFDRSxTQUFTLEVBQUU7b0JBQ1Qsc0RBQXNEO29CQUN0RCxVQUFVLEVBQUUsSUFBSSxJQUFJLEVBQUUsQ0FBQyxZQUFZLEVBQUU7aUJBQ3RDO2dCQUNELFNBQVMsRUFBRSxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFO2FBQ2pDLENBQ0Y7WUFDRCxPQUFPLEVBQUUsa0JBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQzVCLFVBQVUsRUFBRSxJQUFJO1lBQ2hCLFdBQVcsRUFBRTtnQkFDWCxXQUFXLEVBQUUsV0FBVyxDQUFDLFVBQVU7Z0JBQ25DLHVCQUF1QixFQUFFLGtCQUFrQjthQUM1QztTQUNGLENBQUMsQ0FBQztRQUNILE1BQU0sS0FBSyxHQUFHLGVBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFN0IsSUFBSSxhQUFhLENBQUMsSUFBSSxFQUFFO1lBQ3RCLE1BQU0sa0JBQWtCLEdBQUcsT0FBTyxLQUFLLENBQUMsU0FBUyxTQUFTLEtBQUssQ0FBQyxPQUFPLGlCQUFpQixhQUFhLENBQUMsSUFBSSxDQUFDLFFBQVEsSUFBSSxhQUFhLENBQUMsWUFBWSxFQUFFLENBQUM7WUFDcEosTUFBTSwrQkFBK0IsR0FBRyxJQUFJLHNCQUFZLENBQ3RELGtCQUFrQixDQUNuQixDQUFDO1lBQ0YsV0FBVyxDQUFDLG1CQUFtQixDQUM3QixJQUFJLHlCQUFlLENBQUM7Z0JBQ2xCLE1BQU0sRUFBRSxnQkFBTSxDQUFDLElBQUk7Z0JBQ25CLE9BQU8sRUFBRSxDQUFDLGVBQWUsQ0FBQztnQkFDMUIsU0FBUyxFQUFFLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDM0MsYUFBYSxFQUFFLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSwrQkFBK0IsQ0FBQzthQUNyRSxDQUFDLENBQ0gsQ0FBQztZQUNGLFdBQVcsQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLENBQUM7WUFDMUMsNkJBQWUsQ0FBQyx1QkFBdUIsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLENBQUM7b0JBQzNELEVBQUUsRUFBRSxtQkFBbUI7b0JBQ3ZCLE1BQU0sRUFDSixtR0FBbUc7aUJBQ3RHLENBQUMsQ0FBQyxDQUFDO1lBQ0osNkJBQWUsQ0FBQyx1QkFBdUIsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLENBQUM7b0JBQzNELEVBQUUsRUFBRSxtQkFBbUI7b0JBQ3ZCLE1BQU0sRUFDSix3RkFBd0Y7aUJBQzNGLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztTQUNYO1FBRUQsSUFBSSxpQkFBSSxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtZQUNwQyxRQUFRLEVBQUUscUJBQVEsQ0FBQyxJQUFJLENBQUMsa0JBQVEsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDM0MsT0FBTyxFQUFFLENBQUMsSUFBSSxtQ0FBYyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQzdDLENBQUMsQ0FBQztRQUVILE1BQU0sWUFBWSxHQUFHLElBQUkscUJBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ2xELE9BQU8sRUFBRSxvQkFBTyxDQUFDLFVBQVU7WUFDM0IsSUFBSSxFQUFFLGlCQUFJLENBQUMsU0FBUyxDQUNsQixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwwQ0FBMEMsQ0FBQyxDQUNqRTtZQUNELE9BQU8sRUFBRSx1QkFBdUI7WUFDaEMsT0FBTyxFQUFFLGtCQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztTQUM3QixDQUFDLENBQUM7UUFDSCxhQUFhLENBQUMsV0FBVyxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQ3hDLElBQUksWUFBWSxDQUFDLElBQUksRUFBRTtZQUNyQiw2QkFBZSxDQUFDLHVCQUF1QixDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsQ0FBQztvQkFDMUQsRUFBRSxFQUFFLG1CQUFtQjtvQkFDdkIsTUFBTSxFQUNKLG1HQUFtRztpQkFDdEcsQ0FBQyxDQUFDLENBQUM7U0FDTDtRQUNELElBQUksd0JBQWMsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ3JDLFlBQVksRUFBRSxZQUFZLENBQUMsV0FBVztZQUN0QyxVQUFVLEVBQUU7Z0JBQ1YsTUFBTSxFQUFFLGFBQWEsQ0FBQyxZQUFZO2FBQ25DO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFO1lBQ2pCLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUU7Z0JBQy9CLElBQUksQ0FBQyxlQUFlLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDL0IsQ0FBQyxDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFRDs7Ozs7T0FLRztJQUNILGVBQWUsQ0FBQyxNQUFjO1FBQzVCLElBQUksQ0FBQyxhQUFhLENBQUMsY0FBYyxDQUMvQixJQUFJLHdDQUFhLENBQUMsTUFBTSxFQUFFLEVBQUUsTUFBTSxFQUFFLENBQUMsa0JBQVMsQ0FBQyxjQUFjLENBQUMsRUFBRSxDQUFDLENBQ2xFLENBQUM7UUFDRixNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUNyQyxJQUFJLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FDaEMsSUFBSSx5QkFBZSxDQUFDO1lBQ2xCLE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7WUFDcEIsT0FBTyxFQUFFLENBQUMscUJBQXFCLEVBQUUsNEJBQTRCLENBQUM7WUFDOUQsU0FBUyxFQUFFLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUN2QyxDQUFDLENBQ0gsQ0FBQztRQUVGLElBQUksSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUU7WUFDM0IsTUFBTSxLQUFLLEdBQUcsZUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUM3QixNQUFNLGlCQUFpQixHQUFHLE9BQU8sS0FBSyxDQUFDLFNBQVMsU0FBUyxLQUFLLENBQUMsT0FBTyxpQkFBaUIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsUUFBUSxJQUFJLElBQUksQ0FBQyxhQUFhLENBQUMsWUFBWSxFQUFFLENBQUM7WUFDN0osTUFBTSxzQkFBc0IsR0FBRyxJQUFJLHNCQUFZLENBQUMsaUJBQWlCLENBQUMsQ0FBQztZQUNuRSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FDcEIsSUFBSSx5QkFBZSxDQUFDO2dCQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLO2dCQUNwQixPQUFPLEVBQUUsQ0FBQyxlQUFlLEVBQUUsZUFBZSxFQUFFLFVBQVUsQ0FBQztnQkFDdkQsU0FBUyxFQUFFLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDO2dCQUN4RCxVQUFVLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxzQkFBc0IsQ0FBQzthQUM5RCxDQUFDLENBQ0gsQ0FBQztZQUNGLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUNwQixJQUFJLHlCQUFlLENBQUM7Z0JBQ2xCLE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7Z0JBQ3BCLE9BQU8sRUFBRSxDQUFDLHFCQUFxQixFQUFFLDRCQUE0QixDQUFDO2dCQUM5RCxTQUFTLEVBQUUsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDO2dCQUN0QyxVQUFVLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxzQkFBc0IsQ0FBQzthQUM5RCxDQUFDLENBQ0gsQ0FBQztZQUVGLGlFQUFpRTtZQUNqRSxNQUFNLENBQUMsbUJBQW1CLENBQ3hCLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsSUFBSTtnQkFDbkIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO2dCQUN6QixTQUFTLEVBQUUsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDO2dCQUN0QyxhQUFhLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxzQkFBc0IsQ0FBQztnQkFDaEUsVUFBVSxFQUFFO29CQUNWLFlBQVksRUFBRTt3QkFDWixrQ0FBa0MsRUFBRTs0QkFDbEMsYUFBYTs0QkFDYixVQUFVOzRCQUNWLE9BQU87eUJBQ1I7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQ0gsQ0FBQztTQUNIO0lBQ0gsQ0FBQzs7QUEzYkgsZ0RBNGJDIiwic291cmNlc0NvbnRlbnQiOlsiLy8gQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4vLyBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuXG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHtcbiAgVnBjLFxuICBTdWJuZXRUeXBlLFxuICBHYXRld2F5VnBjRW5kcG9pbnQsXG4gIEdhdGV3YXlWcGNFbmRwb2ludEF3c1NlcnZpY2UsXG4gIFBvcnQsXG4gIFNlY3VyaXR5R3JvdXAsXG59IGZyb20gJ21vbm9jZGsvYXdzLWVjMic7XG5pbXBvcnQgeyBGaWxlU3lzdGVtLCBMaWZlY3ljbGVQb2xpY3ksIFBlcmZvcm1hbmNlTW9kZSB9IGZyb20gJ21vbm9jZGsvYXdzLWVmcyc7XG5pbXBvcnQgeyBFdmVudEJ1cywgUnVsZSwgU2NoZWR1bGUgfSBmcm9tICdtb25vY2RrL2F3cy1ldmVudHMnO1xuaW1wb3J0IHsgTGFtYmRhRnVuY3Rpb24gfSBmcm9tICdtb25vY2RrL2F3cy1ldmVudHMtdGFyZ2V0cyc7XG5pbXBvcnQge1xuICBFZmZlY3QsXG4gIFBvbGljeVN0YXRlbWVudCxcbiAgQXJuUHJpbmNpcGFsLFxuICBBbnlQcmluY2lwYWwsXG4gIEFjY291bnRSb290UHJpbmNpcGFsLFxufSBmcm9tICdtb25vY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHtcbiAgRG9ja2VySW1hZ2VDb2RlLFxuICBEb2NrZXJJbWFnZUZ1bmN0aW9uLFxuICBGdW5jdGlvbixcbiAgSURlc3RpbmF0aW9uLFxuICBGaWxlU3lzdGVtIGFzIExhbWJkYUZpbGVTeXN0ZW0sXG4gIFJ1bnRpbWUsXG4gIENvZGUsXG59IGZyb20gJ21vbm9jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQge1xuICBFdmVudEJyaWRnZURlc3RpbmF0aW9uLFxuICBTcXNEZXN0aW5hdGlvbixcbn0gZnJvbSAnbW9ub2Nkay9hd3MtbGFtYmRhLWRlc3RpbmF0aW9ucyc7XG5pbXBvcnQgeyBTM0V2ZW50U291cmNlIH0gZnJvbSAnbW9ub2Nkay9hd3MtbGFtYmRhLWV2ZW50LXNvdXJjZXMnO1xuaW1wb3J0IHsgQnVja2V0LCBCdWNrZXRFbmNyeXB0aW9uLCBFdmVudFR5cGUgfSBmcm9tICdtb25vY2RrL2F3cy1zMyc7XG5pbXBvcnQgeyBRdWV1ZSwgUXVldWVFbmNyeXB0aW9uIH0gZnJvbSAnbW9ub2Nkay9hd3Mtc3FzJztcbmltcG9ydCB7XG4gIENvbnN0cnVjdCxcbiAgRHVyYXRpb24sXG4gIEN1c3RvbVJlc291cmNlLFxuICBSZW1vdmFsUG9saWN5LFxuICBTdGFjayxcbn0gZnJvbSAnbW9ub2Nkayc7XG5pbXBvcnQgeyBOYWdTdXBwcmVzc2lvbnMgfSBmcm9tICdtb25vY2RrLW5hZyc7XG4vKipcbiAqIEludGVyZmFjZSBmb3IgU2VydmVybGVzc0NsYW1zY2FuIFZpcnVzIERlZmluaXRpb25zIFMzIEJ1Y2tldCBMb2dnaW5nLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNlcnZlcmxlc3NDbGFtc2NhbkxvZ2dpbmdQcm9wcyB7XG4gIC8qKlxuICAgKiBEZXN0aW5hdGlvbiBidWNrZXQgZm9yIHRoZSBzZXJ2ZXIgYWNjZXNzIGxvZ3MgKERlZmF1bHQ6IENyZWF0ZXMgYSBuZXcgUzMgQnVja2V0IGZvciBhY2Nlc3MgbG9ncyApLlxuICAgKi9cbiAgcmVhZG9ubHkgbG9nc0J1Y2tldD86IGJvb2xlYW4gfCBCdWNrZXQ7XG4gIC8qKlxuICAgKiBPcHRpb25hbCBsb2cgZmlsZSBwcmVmaXggdG8gdXNlIGZvciB0aGUgYnVja2V0J3MgYWNjZXNzIGxvZ3MsIG9wdGlvbiBpcyBpZ25vcmVkIGlmIGxvZ3NfYnVja2V0IGlzIHNldCB0byBmYWxzZS5cbiAgICovXG4gIHJlYWRvbmx5IGxvZ3NQcmVmaXg/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogSW50ZXJmYWNlIGZvciBjcmVhdGluZyBhIFNlcnZlcmxlc3NDbGFtc2Nhbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTZXJ2ZXJsZXNzQ2xhbXNjYW5Qcm9wcyB7XG4gIC8qKlxuICAgKiBBbiBvcHRpb25hbCBsaXN0IG9mIFMzIGJ1Y2tldHMgdG8gY29uZmlndXJlIGZvciBDbGFtQVYgVmlydXMgU2Nhbm5pbmc7IGJ1Y2tldHMgY2FuIGJlIGFkZGVkIGxhdGVyIGJ5IGNhbGxpbmcgYWRkU291cmNlQnVja2V0LlxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0cz86IEJ1Y2tldFtdO1xuICAvKipcbiAgICogT3B0aW9uYWxseSBzZXQgYSByZXNlcnZlZCBjb25jdXJyZW5jeSBmb3IgdGhlIHZpcnVzIHNjYW5uaW5nIExhbWJkYS5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGFtYmRhL2xhdGVzdC9vcGVyYXRvcmd1aWRlL3Jlc2VydmVkLWNvbmN1cnJlbmN5Lmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHJlc2VydmVkQ29uY3VycmVuY3k/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBUaGUgTGFtYmRhIERlc3RpbmF0aW9uIGZvciBmaWxlcyBtYXJrZWQgJ0NMRUFOJyBvciAnSU5GRUNURUQnIGJhc2VkIG9uIHRoZSBDbGFtQVYgVmlydXMgc2NhbiBvciAnTi9BJyBmb3Igc2NhbnMgdHJpZ2dlcmVkIGJ5IFMzIGZvbGRlciBjcmVhdGlvbiBldmVudHMgbWFya2VkIChEZWZhdWx0OiBDcmVhdGVzIGFuZCBwdWJsaXNoZXMgdG8gYSBuZXcgRXZlbnQgQnJpZGdlIEJ1cyBpZiB1bnNwZWNpZmllZCkuXG4gICAqL1xuICByZWFkb25seSBvblJlc3VsdD86IElEZXN0aW5hdGlvbjtcbiAgLyoqXG4gICAqIFRoZSBMYW1iZGEgRGVzdGluYXRpb24gZm9yIGZpbGVzIHRoYXQgZmFpbCB0byBzY2FuIGFuZCBhcmUgbWFya2VkICdFUlJPUicgb3Igc3R1Y2sgJ0lOIFBST0dSRVNTJyBkdWUgdG8gYSBMYW1iZGEgdGltZW91dCAoRGVmYXVsdDogQ3JlYXRlcyBhbmQgcHVibGlzaGVzIHRvIGEgbmV3IFNRUyBxdWV1ZSBpZiB1bnNwZWNpZmllZCkuXG4gICAqL1xuICByZWFkb25seSBvbkVycm9yPzogSURlc3RpbmF0aW9uO1xuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdG8gZW5hYmxlIGVuY3J5cHRpb24gb24gRUZTIGZpbGVzeXN0ZW0gKERlZmF1bHQ6IGVuYWJsZWQpLlxuICAgKi9cbiAgcmVhZG9ubHkgZWZzRW5jcnlwdGlvbj86IGJvb2xlYW47XG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0byBlbmFibGUgQWNjZXNzIExvZ2dpbmcgZm9yIHRoZSBWaXJ1cyBEZWZpbml0aW9ucyBidWNrZXQsIHlvdSBjYW4gc3BlY2lmeSBhbiBleGlzdGluZyBidWNrZXQgYW5kIHByZWZpeCAoRGVmYXVsdDogQ3JlYXRlcyBhIG5ldyBTMyBCdWNrZXQgZm9yIGFjY2VzcyBsb2dzICkuXG4gICAqL1xuICByZWFkb25seSBkZWZzQnVja2V0QWNjZXNzTG9nc0NvbmZpZz86IFNlcnZlcmxlc3NDbGFtc2NhbkxvZ2dpbmdQcm9wcztcbn1cblxuLyoqXG4gIEFuIFthd3MtY2RrXShodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1jZGspIGNvbnN0cnVjdCB0aGF0IHVzZXMgW0NsYW1BVsKuXShodHRwczovL3d3dy5jbGFtYXYubmV0LykuXG4gIHRvIHNjYW4gb2JqZWN0cyBpbiBBbWF6b24gUzMgZm9yIHZpcnVzZXMuIFRoZSBjb25zdHJ1Y3QgcHJvdmlkZXMgYSBmbGV4aWJsZSBpbnRlcmZhY2UgZm9yIGEgc3lzdGVtXG4gIHRvIGFjdCBiYXNlZCBvbiB0aGUgcmVzdWx0cyBvZiBhIENsYW1BViB2aXJ1cyBzY2FuLlxuXG4gIFRoZSBjb25zdHJ1Y3QgY3JlYXRlcyBhIExhbWJkYSBmdW5jdGlvbiB3aXRoIEVGUyBpbnRlZ3JhdGlvbiB0byBzdXBwb3J0IGxhcmdlciBmaWxlcy5cbiAgQSBWUEMgd2l0aCBpc29sYXRlZCBzdWJuZXRzLCBhIFMzIEdhdGV3YXkgZW5kcG9pbnQgd2lsbCBhbHNvIGJlIGNyZWF0ZWQuXG5cbiAgQWRkaXRpb25hbGx5IGNyZWF0ZXMgYW4gdHdpY2UtZGFpbHkgam9iIHRvIGRvd25sb2FkIHRoZSBsYXRlc3QgQ2xhbUFWIGRlZmluaXRpb24gZmlsZXMgdG8gdGhlXG4gIFZpcnVzIERlZmluaXRpb25zIFMzIEJ1Y2tldCBieSB1dGlsaXppbmcgYW4gRXZlbnRCcmlkZ2UgcnVsZSBhbmQgYSBMYW1iZGEgZnVuY3Rpb24gYW5kXG4gIHB1Ymxpc2hlcyBDbG91ZFdhdGNoIE1ldHJpY3MgdG8gdGhlICdzZXJ2ZXJsZXNzLWNsYW1zY2FuJyBuYW1lc3BhY2UuXG5cbiAgX19JbXBvcnRhbnQgTyZNX186XG4gIFdoZW4gQ2xhbUFWIHB1Ymxpc2hlcyB1cGRhdGVzIHRvIHRoZSBzY2FubmVyIHlvdSB3aWxsIHNlZSDigJxZb3VyIENsYW1BViBpbnN0YWxsYXRpb24gaXMgT1VUREFURUTigJ0gaW4geW91ciBzY2FuIHJlc3VsdHMuXG4gIFdoaWxlIHRoZSBjb25zdHJ1Y3QgY3JlYXRlcyBhIHN5c3RlbSB0byBrZWVwIHRoZSBkYXRhYmFzZSBkZWZpbml0aW9ucyB1cCB0byBkYXRlLCB5b3UgbXVzdCB1cGRhdGUgdGhlIHNjYW5uZXIgdG9cbiAgZGV0ZWN0IGFsbCB0aGUgbGF0ZXN0IFZpcnVzZXMuXG5cbiAgVXBkYXRlIHRoZSBkb2NrZXIgaW1hZ2VzIG9mIHRoZSBMYW1iZGEgZnVuY3Rpb25zIHdpdGggdGhlIGxhdGVzdCB2ZXJzaW9uIG9mIENsYW1BViBieSByZS1ydW5uaW5nIGBjZGsgZGVwbG95YC5cblxuICBTdWNjZXNzZnVsIFNjYW4gRXZlbnQgZm9ybWF0XG4gIGBgYGpzb25cbiAge1xuICAgICBcInNvdXJjZVwiOiBcInNlcnZlcmxlc3MtY2xhbXNjYW5cIixcbiAgICAgXCJpbnB1dF9idWNrZXRcIjogPGlucHV0X2J1Y2tldF9uYW1lPixcbiAgICAgXCJpbnB1dF9rZXlcIjogPG9iamVjdF9rZXk+LFxuICAgICBcInN0YXR1c1wiOiA8XCJDTEVBTlwifFwiSU5GRUNURURcInxcIk4vQVwiPixcbiAgICAgXCJtZXNzYWdlXCI6IDxzY2FuX3N1bW1hcnk+LFxuICAgfVxuICBgYGBcblxuICBOb3RlOiBUaGUgVmlydXMgRGVmaW5pdGlvbnMgYnVja2V0IHBvbGljeSB3aWxsIGxpa2VseSBjYXVzZSBhIGRlbGV0aW9uIGVycm9yIGlmIHlvdSBjaG9vc2UgdG8gZGVsZXRlXG4gIHRoZSBzdGFjayBhc3NvY2lhdGVkIGluIHRoZSBjb25zdHJ1Y3QuIEhvd2V2ZXIgc2luY2UgdGhlIGJ1Y2tldCBpdHNlbGYgZ2V0cyBkZWxldGVkLCB5b3UgY2FuIGRlbGV0ZVxuICB0aGUgc3RhY2sgYWdhaW4gdG8gcmVzb2x2ZSB0aGUgZXJyb3IuXG4gKi9cbmV4cG9ydCBjbGFzcyBTZXJ2ZXJsZXNzQ2xhbXNjYW4gZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAvKipcbiAgICBUaGUgTGFtYmRhIERlc3RpbmF0aW9uIGZvciBmYWlsZWQgb24gZXJyZWQgc2NhbnMgW0VSUk9SLCBJTiBQUk9HUkVTUyAoSWYgZXJyb3IgaXMgZHVlIHRvIExhbWJkYSB0aW1lb3V0KV0uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZXJyb3JEZXN0OiBJRGVzdGluYXRpb247XG5cbiAgLyoqXG4gICAgVGhlIExhbWJkYSBEZXN0aW5hdGlvbiBmb3IgY29tcGxldGVkIENsYW1BViBzY2FucyBbQ0xFQU4sIElORkVDVEVEXS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByZXN1bHREZXN0OiBJRGVzdGluYXRpb247XG5cbiAgLyoqXG4gICAgQ29uZGl0aW9uYWw6IFRoZSBTUVMgUXVldWUgZm9yIGVycmVkIHNjYW5zIGlmIGEgZmFpbHVyZSAob25FcnJvcikgZGVzdGluYXRpb24gd2FzIG5vdCBzcGVjaWZpZWQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZXJyb3JRdWV1ZT86IFF1ZXVlO1xuXG4gIC8qKlxuICAgIENvbmRpdGlvbmFsOiBUaGUgU1FTIERlYWQgTGV0dGVyIFF1ZXVlIGZvciB0aGUgZXJyb3JRdWV1ZSBpZiBhIGZhaWx1cmUgKG9uRXJyb3IpIGRlc3RpbmF0aW9uIHdhcyBub3Qgc3BlY2lmaWVkLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGVycm9yRGVhZExldHRlclF1ZXVlPzogUXVldWU7XG5cbiAgLyoqXG4gICAgQ29uZGl0aW9uYWw6IFRoZSBFdmVudCBCcmlkZ2UgQnVzIGZvciBjb21wbGV0ZWQgQ2xhbUFWIHNjYW5zIGlmIGEgc3VjY2VzcyAob25SZXN1bHQpIGRlc3RpbmF0aW9uIHdhcyBub3Qgc3BlY2lmaWVkLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlc3VsdEJ1cz86IEV2ZW50QnVzO1xuXG4gIC8qKlxuICAgIENvbmRpdGlvbmFsOiBBbiBFdmVudCBCcmlkZ2UgUnVsZSBmb3IgZmlsZXMgdGhhdCBhcmUgbWFya2VkICdDTEVBTicgYnkgQ2xhbUFWIGlmIGEgc3VjY2VzcyBkZXN0aW5hdGlvbiB3YXMgbm90IHNwZWNpZmllZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjbGVhblJ1bGU/OiBSdWxlO1xuXG4gIC8qKlxuICAgIENvbmRpdGlvbmFsOiBBbiBFdmVudCBCcmlkZ2UgUnVsZSBmb3IgZmlsZXMgdGhhdCBhcmUgbWFya2VkICdJTkZFQ1RFRCcgYnkgQ2xhbUFWIGlmIGEgc3VjY2VzcyBkZXN0aW5hdGlvbiB3YXMgbm90IHNwZWNpZmllZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBpbmZlY3RlZFJ1bGU/OiBSdWxlO1xuXG4gIC8qKlxuICAgIENvbmRpdGlvbmFsOiBUaGUgQnVja2V0IGZvciBhY2Nlc3MgbG9ncyBmb3IgdGhlIHZpcnVzIGRlZmluaXRpb25zIGJ1Y2tldCBpZiBsb2dnaW5nIGlzIGVuYWJsZWQgKGRlZnNCdWNrZXRBY2Nlc3NMb2dzQ29uZmlnKS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZWZzQWNjZXNzTG9nc0J1Y2tldD86IEJ1Y2tldDtcblxuICBwcml2YXRlIF9zY2FuRnVuY3Rpb246IERvY2tlckltYWdlRnVuY3Rpb247XG4gIHByaXZhdGUgX3MzR3c6IEdhdGV3YXlWcGNFbmRwb2ludDtcbiAgcHJpdmF0ZSBfZWZzUm9vdFBhdGggPSAnL2xhbWJkYSc7XG4gIHByaXZhdGUgX2Vmc01vdW50UGF0aCA9IGAvbW50JHt0aGlzLl9lZnNSb290UGF0aH1gO1xuICBwcml2YXRlIF9lZnNEZWZzUGF0aCA9ICd2aXJ1c19kYXRhYmFzZS8nO1xuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgU2VydmVybGVzc0NsYW1zY2FuIGNvbnN0cnVjdC5cbiAgICogQHBhcmFtIHNjb3BlIFRoZSBwYXJlbnQgY3JlYXRpbmcgY29uc3RydWN0ICh1c3VhbGx5IGB0aGlzYCkuXG4gICAqIEBwYXJhbSBpZCBUaGUgY29uc3RydWN0J3MgbmFtZS5cbiAgICogQHBhcmFtIHByb3BzIEEgYFNlcnZlcmxlc3NDbGFtc2NhblByb3BzYCBpbnRlcmZhY2UuXG4gICAqL1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogU2VydmVybGVzc0NsYW1zY2FuUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgaWYgKCFwcm9wcy5vblJlc3VsdCkge1xuICAgICAgdGhpcy5yZXN1bHRCdXMgPSBuZXcgRXZlbnRCdXModGhpcywgJ1NjYW5SZXN1bHRCdXMnKTtcbiAgICAgIHRoaXMucmVzdWx0RGVzdCA9IG5ldyBFdmVudEJyaWRnZURlc3RpbmF0aW9uKHRoaXMucmVzdWx0QnVzKTtcbiAgICAgIHRoaXMuaW5mZWN0ZWRSdWxlID0gbmV3IFJ1bGUodGhpcywgJ0luZmVjdGVkUnVsZScsIHtcbiAgICAgICAgZXZlbnRCdXM6IHRoaXMucmVzdWx0QnVzLFxuICAgICAgICBkZXNjcmlwdGlvbjogJ0V2ZW50IGZvciB3aGVuIGEgZmlsZSBpcyBtYXJrZWQgSU5GRUNURUQnLFxuICAgICAgICBldmVudFBhdHRlcm46IHtcbiAgICAgICAgICBkZXRhaWw6IHtcbiAgICAgICAgICAgIHJlc3BvbnNlUGF5bG9hZDoge1xuICAgICAgICAgICAgICBzb3VyY2U6IFsnc2VydmVybGVzcy1jbGFtc2NhbiddLFxuICAgICAgICAgICAgICBzdGF0dXM6IFsnSU5GRUNURUQnXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgICAgdGhpcy5jbGVhblJ1bGUgPSBuZXcgUnVsZSh0aGlzLCAnQ2xlYW5SdWxlJywge1xuICAgICAgICBldmVudEJ1czogdGhpcy5yZXN1bHRCdXMsXG4gICAgICAgIGRlc2NyaXB0aW9uOiAnRXZlbnQgZm9yIHdoZW4gYSBmaWxlIGlzIG1hcmtlZCBDTEVBTicsXG4gICAgICAgIGV2ZW50UGF0dGVybjoge1xuICAgICAgICAgIGRldGFpbDoge1xuICAgICAgICAgICAgcmVzcG9uc2VQYXlsb2FkOiB7XG4gICAgICAgICAgICAgIHNvdXJjZTogWydzZXJ2ZXJsZXNzLWNsYW1zY2FuJ10sXG4gICAgICAgICAgICAgIHN0YXR1czogWydDTEVBTiddLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMucmVzdWx0RGVzdCA9IHByb3BzLm9uUmVzdWx0O1xuICAgIH1cblxuICAgIGlmICghcHJvcHMub25FcnJvcikge1xuICAgICAgdGhpcy5lcnJvckRlYWRMZXR0ZXJRdWV1ZSA9IG5ldyBRdWV1ZSh0aGlzLCAnU2NhbkVycm9yRGVhZExldHRlclF1ZXVlJywge1xuICAgICAgICBlbmNyeXB0aW9uOiBRdWV1ZUVuY3J5cHRpb24uS01TX01BTkFHRUQsXG4gICAgICB9KTtcbiAgICAgIHRoaXMuZXJyb3JEZWFkTGV0dGVyUXVldWUuYWRkVG9SZXNvdXJjZVBvbGljeShuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWydzcXM6KiddLFxuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5ERU5ZLFxuICAgICAgICBwcmluY2lwYWxzOiBbbmV3IEFueVByaW5jaXBhbCgpXSxcbiAgICAgICAgY29uZGl0aW9uczogeyBCb29sOiB7ICdhd3M6U2VjdXJlVHJhbnNwb3J0JzogZmFsc2UgfSB9LFxuICAgICAgICByZXNvdXJjZXM6IFt0aGlzLmVycm9yRGVhZExldHRlclF1ZXVlLnF1ZXVlQXJuXSxcbiAgICAgIH0pKTtcbiAgICAgIHRoaXMuZXJyb3JRdWV1ZSA9IG5ldyBRdWV1ZSh0aGlzLCAnU2NhbkVycm9yUXVldWUnLCB7XG4gICAgICAgIGVuY3J5cHRpb246IFF1ZXVlRW5jcnlwdGlvbi5LTVNfTUFOQUdFRCxcbiAgICAgICAgZGVhZExldHRlclF1ZXVlOiB7XG4gICAgICAgICAgbWF4UmVjZWl2ZUNvdW50OiAzLFxuICAgICAgICAgIHF1ZXVlOiB0aGlzLmVycm9yRGVhZExldHRlclF1ZXVlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgICB0aGlzLmVycm9yUXVldWUuYWRkVG9SZXNvdXJjZVBvbGljeShuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWydzcXM6KiddLFxuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5ERU5ZLFxuICAgICAgICBwcmluY2lwYWxzOiBbbmV3IEFueVByaW5jaXBhbCgpXSxcbiAgICAgICAgY29uZGl0aW9uczogeyBCb29sOiB7ICdhd3M6U2VjdXJlVHJhbnNwb3J0JzogZmFsc2UgfSB9LFxuICAgICAgICByZXNvdXJjZXM6IFt0aGlzLmVycm9yUXVldWUucXVldWVBcm5dLFxuICAgICAgfSkpO1xuICAgICAgdGhpcy5lcnJvckRlc3QgPSBuZXcgU3FzRGVzdGluYXRpb24odGhpcy5lcnJvclF1ZXVlKTtcbiAgICAgIE5hZ1N1cHByZXNzaW9ucy5hZGRSZXNvdXJjZVN1cHByZXNzaW9ucyh0aGlzLmVycm9yRGVhZExldHRlclF1ZXVlLCBbXG4gICAgICAgIHsgaWQ6ICdBd3NTb2x1dGlvbnMtU1FTMycsIHJlYXNvbjogJ1RoaXMgcXVldWUgaXMgYSBETFEuJyB9LFxuICAgICAgXSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuZXJyb3JEZXN0ID0gcHJvcHMub25FcnJvcjtcbiAgICB9XG5cbiAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHRoaXMsICdTY2FuVlBDJywge1xuICAgICAgc3VibmV0Q29uZmlndXJhdGlvbjogW1xuICAgICAgICB7XG4gICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFX0lTT0xBVEVELFxuICAgICAgICAgIG5hbWU6ICdJc29sYXRlZCcsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgdnBjLmFkZEZsb3dMb2coJ0Zsb3dMb2dzJyk7XG5cbiAgICB0aGlzLl9zM0d3ID0gdnBjLmFkZEdhdGV3YXlFbmRwb2ludCgnUzNFbmRwb2ludCcsIHtcbiAgICAgIHNlcnZpY2U6IEdhdGV3YXlWcGNFbmRwb2ludEF3c1NlcnZpY2UuUzMsXG4gICAgfSk7XG5cbiAgICBjb25zdCBmaWxlU3lzdGVtID0gbmV3IEZpbGVTeXN0ZW0odGhpcywgJ1NjYW5GaWxlU3lzdGVtJywge1xuICAgICAgdnBjOiB2cGMsXG4gICAgICBlbmNyeXB0ZWQ6IHByb3BzLmVmc0VuY3J5cHRpb24gPT09IGZhbHNlID8gZmFsc2UgOiB0cnVlLFxuICAgICAgbGlmZWN5Y2xlUG9saWN5OiBMaWZlY3ljbGVQb2xpY3kuQUZURVJfN19EQVlTLFxuICAgICAgcGVyZm9ybWFuY2VNb2RlOiBQZXJmb3JtYW5jZU1vZGUuR0VORVJBTF9QVVJQT1NFLFxuICAgICAgcmVtb3ZhbFBvbGljeTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgICAgc2VjdXJpdHlHcm91cDogbmV3IFNlY3VyaXR5R3JvdXAodGhpcywgJ1NjYW5GaWxlU3lzdGVtU2VjdXJpdHlHcm91cCcsIHtcbiAgICAgICAgdnBjOiB2cGMsXG4gICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IGZhbHNlLFxuICAgICAgfSksXG4gICAgfSk7XG5cbiAgICBjb25zdCBsYW1iZGFfYXAgPSBmaWxlU3lzdGVtLmFkZEFjY2Vzc1BvaW50KCdTY2FuTGFtYmRhQVAnLCB7XG4gICAgICBjcmVhdGVBY2w6IHtcbiAgICAgICAgb3duZXJHaWQ6ICcxMDAwJyxcbiAgICAgICAgb3duZXJVaWQ6ICcxMDAwJyxcbiAgICAgICAgcGVybWlzc2lvbnM6ICc3NTUnLFxuICAgICAgfSxcbiAgICAgIHBvc2l4VXNlcjoge1xuICAgICAgICBnaWQ6ICcxMDAwJyxcbiAgICAgICAgdWlkOiAnMTAwMCcsXG4gICAgICB9LFxuICAgICAgcGF0aDogdGhpcy5fZWZzUm9vdFBhdGgsXG4gICAgfSk7XG5cbiAgICBjb25zdCBsb2dzX2J1Y2tldCA9IHByb3BzLmRlZnNCdWNrZXRBY2Nlc3NMb2dzQ29uZmlnPy5sb2dzQnVja2V0O1xuICAgIGNvbnN0IGxvZ3NfYnVja2V0X3ByZWZpeCA9IHByb3BzLmRlZnNCdWNrZXRBY2Nlc3NMb2dzQ29uZmlnPy5sb2dzUHJlZml4O1xuICAgIGlmIChsb2dzX2J1Y2tldCA9PT0gdHJ1ZSB8fCBsb2dzX2J1Y2tldCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aGlzLmRlZnNBY2Nlc3NMb2dzQnVja2V0ID0gbmV3IEJ1Y2tldChcbiAgICAgICAgdGhpcyxcbiAgICAgICAgJ1ZpcnVzRGVmc0FjY2Vzc0xvZ3NCdWNrZXQnLFxuICAgICAgICB7XG4gICAgICAgICAgZW5jcnlwdGlvbjogQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VELFxuICAgICAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuUkVUQUlOLFxuICAgICAgICAgIHNlcnZlckFjY2Vzc0xvZ3NQcmVmaXg6ICdhY2Nlc3MtbG9ncy1idWNrZXQtbG9ncycsXG4gICAgICAgICAgYmxvY2tQdWJsaWNBY2Nlc3M6IHtcbiAgICAgICAgICAgIGJsb2NrUHVibGljQWNsczogdHJ1ZSxcbiAgICAgICAgICAgIGJsb2NrUHVibGljUG9saWN5OiB0cnVlLFxuICAgICAgICAgICAgaWdub3JlUHVibGljQWNsczogdHJ1ZSxcbiAgICAgICAgICAgIHJlc3RyaWN0UHVibGljQnVja2V0czogdHJ1ZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgKTtcbiAgICAgIHRoaXMuZGVmc0FjY2Vzc0xvZ3NCdWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShcbiAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgZWZmZWN0OiBFZmZlY3QuREVOWSxcbiAgICAgICAgICBhY3Rpb25zOiBbJ3MzOionXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgIHRoaXMuZGVmc0FjY2Vzc0xvZ3NCdWNrZXQuYXJuRm9yT2JqZWN0cygnKicpLFxuICAgICAgICAgICAgdGhpcy5kZWZzQWNjZXNzTG9nc0J1Y2tldC5idWNrZXRBcm4sXG4gICAgICAgICAgXSxcbiAgICAgICAgICBwcmluY2lwYWxzOiBbbmV3IEFueVByaW5jaXBhbCgpXSxcbiAgICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgICBCb29sOiB7XG4gICAgICAgICAgICAgICdhd3M6U2VjdXJlVHJhbnNwb3J0JzogZmFsc2UsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pLFxuICAgICAgKTtcbiAgICB9IGVsc2UgaWYgKGxvZ3NfYnVja2V0ICE9IGZhbHNlKSB7XG4gICAgICB0aGlzLmRlZnNBY2Nlc3NMb2dzQnVja2V0ID0gbG9nc19idWNrZXQ7XG4gICAgfVxuXG4gICAgY29uc3QgZGVmc19idWNrZXQgPSBuZXcgQnVja2V0KHRoaXMsICdWaXJ1c0RlZnNCdWNrZXQnLCB7XG4gICAgICBlbmNyeXB0aW9uOiBCdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQsXG4gICAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICBhdXRvRGVsZXRlT2JqZWN0czogdHJ1ZSxcbiAgICAgIHNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQ6IHRoaXMuZGVmc0FjY2Vzc0xvZ3NCdWNrZXQsXG4gICAgICBzZXJ2ZXJBY2Nlc3NMb2dzUHJlZml4OlxuICAgICAgICBsb2dzX2J1Y2tldCA9PT0gZmFsc2UgPyB1bmRlZmluZWQgOiBsb2dzX2J1Y2tldF9wcmVmaXgsXG4gICAgICBibG9ja1B1YmxpY0FjY2Vzczoge1xuICAgICAgICBibG9ja1B1YmxpY0FjbHM6IHRydWUsXG4gICAgICAgIGJsb2NrUHVibGljUG9saWN5OiB0cnVlLFxuICAgICAgICBpZ25vcmVQdWJsaWNBY2xzOiB0cnVlLFxuICAgICAgICByZXN0cmljdFB1YmxpY0J1Y2tldHM6IHRydWUsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgZGVmc19idWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5ERU5ZLFxuICAgICAgICBhY3Rpb25zOiBbJ3MzOionXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbZGVmc19idWNrZXQuYXJuRm9yT2JqZWN0cygnKicpLCBkZWZzX2J1Y2tldC5idWNrZXRBcm5dLFxuICAgICAgICBwcmluY2lwYWxzOiBbbmV3IEFueVByaW5jaXBhbCgpXSxcbiAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgIEJvb2w6IHtcbiAgICAgICAgICAgICdhd3M6U2VjdXJlVHJhbnNwb3J0JzogZmFsc2UsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pLFxuICAgICk7XG4gICAgZGVmc19idWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnLCAnczM6TGlzdEJ1Y2tldCddLFxuICAgICAgICByZXNvdXJjZXM6IFtkZWZzX2J1Y2tldC5hcm5Gb3JPYmplY3RzKCcqJyksIGRlZnNfYnVja2V0LmJ1Y2tldEFybl0sXG4gICAgICAgIHByaW5jaXBhbHM6IFtuZXcgQW55UHJpbmNpcGFsKCldLFxuICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgICAnYXdzOlNvdXJjZVZwY2UnOiB0aGlzLl9zM0d3LnZwY0VuZHBvaW50SWQsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pLFxuICAgICk7XG4gICAgZGVmc19idWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5ERU5ZLFxuICAgICAgICBhY3Rpb25zOiBbJ3MzOlB1dEJ1Y2tldFBvbGljeScsICdzMzpEZWxldGVCdWNrZXRQb2xpY3knXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbZGVmc19idWNrZXQuYnVja2V0QXJuXSxcbiAgICAgICAgbm90UHJpbmNpcGFsczogW25ldyBBY2NvdW50Um9vdFByaW5jaXBhbCgpXSxcbiAgICAgIH0pLFxuICAgICk7XG4gICAgdGhpcy5fczNHdy5hZGRUb1BvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnLCAnczM6TGlzdEJ1Y2tldCddLFxuICAgICAgICByZXNvdXJjZXM6IFtkZWZzX2J1Y2tldC5hcm5Gb3JPYmplY3RzKCcqJyksIGRlZnNfYnVja2V0LmJ1Y2tldEFybl0sXG4gICAgICAgIHByaW5jaXBhbHM6IFtuZXcgQW55UHJpbmNpcGFsKCldLFxuICAgICAgfSksXG4gICAgKTtcblxuICAgIHRoaXMuX3NjYW5GdW5jdGlvbiA9IG5ldyBEb2NrZXJJbWFnZUZ1bmN0aW9uKHRoaXMsICdTZXJ2ZXJsZXNzQ2xhbXNjYW4nLCB7XG4gICAgICBjb2RlOiBEb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQoXG4gICAgICAgIHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9hc3NldHMvbGFtYmRhL2NvZGUvc2NhbicpLFxuICAgICAgICB7XG4gICAgICAgICAgYnVpbGRBcmdzOiB7XG4gICAgICAgICAgICAvLyBPbmx5IGZvcmNlIHVwZGF0ZSB0aGUgZG9ja2VyIGxheWVyIGNhY2hlIG9uY2UgYSBkYXlcbiAgICAgICAgICAgIENBQ0hFX0RBVEU6IG5ldyBEYXRlKCkudG9EYXRlU3RyaW5nKCksXG4gICAgICAgICAgfSxcbiAgICAgICAgICBleHRyYUhhc2g6IERhdGUubm93KCkudG9TdHJpbmcoKSxcbiAgICAgICAgfSxcbiAgICAgICksXG4gICAgICBvblN1Y2Nlc3M6IHRoaXMucmVzdWx0RGVzdCxcbiAgICAgIG9uRmFpbHVyZTogdGhpcy5lcnJvckRlc3QsXG4gICAgICBmaWxlc3lzdGVtOiBMYW1iZGFGaWxlU3lzdGVtLmZyb21FZnNBY2Nlc3NQb2ludChcbiAgICAgICAgbGFtYmRhX2FwLFxuICAgICAgICB0aGlzLl9lZnNNb3VudFBhdGgsXG4gICAgICApLFxuICAgICAgdnBjOiB2cGMsXG4gICAgICB2cGNTdWJuZXRzOiB7IHN1Ym5ldHM6IHZwYy5pc29sYXRlZFN1Ym5ldHMgfSxcbiAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IGZhbHNlLFxuICAgICAgdGltZW91dDogRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgICBtZW1vcnlTaXplOiAxMDI0MCxcbiAgICAgIHJlc2VydmVkQ29uY3VycmVudEV4ZWN1dGlvbnM6IHByb3BzLnJlc2VydmVkQ29uY3VycmVuY3ksXG4gICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICBFRlNfTU9VTlRfUEFUSDogdGhpcy5fZWZzTW91bnRQYXRoLFxuICAgICAgICBFRlNfREVGX1BBVEg6IHRoaXMuX2Vmc0RlZnNQYXRoLFxuICAgICAgICBERUZTX1VSTDogZGVmc19idWNrZXQudmlydHVhbEhvc3RlZFVybEZvck9iamVjdCgpLFxuICAgICAgICBQT1dFUlRPT0xTX01FVFJJQ1NfTkFNRVNQQUNFOiAnc2VydmVybGVzcy1jbGFtc2NhbicsXG4gICAgICAgIFBPV0VSVE9PTFNfU0VSVklDRV9OQU1FOiAndmlydXMtc2NhbicsXG4gICAgICB9LFxuICAgIH0pO1xuICAgIGlmICh0aGlzLl9zY2FuRnVuY3Rpb24ucm9sZSkge1xuICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKHRoaXMuX3NjYW5GdW5jdGlvbi5yb2xlLCBbXG4gICAgICAgIHsgaWQ6ICdBd3NTb2x1dGlvbnMtSUFNNCcsIHJlYXNvbjogJ1RoZSBBV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGUgZG9lcyBub3QgcHJvdmlkZSBwZXJtaXNzaW9ucyBiZXlvbmQgdXBsb2FkaW5nIGxvZ3MgdG8gQ2xvdWRXYXRjaC4gVGhlIEFXU0xhbWJkYVZQQ0FjY2Vzc0V4ZWN1dGlvblJvbGUgaXMgcmVxdWlyZWQgZm9yIGZ1bmN0aW9ucyB3aXRoIFZQQyBhY2Nlc3MgdG8gbWFuYWdlIGVsYXN0aWMgbmV0d29yayBpbnRlcmZhY2VzLicgfSxcbiAgICAgIF0pO1xuICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKHRoaXMuX3NjYW5GdW5jdGlvbi5yb2xlLCBbe1xuICAgICAgICBpZDogJ0F3c1NvbHV0aW9ucy1JQU01JyxcbiAgICAgICAgcmVhc29uOlxuICAgICAgICAgICdUaGUgRUZTIG1vdW50IHBvaW50IHBlcm1pc3Npb25zIGFyZSBjb250cm9sbGVkIHRocm91Z2ggYSBjb25kaXRpb24gd2hpY2ggbGltaXQgdGhlIHNjb3BlIG9mIHRoZSAqIHJlc291cmNlcy4nLFxuICAgICAgfV0sIHRydWUpO1xuICAgIH1cbiAgICB0aGlzLl9zY2FuRnVuY3Rpb24uY29ubmVjdGlvbnMuYWxsb3dUb0FueUlwdjQoXG4gICAgICBQb3J0LnRjcCg0NDMpLFxuICAgICAgJ0FsbG93IG91dGJvdW5kIEhUVFBTIHRyYWZmaWMgZm9yIFMzIGFjY2Vzcy4nLFxuICAgICk7XG4gICAgZGVmc19idWNrZXQuZ3JhbnRSZWFkKHRoaXMuX3NjYW5GdW5jdGlvbik7XG5cbiAgICBjb25zdCBkb3dubG9hZF9kZWZzID0gbmV3IERvY2tlckltYWdlRnVuY3Rpb24odGhpcywgJ0Rvd25sb2FkRGVmcycsIHtcbiAgICAgIGNvZGU6IERvY2tlckltYWdlQ29kZS5mcm9tSW1hZ2VBc3NldChcbiAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL2Fzc2V0cy9sYW1iZGEvY29kZS9kb3dubG9hZF9kZWZzJyksXG4gICAgICAgIHtcbiAgICAgICAgICBidWlsZEFyZ3M6IHtcbiAgICAgICAgICAgIC8vIE9ubHkgZm9yY2UgdXBkYXRlIHRoZSBkb2NrZXIgbGF5ZXIgY2FjaGUgb25jZSBhIGRheVxuICAgICAgICAgICAgQ0FDSEVfREFURTogbmV3IERhdGUoKS50b0RhdGVTdHJpbmcoKSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGV4dHJhSGFzaDogRGF0ZS5ub3coKS50b1N0cmluZygpLFxuICAgICAgICB9LFxuICAgICAgKSxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoNSksXG4gICAgICBtZW1vcnlTaXplOiAxMDI0LFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgREVGU19CVUNLRVQ6IGRlZnNfYnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgIFBPV0VSVE9PTFNfU0VSVklDRV9OQU1FOiAnZnJlc2hjbGFtLXVwZGF0ZScsXG4gICAgICB9LFxuICAgIH0pO1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2YodGhpcyk7XG5cbiAgICBpZiAoZG93bmxvYWRfZGVmcy5yb2xlKSB7XG4gICAgICBjb25zdCBkb3dubG9hZF9kZWZzX3JvbGUgPSBgYXJuOiR7c3RhY2sucGFydGl0aW9ufTpzdHM6OiR7c3RhY2suYWNjb3VudH06YXNzdW1lZC1yb2xlLyR7ZG93bmxvYWRfZGVmcy5yb2xlLnJvbGVOYW1lfS8ke2Rvd25sb2FkX2RlZnMuZnVuY3Rpb25OYW1lfWA7XG4gICAgICBjb25zdCBkb3dubG9hZF9kZWZzX2Fzc3VtZWRfcHJpbmNpcGFsID0gbmV3IEFyblByaW5jaXBhbChcbiAgICAgICAgZG93bmxvYWRfZGVmc19yb2xlLFxuICAgICAgKTtcbiAgICAgIGRlZnNfYnVja2V0LmFkZFRvUmVzb3VyY2VQb2xpY3koXG4gICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGVmZmVjdDogRWZmZWN0LkRFTlksXG4gICAgICAgICAgYWN0aW9uczogWydzMzpQdXRPYmplY3QqJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbZGVmc19idWNrZXQuYXJuRm9yT2JqZWN0cygnKicpXSxcbiAgICAgICAgICBub3RQcmluY2lwYWxzOiBbZG93bmxvYWRfZGVmcy5yb2xlLCBkb3dubG9hZF9kZWZzX2Fzc3VtZWRfcHJpbmNpcGFsXSxcbiAgICAgICAgfSksXG4gICAgICApO1xuICAgICAgZGVmc19idWNrZXQuZ3JhbnRSZWFkV3JpdGUoZG93bmxvYWRfZGVmcyk7XG4gICAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoZG93bmxvYWRfZGVmcy5yb2xlLCBbe1xuICAgICAgICBpZDogJ0F3c1NvbHV0aW9ucy1JQU00JyxcbiAgICAgICAgcmVhc29uOlxuICAgICAgICAgICdUaGUgQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlIGRvZXMgbm90IHByb3ZpZGUgcGVybWlzc2lvbnMgYmV5b25kIHVwbG9hZGluZyBsb2dzIHRvIENsb3VkV2F0Y2guJyxcbiAgICAgIH1dKTtcbiAgICAgIE5hZ1N1cHByZXNzaW9ucy5hZGRSZXNvdXJjZVN1cHByZXNzaW9ucyhkb3dubG9hZF9kZWZzLnJvbGUsIFt7XG4gICAgICAgIGlkOiAnQXdzU29sdXRpb25zLUlBTTUnLFxuICAgICAgICByZWFzb246XG4gICAgICAgICAgJ1RoZSBmdW5jdGlvbiBpcyBhbGxvd2VkIHRvIHBlcmZvcm0gb3BlcmF0aW9ucyBvbiBhbGwgcHJlZml4ZXMgaW4gdGhlIHNwZWNpZmllZCBidWNrZXQuJyxcbiAgICAgIH1dLCB0cnVlKTtcbiAgICB9XG5cbiAgICBuZXcgUnVsZSh0aGlzLCAnVmlydXNEZWZzVXBkYXRlUnVsZScsIHtcbiAgICAgIHNjaGVkdWxlOiBTY2hlZHVsZS5yYXRlKER1cmF0aW9uLmhvdXJzKDEyKSksXG4gICAgICB0YXJnZXRzOiBbbmV3IExhbWJkYUZ1bmN0aW9uKGRvd25sb2FkX2RlZnMpXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGluaXRfZGVmc19jciA9IG5ldyBGdW5jdGlvbih0aGlzLCAnSW5pdERlZnMnLCB7XG4gICAgICBydW50aW1lOiBSdW50aW1lLlBZVEhPTl8zXzgsXG4gICAgICBjb2RlOiBDb2RlLmZyb21Bc3NldChcbiAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL2Fzc2V0cy9sYW1iZGEvY29kZS9pbml0aWFsaXplX2RlZnNfY3InKSxcbiAgICAgICksXG4gICAgICBoYW5kbGVyOiAnbGFtYmRhLmxhbWJkYV9oYW5kbGVyJyxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoNSksXG4gICAgfSk7XG4gICAgZG93bmxvYWRfZGVmcy5ncmFudEludm9rZShpbml0X2RlZnNfY3IpO1xuICAgIGlmIChpbml0X2RlZnNfY3Iucm9sZSkge1xuICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKGluaXRfZGVmc19jci5yb2xlLCBbe1xuICAgICAgICBpZDogJ0F3c1NvbHV0aW9ucy1JQU00JyxcbiAgICAgICAgcmVhc29uOlxuICAgICAgICAgICdUaGUgQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlIGRvZXMgbm90IHByb3ZpZGUgcGVybWlzc2lvbnMgYmV5b25kIHVwbG9hZGluZyBsb2dzIHRvIENsb3VkV2F0Y2guJyxcbiAgICAgIH1dKTtcbiAgICB9XG4gICAgbmV3IEN1c3RvbVJlc291cmNlKHRoaXMsICdJbml0RGVmc0NyJywge1xuICAgICAgc2VydmljZVRva2VuOiBpbml0X2RlZnNfY3IuZnVuY3Rpb25Bcm4sXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIEZuTmFtZTogZG93bmxvYWRfZGVmcy5mdW5jdGlvbk5hbWUsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgaWYgKHByb3BzLmJ1Y2tldHMpIHtcbiAgICAgIHByb3BzLmJ1Y2tldHMuZm9yRWFjaCgoYnVja2V0KSA9PiB7XG4gICAgICAgIHRoaXMuYWRkU291cmNlQnVja2V0KGJ1Y2tldCk7XG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogU2V0cyB0aGUgc3BlY2lmaWVkIFMzIEJ1Y2tldCBhcyBhIHMzOk9iamVjdENyZWF0ZSogZm9yIHRoZSBDbGFtQVYgZnVuY3Rpb24uXG4gICAgIEdyYW50cyB0aGUgQ2xhbUFWIGZ1bmN0aW9uIHBlcm1pc3Npb25zIHRvIGdldCBhbmQgdGFnIG9iamVjdHMuXG4gICAgIEFkZHMgYSBidWNrZXQgcG9saWN5IHRvIGRpc2FsbG93IEdldE9iamVjdCBvcGVyYXRpb25zIG9uIGZpbGVzIHRoYXQgYXJlIHRhZ2dlZCAnSU4gUFJPR1JFU1MnLCAnSU5GRUNURUQnLCBvciAnRVJST1InLlxuICAgKiBAcGFyYW0gYnVja2V0IFRoZSBidWNrZXQgdG8gYWRkIHRoZSBzY2FubmluZyBidWNrZXQgcG9saWN5IGFuZCBzMzpPYmplY3RDcmVhdGUqIHRyaWdnZXIgdG8uXG4gICAqL1xuICBhZGRTb3VyY2VCdWNrZXQoYnVja2V0OiBCdWNrZXQpIHtcbiAgICB0aGlzLl9zY2FuRnVuY3Rpb24uYWRkRXZlbnRTb3VyY2UoXG4gICAgICBuZXcgUzNFdmVudFNvdXJjZShidWNrZXQsIHsgZXZlbnRzOiBbRXZlbnRUeXBlLk9CSkVDVF9DUkVBVEVEXSB9KSxcbiAgICApO1xuICAgIGJ1Y2tldC5ncmFudFJlYWQodGhpcy5fc2NhbkZ1bmN0aW9uKTtcbiAgICB0aGlzLl9zY2FuRnVuY3Rpb24uYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbJ3MzOlB1dE9iamVjdFRhZ2dpbmcnLCAnczM6UHV0T2JqZWN0VmVyc2lvblRhZ2dpbmcnXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbYnVja2V0LmFybkZvck9iamVjdHMoJyonKV0sXG4gICAgICB9KSxcbiAgICApO1xuXG4gICAgaWYgKHRoaXMuX3NjYW5GdW5jdGlvbi5yb2xlKSB7XG4gICAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHRoaXMpO1xuICAgICAgY29uc3Qgc2Nhbl9hc3N1bWVkX3JvbGUgPSBgYXJuOiR7c3RhY2sucGFydGl0aW9ufTpzdHM6OiR7c3RhY2suYWNjb3VudH06YXNzdW1lZC1yb2xlLyR7dGhpcy5fc2NhbkZ1bmN0aW9uLnJvbGUucm9sZU5hbWV9LyR7dGhpcy5fc2NhbkZ1bmN0aW9uLmZ1bmN0aW9uTmFtZX1gO1xuICAgICAgY29uc3Qgc2Nhbl9hc3N1bWVkX3ByaW5jaXBhbCA9IG5ldyBBcm5QcmluY2lwYWwoc2Nhbl9hc3N1bWVkX3JvbGUpO1xuICAgICAgdGhpcy5fczNHdy5hZGRUb1BvbGljeShcbiAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgZWZmZWN0OiBFZmZlY3QuQUxMT1csXG4gICAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QqJywgJ3MzOkdldEJ1Y2tldConLCAnczM6TGlzdConXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFtidWNrZXQuYnVja2V0QXJuLCBidWNrZXQuYXJuRm9yT2JqZWN0cygnKicpXSxcbiAgICAgICAgICBwcmluY2lwYWxzOiBbdGhpcy5fc2NhbkZ1bmN0aW9uLnJvbGUsIHNjYW5fYXNzdW1lZF9wcmluY2lwYWxdLFxuICAgICAgICB9KSxcbiAgICAgICk7XG4gICAgICB0aGlzLl9zM0d3LmFkZFRvUG9saWN5KFxuICAgICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgICBhY3Rpb25zOiBbJ3MzOlB1dE9iamVjdFRhZ2dpbmcnLCAnczM6UHV0T2JqZWN0VmVyc2lvblRhZ2dpbmcnXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFtidWNrZXQuYXJuRm9yT2JqZWN0cygnKicpXSxcbiAgICAgICAgICBwcmluY2lwYWxzOiBbdGhpcy5fc2NhbkZ1bmN0aW9uLnJvbGUsIHNjYW5fYXNzdW1lZF9wcmluY2lwYWxdLFxuICAgICAgICB9KSxcbiAgICAgICk7XG5cbiAgICAgIC8vIE5lZWQgdGhlIGFzc3VtZWQgcm9sZSBmb3IgdGhlIG5vdCBQcmluY2lwYWwgQWN0aW9uIHdpdGggTGFtYmRhXG4gICAgICBidWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShcbiAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgZWZmZWN0OiBFZmZlY3QuREVOWSxcbiAgICAgICAgICBhY3Rpb25zOiBbJ3MzOkdldE9iamVjdCddLFxuICAgICAgICAgIHJlc291cmNlczogW2J1Y2tldC5hcm5Gb3JPYmplY3RzKCcqJyldLFxuICAgICAgICAgIG5vdFByaW5jaXBhbHM6IFt0aGlzLl9zY2FuRnVuY3Rpb24ucm9sZSwgc2Nhbl9hc3N1bWVkX3ByaW5jaXBhbF0sXG4gICAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgICAgICdzMzpFeGlzdGluZ09iamVjdFRhZy9zY2FuLXN0YXR1cyc6IFtcbiAgICAgICAgICAgICAgICAnSU4gUFJPR1JFU1MnLFxuICAgICAgICAgICAgICAgICdJTkZFQ1RFRCcsXG4gICAgICAgICAgICAgICAgJ0VSUk9SJyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSksXG4gICAgICApO1xuICAgIH1cbiAgfVxufVxuIl19