# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['enumtables', 'enumtables.alembic_ops']

package_data = \
{'': ['*']}

install_requires = \
['sqlalchemy>=1.3.18,<2.0.0']

setup_kwargs = {
    'name': 'gino-enum-tables',
    'version': '1.0.7',
    'description': 'Support for storing enum values in a table instead of a type',
    'long_description': 'GINO Enumeration Tables\n====================================\n\nThis is a port of `SqlAlchemy Enum Tables <https://git.heptacle.fr/neshiraini/sqlalchemy-enum-tables>`__ to support `GINO orm <https://github.com/python-gino/gino>`__\n\nSQLAlchemy has built-in ``enum.Enum`` support,\nvia its column type ``sqlalchemy.Enum``.\nHowever, this type relies either on the backend\'s enum type,\nor on a check constraints. Both of these are immutable objects,\nwhich are a pain in the butt to modify\n(only PostgreSQL supports adding values to an enum type,\nand even then it doesn\'t support removing them).\n\nAnother often-used pattern to support enums in a database\nis via a dedicated table that reflects the enum values.\nThis requires updating the table everytime the enum is modified,\nbut doing so is much simpler than replacing a type.\n\nThis package allows you to create the enum table,\nand columns referencing that table, directly from\na Python enum class. It also interfaces with Alembic\nto automatically add ``INSERT`` and ``DELETE`` statements\nto your autogenerated migration scripts.\n\nWhen to use\n-----------\n\n1. Only works with Pythons\'s enumeration classes,\n   or at least one with a behavior similar to ``enum.Enum``.\n   Does not work with collections of arbitrary entries.\n2. Better used for frequently updated enumeration classes.\n3. **Do not** use with another package that provides\n   ``op.enum_insert`` and ``op.enum_delete`` operations in Alembic.\n\nInstallation\n------------\n::\n\n    pip install gino-enum-tables\n\nHow to use with GINO\n----------------------\n\n::\n\n    import enum\n\n    # Gino instance\n    from . import db\n\n    import enumtables as et\n\n    # Create the Python enumeration class\n    class MyEnum(enum.Enum):\n        HELLO = "HELLO"\n        WORLD = "WORLD"\n\n    # Create the enumeration table\n    # Pass your enum class and the GINO base model to enumtables.EnumTable\n    MyEnumTable = et.EnumTable(MyEnum, db.Model)\n\n    # Create a model class that uses the enum\n    class MyModel(Base):\n        __tablename__ = "my_model"\n        # Pass the enum table (not the enum class) to enumtables.EnumType\n        # Add the foreign key to the enum table\n        enum_value = db.Column(et.EnumType(MyEnumTable), db.ForeignKey(\'my_enum_table.item_id\') primary_key = True)\n\n        # When valued (on an instance of MyModel), enum_value will be an instance of MyEnum.\n\nFirst, the ``EnumTable`` factory takes the enum class and the base GINO model class\nto create the actual ORM class. Then this ORM class is passed to the ``EnumType`` custom type class, along with a ForeignKey to the enum table,\nto create the SQLAlchemy column linked to the enum table.\nThe column behaves just as if it had SqlAlchemy\'s own ``Enum`` type.\n\nNote that the ForeignKey points to \'my_enum_table.item_id\'. If the tablename argument isn\'t passed to EnumTable, the table name is created from the Enum name by doing a camelCase to snake_case conversion. \'item_id\' is the primary key of the enum table (my_enum_table) holding enum values.\n\nOn the implementation side, ``EnumTable`` is not a class,\nit\'s a factory function that performs Python black magic\nto create a subclass of the declarative base, and set it up to be a DB table\ncontaining the enum items (actually it just has one column ``item_id`` of type String).\n\nHow to use with Alembic\n-----------------------\n\nFirst add::\n\n    from enumtables import alembic_ops\n\nat the begining of your ``env.py`` file,\nthen add the same line in the imports of your ``script.py.mako`` file.\nThe package uses Alembic\'s standard hooks to take care of migration generation.\n\nDon\'t forget to review the migrations afterwards.\nEspecially make sure that, if the table did not exist before,\nthe ``op.enum_insert`` commands are located *after* the corresponding ``op.create_table`` command.\n\nOther uses\n-----------\n\nUsing the enum table class directly\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nThe enum table class behaves like any SqlAlchemy ORM class::\n\n    enum_query = session.query(MyEnumTable)\n    result = enum_query.first()\n\n    # The column item_id stores the name of the enum item as a string\n    enum_name = result.item_id\n\nAdding more columns to the enum tables\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nAny keyword argument passed to the ``EnumTable`` factory becomes a member of the table class.\nWhich means, you can pass anything (like a column) exactly as you would defined a usual ORM class::\n\n    BetterEnumTable = et.EnumTable(\n        MyEnum,\n\t\tBase,\n\n        # tablename is turned into __tablename__\n        tablename = "better_enum",\n\n        # Let\'s add a new column!\n        order = sa.Column(sa.Integer, nullable = False),\n\n        # And since it\'s an ordering number, let\'s make it unique too.\n        __table_args__ = (\n            sa.UniqueConstraint(\'order\'),\n        ),\n    )\n',
    'author': 'Gazorby',
    'author_email': 'gazorby@pm.me',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
}


setup(**setup_kwargs)
