"""
Designed to generate a rust module of types from an xsd file.

Xsd entry:

  <xsd:complexType name="compounddefType">
    <xsd:sequence>

Should map to a Rust struct with fields for child elements and attributes.

  <xsd:complexType name="incType">
    <xsd:simpleContent>

According to Wikipedia, simple content refers to text being allowed but no child elements.

  <xsd:complexType name="highlightType" mixed="true">
    <xsd:choice minOccurs="0" maxOccurs="unbounded">

mixed=true means that it should be a struct with a field to represent an array of entries which can either be text
or the child elements described by the xsd.

choice indicates that only one of the contained elements can be allowed which corresponds to an enum. If multiple
entries are allowed then we assume it means something like a Vec of an enum value


"""
from collections import defaultdict
from pprint import pprint
from typing import List
import xml.etree.ElementTree as ET
import re
import sys


class DocComment:
    def __init__(self, content: List[str]):
        self.content = content

    def __str__(self):
        return "//! " + "\n//! ".join(self.content)


class Struct:
    def __init__(self, name, field_groups):
        self.name = name
        self.field_groups = field_groups

    def __str__(self):
        field_groups_text = []
        for group, fields in self.field_groups.items():
            if group:
                field_groups_text.append(f"// {group}")

            fields = ",\n".join(str(field) for field in fields)
            fields = f"{fields}," if fields else fields

            field_groups_text.append(fields)

        field_groups_text = "\n".join(field_groups_text)
        return f"""
#[derive(Debug, PartialEq)]
pub struct {self.name} {{
{field_groups_text}
}}
"""


class Field:
    def __init__(self, name, type, comment: bool):
        self.name = name
        self.type = type
        self.comment = comment

    def __str__(self):
        comment = "// " if self.comment else ""
        return f"    {comment}pub {self.name}: {self.type}"


class Enum:
    def __init__(self, name, simple, entries):
        self.name = name
        self.entries = entries
        self.simple = simple

    def __str__(self):
        enum_string = "strum::EnumString, " if self.simple else ""
        strum_setting = '\n#[strum(serialize_all = "kebab-case")]' if self.simple else ""
        return f"""
#[derive(Debug, {enum_string}PartialEq)]{strum_setting}
pub enum {self.name} {{
    {self.entries}
}}
"""


class Option:
    def __init__(self, inner):
        self.inner = inner

    def __str__(self):
        return f"Option<{self.inner}>"


class Vec:
    def __init__(self, inner):
        self.inner = inner

    def __str__(self):
        return f"Vec<{self.inner}>"


class Vec1:
    def __init__(self, inner):
        self.inner = inner

    def __str__(self):
        return f"vec1::Vec1<{self.inner}>"


class PassThrough:
    def __init__(self, inner):
        self.inner = inner

    def __str__(self):
        return self.inner


class Text:
    def __init__(self, content):
        self.content = content

    def __str__(self):
        return self.content


def main(args):
    target_xsd_file = args[0]
    target_rs_file = args[1]

    comment_lookup = generate_comment_lookup(target_rs_file)

    tree = ET.parse(target_xsd_file)
    root = tree.getroot()

    with open(target_rs_file, "w") as output:
        entries = []

        entries.append(
            DocComment(
                [
                    "Automatically generated by:",
                    "",
                    f"   generate.py {' '.join(args)}",
                    "",
                ]
            )
        )

        for child in root:
            if "name" in child.attrib:
                if child.tag == "{http://www.w3.org/2001/XMLSchema}complexType":
                    if is_simple_content(child):
                        entries.extend(create_simple_content(child, comment_lookup))
                    elif is_mixed(child):
                        mixed = create_mixed(output, child, comment_lookup)
                        if mixed:
                            entries.extend(mixed)
                    else:
                        entries.extend(create_struct(output, child, comment_lookup))
                elif child.tag == "{http://www.w3.org/2001/XMLSchema}simpleType":
                    for grandchild in child:
                        if grandchild.tag == "{http://www.w3.org/2001/XMLSchema}restriction":
                            name = child.attrib["name"]
                            restriction = create_restriction(output, name, grandchild)
                            if restriction:
                                entries.extend(restriction)
                elif child.tag == "{http://www.w3.org/2001/XMLSchema}group":
                    group = create_group(child)
                    if group:
                        entries.extend(group)

        for entry in entries:
            print(str(entry), file=output)


# Types that are listed as mixed but that we don't really believe should be
mixed_exceptions = ["enumvalueType"]


def is_mixed(element) -> bool:
    if element.attrib["name"] in mixed_exceptions:
        return False
    return "mixed" in element.attrib


def is_simple_content(tag) -> bool:
    for child in tag:
        if child.tag == "{http://www.w3.org/2001/XMLSchema}simpleContent":
            return True

    return False


def create_group(element):

    name = convert_type_name(element.attrib["name"], False)

    for child in element:
        if child.tag == "{http://www.w3.org/2001/XMLSchema}choice":

            entries = []
            names = []
            for grandchild in child:
                if grandchild.tag == "{http://www.w3.org/2001/XMLSchema}group":
                    type_name = convert_type_name(grandchild.attrib["ref"], False)
                    entries.append(f"{type_name}({type_name})")

                if "name" in grandchild.attrib and "type" in grandchild.attrib:
                    entry_name = grandchild.attrib["name"]
                    entry_type = grandchild.attrib["type"]
                    entry_name = convert_type_name(entry_name, False)
                    entry_type = convert_type_name(entry_type, False)
                    if entry_name not in names:
                        names.append(entry_name)
                        entries.append(f"{entry_name}({entry_type})")

            entries = ",\n    ".join(entries)

            return [Enum(name, False, entries)]

    return []


def create_simple_content(tag, comment_lookup):
    name = convert_type_name(tag.attrib["name"], False)

    simple_content = None
    for child in tag:
        if child.tag == "{http://www.w3.org/2001/XMLSchema}simpleContent":
            simple_content = child

    if not simple_content:
        return []

    for child in simple_content:
        if child.tag == "{http://www.w3.org/2001/XMLSchema}extension":
            type = child.attrib["base"]
            attribute_fields = get_attribute_fields(child, name, comment_lookup)

            type = convert_type_name(type, True)
            fields = {"Attributes": attribute_fields, "Content": [f"pub content: {type}"]}

            return [Struct(name, fields)]

    return []


def create_struct(output, tag, comment_lookup):
    """
    Output a rust struct for a given xsd tag
    """

    name = convert_type_name(tag.attrib["name"], False)
    attribute_fields = get_attribute_fields(tag, name, comment_lookup)

    element_fields = []
    for child in tag:
        if child.tag == "{http://www.w3.org/2001/XMLSchema}sequence":
            element_fields.extend(get_element_fields(child, name, comment_lookup))

            for grandchild in child:
                if grandchild.tag == "{http://www.w3.org/2001/XMLSchema}sequence":
                    element_fields.extend(get_element_fields(grandchild, name, comment_lookup))

    return [Struct(name, {"Attributes": attribute_fields, "Elements": element_fields})]


def get_element_fields(element, name, comment_lookup):

    element_fields = []

    for child in element:
        if "name" in child.attrib:
            field_name = convert_field_name(child.attrib["name"])
            # Some childs don't have a 'type' attribute, I'm not sure if "String" is a reasonable fallback
            # but we'll try it until proven wrong
            field_type = (
                convert_type_name(child.attrib["type"], True)
                if "type" in child.attrib
                else "String"
            )

            comment_out = comment_lookup.get(name.lower(), {}).get(field_name.lower(), True)
            Wrapper = count_based_wrapper(child)
            element_fields.append(Field(field_name, Wrapper(field_type), comment_out))

    return element_fields


def count_based_wrapper(element):

    min_occurs = int(element.attrib["minOccurs"]) if "minOccurs" in element.attrib else 1
    max_occurs = element.attrib["maxOccurs"] if "maxOccurs" in element.attrib else 1

    if min_occurs == 0 and max_occurs in [1, "1"]:
        return Option
    elif min_occurs == 0 and max_occurs == "unbounded":
        return Vec
    elif min_occurs == 1 and max_occurs == "unbounded":
        return Vec1
    elif min_occurs == 1 and max_occurs in [1, "1"]:
        return PassThrough
    else:
        raise Exception(f"min:{repr(min_occurs)} max:{repr(max_occurs)}")


def create_mixed(output, tag, comment_lookup):
    """
    Output a rust enum for a mixed complex type
    """

    name = convert_type_name(tag.attrib["name"], False)
    attribute_fields = get_attribute_fields(tag, name, comment_lookup)

    entries = []
    for child in tag:
        if child.tag == "{http://www.w3.org/2001/XMLSchema}sequence":
            for grandchild in child:
                if "name" in grandchild.attrib and "type" in grandchild.attrib:
                    entry_name = grandchild.attrib["name"]
                    entry_type = grandchild.attrib["type"]
                    entry_name = convert_type_name(entry_name, False)
                    entry_type = convert_type_name(entry_type, False)
                    entries.append(f"{entry_name}({entry_type})")

            entries.append("Text(String)")

            entries = ",\n    ".join(entries)

        elif child.tag == "{http://www.w3.org/2001/XMLSchema}group":
            if "ref" in child.attrib:
                type_name = convert_type_name(child.attrib["ref"], False)
                entries.append(f"{type_name}({type_name})")
                entries.append("Text(String)")
                entries = ",\n    ".join(entries)

        elif child.tag == "{http://www.w3.org/2001/XMLSchema}choice":
            for grandchild in child:
                if "name" in grandchild.attrib and "type" in grandchild.attrib:
                    entry_name = grandchild.attrib["name"]
                    entry_type = grandchild.attrib["type"]
                    entry_name = convert_type_name(entry_name, False)
                    entry_type = convert_type_name(entry_type, False)
                    entries.append(f"{entry_name}({entry_type})")

            entries.append("Text(String)")

            entries = ",\n    ".join(entries)

    item_name = name + "Item"

    if entries:
        return [
            Struct(
                name,
                {"Attributes": attribute_fields, "Contents": [f"pub content: Vec<{item_name}>"]},
            ),
            Enum(item_name, False, entries),
        ]
    else:
        return [
            Struct(name, {"Attributes": attribute_fields, "Contents": [f"pub content: String"]})
        ]


def create_restriction(output, name, tag):
    """
    Output a rust enum for a given xsd tag
    """

    # Handle special cases which are patterns rather than enums
    if name in ["DoxVersionNumber", "DoxCharRange"]:
        return [Text(f"type {name} = String;")]

    # Skip weird type until we need it
    if name in ["DoxOlType", "DoxBool"]:
        return

    name = convert_type_name(name, False)

    entries = []
    for child in tag:
        entry_name = child.attrib["value"]
        entry_name = convert_enum_name(entry_name)
        entries.append(entry_name)

    entries = ",\n    ".join(entries)

    return [Enum(name, True, entries)]


def convert_enum_name(name):
    name = capitalize(name)
    while True:
        match = re.search("-[A-Za-z]", name)
        if match:
            span = match.span()
            name = name[: span[0]] + match.group()[1:].upper() + name[span[1] :]
        else:
            break

    name = name.replace("#", "Sharp")
    name = name.replace("+", "Plus")

    return name


def get_attribute_fields(element, name, comment_lookup):

    fields = []

    for child in element:
        if child.tag == "{http://www.w3.org/2001/XMLSchema}attribute":
            if "name" in child.attrib:
                field_name = convert_field_name(child.attrib["name"])
                field_type = convert_type_name(child.attrib["type"], True)

                comment_out = comment_lookup.get(name.lower(), {}).get(field_name.lower(), True)
                fields.append(Field(field_name, field_type, comment_out))

    return fields


def capitalize(name):
    return name[0].upper() + name[1:]


type_lookup = {
    "xsd:string": "String",
    "xsd:integer": "i32",
    "DoxBool": "bool",
    "DoxOlType": "String",
}


def convert_type_name(name, as_field_type):
    if name in type_lookup:
        return type_lookup[name]

    name = capitalize(name)
    name = name.replace("def", "Def")
    name = name.replace("type", "Type")
    name = name.replace("kind", "Kind")
    name = name.replace("class", "Class")
    name = name.replace("value", "Value")
    name = name.replace("param", "Param")
    name = name.replace("list", "List")
    name = name.replace("contents", "Contents")
    name = name.replace("ofC", "OfC")

    return name


field_lookup = {
    "compounddef": "compound_def",
    "compoundname": "compound_name",
    "sectiondef": "section_defs",
    "memberdef": "member_defs",
    "briefdescription": "brief_description",
    "detaileddescription": "detailed_description",
    "refid": "ref_id",
    "parameteritem": "parameter_item",
    "parametername": "parameter_name",
    "parametertype": "parameter_type",
    "parameterdescription": "parameter_description",
    "parameternamelist": "parameter_name_list",
}


def convert_field_name(name):
    keywords = ["ref", "type", "static", "const", "final", "abstract"]
    if name in keywords:
        return f"{name}_"

    name = field_lookup.get(name.lower(), name)
    return name


def generate_comment_lookup(filepath):
    """
    Parses the current rust file with regexes to track which fields in which structs have
    been commented out so we can put those comments in when re-creating the file. This means
    that we're free to comment fields during development and not have to deal with them
    being uncommented by the script if we need to regenerate anything.
    """

    lookup = defaultdict(dict)
    key = None

    struct_re = re.compile(r"pub struct (.*) \{")
    enum_re = re.compile(r"pub enum (.*) \{")
    field_re = re.compile(r"    pub (.*): .*")
    comment_field_re = re.compile(r"    // pub (.*): .*")

    with open(filepath) as f:
        for line in f:
            if match := struct_re.match(line):
                # Lower case entries to make them robust to future case changes
                key = match.group(1).lower()
            elif match := enum_re.match(line):
                key = None
            elif match := field_re.match(line):
                if key:
                    # Lower case entries to make them robust to future case changes
                    lookup[key][match.group(1).lower()] = False
            elif match := comment_field_re.match(line):
                if key:
                    # Lower case entries to make them robust to future case changes
                    lookup[key][match.group(1).lower()] = True

    return lookup


if __name__ == "__main__":
    main(sys.argv[1:])
