"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvocationType = exports.LambdaInvoke = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Invoke a Lambda function as a Task
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 */
class LambdaInvoke extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_LambdaInvokeProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LambdaInvoke);
            }
            throw error;
        }
        this.integrationPattern = props.integrationPattern ?? sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is required in `payload` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (props.payloadResponseOnly &&
            (props.integrationPattern || props.invocationType || props.clientContext || props.qualifier)) {
            throw new Error("The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified.");
        }
        this.taskMetrics = {
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: {
                LambdaFunctionArn: this.props.lambdaFunction.functionArn,
                ...(this.props.qualifier && { Qualifier: this.props.qualifier }),
            },
        };
        this.taskPolicies = [
            new iam.PolicyStatement({
                resources: this.props.lambdaFunction.resourceArnsForGrantInvoke,
                actions: ['lambda:InvokeFunction'],
            }),
        ];
        if (props.retryOnServiceExceptions ?? true) {
            // Best practice from https://docs.aws.amazon.com/step-functions/latest/dg/bp-lambda-serviceexception.html
            this.addRetry({
                errors: ['Lambda.ServiceException', 'Lambda.AWSLambdaException', 'Lambda.SdkClientException'],
                interval: cdk.Duration.seconds(2),
                maxAttempts: 6,
                backoffRate: 2,
            });
        }
    }
    /**
     * Provides the Lambda Invoke service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        if (this.props.payloadResponseOnly) {
            return {
                Resource: this.props.lambdaFunction.functionArn,
                ...this.props.payload && { Parameters: sfn.FieldUtils.renderObject(this.props.payload.value) },
            };
        }
        else {
            return {
                Resource: task_utils_1.integrationResourceArn('lambda', 'invoke', this.integrationPattern),
                Parameters: sfn.FieldUtils.renderObject({
                    FunctionName: this.props.lambdaFunction.functionArn,
                    Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromJsonPathAt('$').value,
                    InvocationType: this.props.invocationType,
                    ClientContext: this.props.clientContext,
                    Qualifier: this.props.qualifier,
                }),
            };
        }
    }
}
exports.LambdaInvoke = LambdaInvoke;
_a = JSII_RTTI_SYMBOL_1;
LambdaInvoke[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.LambdaInvoke", version: "1.182.0" };
LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
/**
 * Invocation type of a Lambda
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * Invoke the function synchronously.
     *
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke the function asynchronously.
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * Validate parameter values and verify that the user or role has permission to invoke the function.
     */
    LambdaInvocationType["DRY_RUN"] = "DryRun";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW52b2tlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiaW52b2tlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7OztBQUFBLHdDQUF3QztBQUV4QyxrREFBa0Q7QUFDbEQscUNBQXFDO0FBRXJDLHNEQUF5RjtBQXVFekY7Ozs7R0FJRztBQUNILE1BQWEsWUFBYSxTQUFRLEdBQUcsQ0FBQyxhQUFhO0lBWWpELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQW1CLEtBQXdCO1FBQ2pGLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRGlDLFVBQUssR0FBTCxLQUFLLENBQW1COzs7Ozs7K0NBWnhFLFlBQVk7Ozs7UUFjckIsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxHQUFHLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCLENBQUM7UUFFOUYscUNBQXdCLENBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFLFlBQVksQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBRS9GLElBQUksSUFBSSxDQUFDLGtCQUFrQixLQUFLLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUI7ZUFDckUsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUNyRCxNQUFNLElBQUksS0FBSyxDQUFDLDRGQUE0RixDQUFDLENBQUM7U0FDL0c7UUFFRCxJQUFJLEtBQUssQ0FBQyxtQkFBbUI7WUFDM0IsQ0FBQyxLQUFLLENBQUMsa0JBQWtCLElBQUksS0FBSyxDQUFDLGNBQWMsSUFBSSxLQUFLLENBQUMsYUFBYSxJQUFJLEtBQUssQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUM5RixNQUFNLElBQUksS0FBSyxDQUNiLDZJQUE2SSxDQUM5SSxDQUFDO1NBQ0g7UUFFRCxJQUFJLENBQUMsV0FBVyxHQUFHO1lBQ2pCLG9CQUFvQixFQUFFLGdCQUFnQjtZQUN0QyxrQkFBa0IsRUFBRSxpQkFBaUI7WUFDckMsZ0JBQWdCLEVBQUU7Z0JBQ2hCLGlCQUFpQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLFdBQVc7Z0JBQ3hELEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsSUFBSSxFQUFFLFNBQVMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxDQUFDO2FBQ2pFO1NBQ0YsQ0FBQztRQUVGLElBQUksQ0FBQyxZQUFZLEdBQUc7WUFDbEIsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixTQUFTLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsMEJBQTBCO2dCQUMvRCxPQUFPLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQzthQUNuQyxDQUFDO1NBQ0gsQ0FBQztRQUVGLElBQUksS0FBSyxDQUFDLHdCQUF3QixJQUFJLElBQUksRUFBRTtZQUMxQywwR0FBMEc7WUFDMUcsSUFBSSxDQUFDLFFBQVEsQ0FBQztnQkFDWixNQUFNLEVBQUUsQ0FBQyx5QkFBeUIsRUFBRSwyQkFBMkIsRUFBRSwyQkFBMkIsQ0FBQztnQkFDN0YsUUFBUSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztnQkFDakMsV0FBVyxFQUFFLENBQUM7Z0JBQ2QsV0FBVyxFQUFFLENBQUM7YUFDZixDQUFDLENBQUM7U0FDSjtLQUNGO0lBRUQ7O09BRUc7SUFDSDs7T0FFRztJQUNPLFdBQVc7UUFDbkIsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixFQUFFO1lBQ2xDLE9BQU87Z0JBQ0wsUUFBUSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLFdBQVc7Z0JBQy9DLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLElBQUksRUFBRSxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUU7YUFDL0YsQ0FBQztTQUNIO2FBQU07WUFDTCxPQUFPO2dCQUNMLFFBQVEsRUFBRSxtQ0FBc0IsQ0FBQyxRQUFRLEVBQUUsUUFBUSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQztnQkFDN0UsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDO29CQUN0QyxZQUFZLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsV0FBVztvQkFDbkQsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEtBQUs7b0JBQ2hHLGNBQWMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWM7b0JBQ3pDLGFBQWEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWE7b0JBQ3ZDLFNBQVMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVM7aUJBQ2hDLENBQUM7YUFDSCxDQUFDO1NBQ0g7S0FDRjs7QUFqRkgsb0NBa0ZDOzs7QUFoRnlCLDJDQUE4QixHQUE2QjtJQUNqRixHQUFHLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCO0lBQ3ZDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUI7Q0FDM0MsQ0FBQztBQStFSjs7R0FFRztBQUNILElBQVksb0JBcUJYO0FBckJELFdBQVksb0JBQW9CO0lBQzlCOzs7OztPQUtHO0lBQ0gsNERBQW9DLENBQUE7SUFFcEM7Ozs7O09BS0c7SUFDSCx1Q0FBZSxDQUFBO0lBRWY7O09BRUc7SUFDSCwwQ0FBa0IsQ0FBQTtBQUNwQixDQUFDLEVBckJXLG9CQUFvQixHQUFwQiw0QkFBb0IsS0FBcEIsNEJBQW9CLFFBcUIvQiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IGludGVncmF0aW9uUmVzb3VyY2VBcm4sIHZhbGlkYXRlUGF0dGVyblN1cHBvcnRlZCB9IGZyb20gJy4uL3ByaXZhdGUvdGFzay11dGlscyc7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgaW52b2tpbmcgYSBMYW1iZGEgZnVuY3Rpb24gd2l0aCBMYW1iZGFJbnZva2VcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFJbnZva2VQcm9wcyBleHRlbmRzIHNmbi5UYXNrU3RhdGVCYXNlUHJvcHMge1xuXG4gIC8qKlxuICAgKiBMYW1iZGEgZnVuY3Rpb24gdG8gaW52b2tlXG4gICAqL1xuICByZWFkb25seSBsYW1iZGFGdW5jdGlvbjogbGFtYmRhLklGdW5jdGlvbjtcblxuICAvKipcbiAgICogVGhlIEpTT04gdGhhdCB3aWxsIGJlIHN1cHBsaWVkIGFzIGlucHV0IHRvIHRoZSBMYW1iZGEgZnVuY3Rpb25cbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgc3RhdGUgaW5wdXQgKEpTT04gcGF0aCAnJCcpXG4gICAqL1xuICByZWFkb25seSBwYXlsb2FkPzogc2ZuLlRhc2tJbnB1dDtcblxuICAvKipcbiAgICogSW52b2NhdGlvbiB0eXBlIG9mIHRoZSBMYW1iZGEgZnVuY3Rpb25cbiAgICpcbiAgICogQGRlZmF1bHQgSW52b2NhdGlvblR5cGUuUkVRVUVTVF9SRVNQT05TRVxuICAgKi9cbiAgcmVhZG9ubHkgaW52b2NhdGlvblR5cGU/OiBMYW1iZGFJbnZvY2F0aW9uVHlwZTtcblxuICAvKipcbiAgICogVXAgdG8gMzU4MyBieXRlcyBvZiBiYXNlNjQtZW5jb2RlZCBkYXRhIGFib3V0IHRoZSBpbnZva2luZyBjbGllbnRcbiAgICogdG8gcGFzcyB0byB0aGUgZnVuY3Rpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gY29udGV4dFxuICAgKi9cbiAgcmVhZG9ubHkgY2xpZW50Q29udGV4dD86IHN0cmluZztcblxuICAvKipcbiAgICogVmVyc2lvbiBvciBhbGlhcyB0byBpbnZva2UgYSBwdWJsaXNoZWQgdmVyc2lvbiBvZiB0aGUgZnVuY3Rpb25cbiAgICpcbiAgICogWW91IG9ubHkgbmVlZCB0byBzdXBwbHkgdGhpcyBpZiB5b3Ugd2FudCB0aGUgdmVyc2lvbiBvZiB0aGUgTGFtYmRhIEZ1bmN0aW9uIHRvIGRlcGVuZFxuICAgKiBvbiBkYXRhIGluIHRoZSBzdGF0ZSBtYWNoaW5lIHN0YXRlLiBJZiBub3QsIHlvdSBjYW4gcGFzcyB0aGUgYXBwcm9wcmlhdGUgQWxpYXMgb3IgVmVyc2lvbiBvYmplY3RcbiAgICogZGlyZWN0bHkgYXMgdGhlIGBsYW1iZGFGdW5jdGlvbmAgYXJndW1lbnQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVmVyc2lvbiBvciBhbGlhcyBpbmhlcmVudCB0byB0aGUgYGxhbWJkYUZ1bmN0aW9uYCBvYmplY3QuXG4gICAqIEBkZXByZWNhdGVkIHBhc3MgYSBWZXJzaW9uIG9yIEFsaWFzIG9iamVjdCBhcyBsYW1iZGFGdW5jdGlvbiBpbnN0ZWFkXG4gICAqL1xuICByZWFkb25seSBxdWFsaWZpZXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEludm9rZSB0aGUgTGFtYmRhIGluIGEgd2F5IHRoYXQgb25seSByZXR1cm5zIHRoZSBwYXlsb2FkIHJlc3BvbnNlIHdpdGhvdXQgYWRkaXRpb25hbCBtZXRhZGF0YS5cbiAgICpcbiAgICogVGhlIGBwYXlsb2FkUmVzcG9uc2VPbmx5YCBwcm9wZXJ0eSBjYW5ub3QgYmUgdXNlZCBpZiBgaW50ZWdyYXRpb25QYXR0ZXJuYCwgYGludm9jYXRpb25UeXBlYCxcbiAgICogYGNsaWVudENvbnRleHRgLCBvciBgcXVhbGlmaWVyYCBhcmUgc3BlY2lmaWVkLlxuICAgKiBJdCBhbHdheXMgdXNlcyB0aGUgUkVRVUVTVF9SRVNQT05TRSBiZWhhdmlvci5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHBheWxvYWRSZXNwb25zZU9ubHk/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIHJldHJ5IG9uIExhbWJkYSBzZXJ2aWNlIGV4Y2VwdGlvbnMuXG4gICAqXG4gICAqIFRoaXMgaGFuZGxlcyBgTGFtYmRhLlNlcnZpY2VFeGNlcHRpb25gLCBgTGFtYmRhLkFXU0xhbWJkYUV4Y2VwdGlvbmAgYW5kXG4gICAqIGBMYW1iZGEuU2RrQ2xpZW50RXhjZXB0aW9uYCB3aXRoIGFuIGludGVydmFsIG9mIDIgc2Vjb25kcywgYSBiYWNrLW9mZiByYXRlXG4gICAqIG9mIDIgYW5kIDYgbWF4aW11bSBhdHRlbXB0cy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL2JwLWxhbWJkYS1zZXJ2aWNlZXhjZXB0aW9uLmh0bWxcbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcmV0cnlPblNlcnZpY2VFeGNlcHRpb25zPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBJbnZva2UgYSBMYW1iZGEgZnVuY3Rpb24gYXMgYSBUYXNrXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL2Nvbm5lY3QtbGFtYmRhLmh0bWxcbiAqL1xuZXhwb3J0IGNsYXNzIExhbWJkYUludm9rZSBleHRlbmRzIHNmbi5UYXNrU3RhdGVCYXNlIHtcblxuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBTVVBQT1JURURfSU5URUdSQVRJT05fUEFUVEVSTlM6IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm5bXSA9IFtcbiAgICBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJFUVVFU1RfUkVTUE9OU0UsXG4gICAgc2ZuLkludGVncmF0aW9uUGF0dGVybi5XQUlUX0ZPUl9UQVNLX1RPS0VOLFxuICBdO1xuXG4gIHByb3RlY3RlZCByZWFkb25seSB0YXNrTWV0cmljcz86IHNmbi5UYXNrTWV0cmljc0NvbmZpZztcbiAgcHJvdGVjdGVkIHJlYWRvbmx5IHRhc2tQb2xpY2llcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXTtcblxuICBwcml2YXRlIHJlYWRvbmx5IGludGVncmF0aW9uUGF0dGVybjogc2ZuLkludGVncmF0aW9uUGF0dGVybjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBMYW1iZGFJbnZva2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgIHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuID0gcHJvcHMuaW50ZWdyYXRpb25QYXR0ZXJuID8/IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUkVRVUVTVF9SRVNQT05TRTtcblxuICAgIHZhbGlkYXRlUGF0dGVyblN1cHBvcnRlZCh0aGlzLmludGVncmF0aW9uUGF0dGVybiwgTGFtYmRhSW52b2tlLlNVUFBPUlRFRF9JTlRFR1JBVElPTl9QQVRURVJOUyk7XG5cbiAgICBpZiAodGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4gPT09IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uV0FJVF9GT1JfVEFTS19UT0tFTlxuICAgICAgJiYgIXNmbi5GaWVsZFV0aWxzLmNvbnRhaW5zVGFza1Rva2VuKHByb3BzLnBheWxvYWQpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Rhc2sgVG9rZW4gaXMgcmVxdWlyZWQgaW4gYHBheWxvYWRgIGZvciBjYWxsYmFjay4gVXNlIEpzb25QYXRoLnRhc2tUb2tlbiB0byBzZXQgdGhlIHRva2VuLicpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5wYXlsb2FkUmVzcG9uc2VPbmx5ICYmXG4gICAgICAocHJvcHMuaW50ZWdyYXRpb25QYXR0ZXJuIHx8IHByb3BzLmludm9jYXRpb25UeXBlIHx8IHByb3BzLmNsaWVudENvbnRleHQgfHwgcHJvcHMucXVhbGlmaWVyKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIlRoZSAncGF5bG9hZFJlc3BvbnNlT25seScgcHJvcGVydHkgY2Fubm90IGJlIHVzZWQgaWYgJ2ludGVncmF0aW9uUGF0dGVybicsICdpbnZvY2F0aW9uVHlwZScsICdjbGllbnRDb250ZXh0Jywgb3IgJ3F1YWxpZmllcicgYXJlIHNwZWNpZmllZC5cIixcbiAgICAgICk7XG4gICAgfVxuXG4gICAgdGhpcy50YXNrTWV0cmljcyA9IHtcbiAgICAgIG1ldHJpY1ByZWZpeFNpbmd1bGFyOiAnTGFtYmRhRnVuY3Rpb24nLFxuICAgICAgbWV0cmljUHJlZml4UGx1cmFsOiAnTGFtYmRhRnVuY3Rpb25zJyxcbiAgICAgIG1ldHJpY0RpbWVuc2lvbnM6IHtcbiAgICAgICAgTGFtYmRhRnVuY3Rpb25Bcm46IHRoaXMucHJvcHMubGFtYmRhRnVuY3Rpb24uZnVuY3Rpb25Bcm4sXG4gICAgICAgIC4uLih0aGlzLnByb3BzLnF1YWxpZmllciAmJiB7IFF1YWxpZmllcjogdGhpcy5wcm9wcy5xdWFsaWZpZXIgfSksXG4gICAgICB9LFxuICAgIH07XG5cbiAgICB0aGlzLnRhc2tQb2xpY2llcyA9IFtcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgcmVzb3VyY2VzOiB0aGlzLnByb3BzLmxhbWJkYUZ1bmN0aW9uLnJlc291cmNlQXJuc0ZvckdyYW50SW52b2tlLFxuICAgICAgICBhY3Rpb25zOiBbJ2xhbWJkYTpJbnZva2VGdW5jdGlvbiddLFxuICAgICAgfSksXG4gICAgXTtcblxuICAgIGlmIChwcm9wcy5yZXRyeU9uU2VydmljZUV4Y2VwdGlvbnMgPz8gdHJ1ZSkge1xuICAgICAgLy8gQmVzdCBwcmFjdGljZSBmcm9tIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdGVwLWZ1bmN0aW9ucy9sYXRlc3QvZGcvYnAtbGFtYmRhLXNlcnZpY2VleGNlcHRpb24uaHRtbFxuICAgICAgdGhpcy5hZGRSZXRyeSh7XG4gICAgICAgIGVycm9yczogWydMYW1iZGEuU2VydmljZUV4Y2VwdGlvbicsICdMYW1iZGEuQVdTTGFtYmRhRXhjZXB0aW9uJywgJ0xhbWJkYS5TZGtDbGllbnRFeGNlcHRpb24nXSxcbiAgICAgICAgaW50ZXJ2YWw6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDIpLFxuICAgICAgICBtYXhBdHRlbXB0czogNixcbiAgICAgICAgYmFja29mZlJhdGU6IDIsXG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogUHJvdmlkZXMgdGhlIExhbWJkYSBJbnZva2Ugc2VydmljZSBpbnRlZ3JhdGlvbiB0YXNrIGNvbmZpZ3VyYXRpb25cbiAgICovXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHByb3RlY3RlZCBfcmVuZGVyVGFzaygpOiBhbnkge1xuICAgIGlmICh0aGlzLnByb3BzLnBheWxvYWRSZXNwb25zZU9ubHkpIHtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIFJlc291cmNlOiB0aGlzLnByb3BzLmxhbWJkYUZ1bmN0aW9uLmZ1bmN0aW9uQXJuLFxuICAgICAgICAuLi50aGlzLnByb3BzLnBheWxvYWQgJiYgeyBQYXJhbWV0ZXJzOiBzZm4uRmllbGRVdGlscy5yZW5kZXJPYmplY3QodGhpcy5wcm9wcy5wYXlsb2FkLnZhbHVlKSB9LFxuICAgICAgfTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgUmVzb3VyY2U6IGludGVncmF0aW9uUmVzb3VyY2VBcm4oJ2xhbWJkYScsICdpbnZva2UnLCB0aGlzLmludGVncmF0aW9uUGF0dGVybiksXG4gICAgICAgIFBhcmFtZXRlcnM6IHNmbi5GaWVsZFV0aWxzLnJlbmRlck9iamVjdCh7XG4gICAgICAgICAgRnVuY3Rpb25OYW1lOiB0aGlzLnByb3BzLmxhbWJkYUZ1bmN0aW9uLmZ1bmN0aW9uQXJuLFxuICAgICAgICAgIFBheWxvYWQ6IHRoaXMucHJvcHMucGF5bG9hZCA/IHRoaXMucHJvcHMucGF5bG9hZC52YWx1ZSA6IHNmbi5UYXNrSW5wdXQuZnJvbUpzb25QYXRoQXQoJyQnKS52YWx1ZSxcbiAgICAgICAgICBJbnZvY2F0aW9uVHlwZTogdGhpcy5wcm9wcy5pbnZvY2F0aW9uVHlwZSxcbiAgICAgICAgICBDbGllbnRDb250ZXh0OiB0aGlzLnByb3BzLmNsaWVudENvbnRleHQsXG4gICAgICAgICAgUXVhbGlmaWVyOiB0aGlzLnByb3BzLnF1YWxpZmllcixcbiAgICAgICAgfSksXG4gICAgICB9O1xuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIEludm9jYXRpb24gdHlwZSBvZiBhIExhbWJkYVxuICovXG5leHBvcnQgZW51bSBMYW1iZGFJbnZvY2F0aW9uVHlwZSB7XG4gIC8qKlxuICAgKiBJbnZva2UgdGhlIGZ1bmN0aW9uIHN5bmNocm9ub3VzbHkuXG4gICAqXG4gICAqIEtlZXAgdGhlIGNvbm5lY3Rpb24gb3BlbiB1bnRpbCB0aGUgZnVuY3Rpb24gcmV0dXJucyBhIHJlc3BvbnNlIG9yIHRpbWVzIG91dC5cbiAgICogVGhlIEFQSSByZXNwb25zZSBpbmNsdWRlcyB0aGUgZnVuY3Rpb24gcmVzcG9uc2UgYW5kIGFkZGl0aW9uYWwgZGF0YS5cbiAgICovXG4gIFJFUVVFU1RfUkVTUE9OU0UgPSAnUmVxdWVzdFJlc3BvbnNlJyxcblxuICAvKipcbiAgICogSW52b2tlIHRoZSBmdW5jdGlvbiBhc3luY2hyb25vdXNseS5cbiAgICpcbiAgICogU2VuZCBldmVudHMgdGhhdCBmYWlsIG11bHRpcGxlIHRpbWVzIHRvIHRoZSBmdW5jdGlvbidzIGRlYWQtbGV0dGVyIHF1ZXVlIChpZiBpdCdzIGNvbmZpZ3VyZWQpLlxuICAgKiBUaGUgQVBJIHJlc3BvbnNlIG9ubHkgaW5jbHVkZXMgYSBzdGF0dXMgY29kZS5cbiAgICovXG4gIEVWRU5UID0gJ0V2ZW50JyxcblxuICAvKipcbiAgICogVmFsaWRhdGUgcGFyYW1ldGVyIHZhbHVlcyBhbmQgdmVyaWZ5IHRoYXQgdGhlIHVzZXIgb3Igcm9sZSBoYXMgcGVybWlzc2lvbiB0byBpbnZva2UgdGhlIGZ1bmN0aW9uLlxuICAgKi9cbiAgRFJZX1JVTiA9ICdEcnlSdW4nXG59XG4iXX0=