"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateTransformJob = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
const utils_1 = require("./private/utils");
/**
 * Class representing the SageMaker Create Transform Job task.
 *
 */
class SageMakerCreateTransformJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_SageMakerCreateTransformJobProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SageMakerCreateTransformJob);
            }
            throw error;
        }
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateTransformJob.SUPPORTED_INTEGRATION_PATTERNS);
        // set the sagemaker role or create new one
        if (props.role) {
            this._role = props.role;
        }
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.transformInput = props.transformInput.transformDataSource.s3DataSource.s3DataType
            ? props.transformInput
            : Object.assign({}, props.transformInput, {
                transformDataSource: { s3DataSource: { ...props.transformInput.transformDataSource.s3DataSource, s3DataType: base_types_1.S3DataType.S3_PREFIX } },
            });
        // set the default value for the transform resources
        this.transformResources = props.transformResources || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
        };
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createTransformJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    /**
     * The execution role for the Sagemaker transform job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    renderParameters() {
        return {
            ...(this.props.batchStrategy ? { BatchStrategy: this.props.batchStrategy } : {}),
            ...utils_1.renderEnvironment(this.props.environment),
            ...(this.props.maxConcurrentTransforms ? { MaxConcurrentTransforms: this.props.maxConcurrentTransforms } : {}),
            ...(this.props.maxPayload ? { MaxPayloadInMB: this.props.maxPayload.toMebibytes() } : {}),
            ...this.props.modelClientOptions ? this.renderModelClientOptions(this.props.modelClientOptions) : {},
            ModelName: this.props.modelName,
            ...utils_1.renderTags(this.props.tags),
            ...this.renderTransformInput(this.transformInput),
            TransformJobName: this.props.transformJobName,
            ...this.renderTransformOutput(this.props.transformOutput),
            ...this.renderTransformResources(this.transformResources),
        };
    }
    renderModelClientOptions(options) {
        const retries = options.invocationsMaxRetries;
        if (!core_1.Token.isUnresolved(retries) && retries ? (retries < 0 || retries > 3) : false) {
            throw new Error(`invocationsMaxRetries should be between 0 and 3. Received: ${retries}.`);
        }
        const timeout = options.invocationsTimeout?.toSeconds();
        if (!core_1.Token.isUnresolved(timeout) && timeout ? (timeout < 1 || timeout > 3600) : false) {
            throw new Error(`invocationsTimeout should be between 1 and 3600 seconds. Received: ${timeout}.`);
        }
        return {
            ModelClientConfig: {
                InvocationsMaxRetries: retries ?? 0,
                InvocationsTimeoutInSeconds: timeout ?? 60,
            },
        };
    }
    renderTransformInput(input) {
        return {
            TransformInput: {
                ...(input.compressionType ? { CompressionType: input.compressionType } : {}),
                ...(input.contentType ? { ContentType: input.contentType } : {}),
                DataSource: {
                    S3DataSource: {
                        S3Uri: input.transformDataSource.s3DataSource.s3Uri,
                        S3DataType: input.transformDataSource.s3DataSource.s3DataType,
                    },
                },
                ...(input.splitType ? { SplitType: input.splitType } : {}),
            },
        };
    }
    renderTransformOutput(output) {
        return {
            TransformOutput: {
                S3OutputPath: output.s3OutputPath,
                ...(output.encryptionKey ? { KmsKeyId: output.encryptionKey.keyArn } : {}),
                ...(output.accept ? { Accept: output.accept } : {}),
                ...(output.assembleWith ? { AssembleWith: output.assembleWith } : {}),
            },
        };
    }
    renderTransformResources(resources) {
        return {
            TransformResources: {
                InstanceCount: resources.instanceCount,
                InstanceType: sfn.JsonPath.isEncodedJsonPath(resources.instanceType.toString())
                    ? resources.instanceType.toString() : `ml.${resources.instanceType}`,
                ...(resources.volumeEncryptionKey ? { VolumeKmsKeyId: resources.volumeEncryptionKey.keyArn } : {}),
            },
        };
    }
    makePolicyStatements() {
        const stack = core_1.Stack.of(this);
        // create new role if doesn't exist
        if (this._role === undefined) {
            this._role = new iam.Role(this, 'SagemakerTransformRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess')],
            });
        }
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTransformJob', 'sagemaker:DescribeTransformJob', 'sagemaker:StopTransformJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'transform-job',
                        resourceName: '*',
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTransformJobsRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.SageMakerCreateTransformJob = SageMakerCreateTransformJob;
_a = JSII_RTTI_SYMBOL_1;
SageMakerCreateTransformJob[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.SageMakerCreateTransformJob", version: "1.182.0" };
SageMakerCreateTransformJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3JlYXRlLXRyYW5zZm9ybS1qb2IuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjcmVhdGUtdHJhbnNmb3JtLWpvYi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSx3Q0FBd0M7QUFDeEMsd0NBQXdDO0FBQ3hDLGtEQUFrRDtBQUNsRCx3Q0FBbUQ7QUFFbkQsc0RBQXlGO0FBQ3pGLDZDQUFrSTtBQUNsSSwyQ0FBZ0U7QUFxRmhFOzs7R0FHRztBQUNILE1BQWEsMkJBQTRCLFNBQVEsR0FBRyxDQUFDLGFBQWE7SUFxQmhFLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQW1CLEtBQXVDO1FBQ2hHLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRGlDLFVBQUssR0FBTCxLQUFLLENBQWtDOzs7Ozs7K0NBckJ2RiwyQkFBMkI7Ozs7UUF1QnBDLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLElBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQixDQUFDO1FBQzlGLHFDQUF3QixDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSwyQkFBMkIsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBRTlHLDJDQUEyQztRQUMzQyxJQUFJLEtBQUssQ0FBQyxJQUFJLEVBQUU7WUFDZCxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7U0FDekI7UUFFRCx3RkFBd0Y7UUFDeEYsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDLG1CQUFtQixDQUFDLFlBQVksQ0FBQyxVQUFVO1lBQ3BGLENBQUMsQ0FBQyxLQUFLLENBQUMsY0FBYztZQUN0QixDQUFDLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsS0FBSyxDQUFDLGNBQWMsRUFBRTtnQkFDeEMsbUJBQW1CLEVBQUUsRUFBRSxZQUFZLEVBQUUsRUFBRSxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUMsbUJBQW1CLENBQUMsWUFBWSxFQUFFLFVBQVUsRUFBRSx1QkFBVSxDQUFDLFNBQVMsRUFBRSxFQUFFO2FBQ3RJLENBQUMsQ0FBQztRQUVMLG9EQUFvRDtRQUNwRCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJO1lBQ3BELGFBQWEsRUFBRSxDQUFDO1lBQ2hCLFlBQVksRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQztTQUNqRixDQUFDO1FBRUYsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsb0JBQW9CLEVBQUUsQ0FBQztLQUNqRDtJQUVEOztPQUVHO0lBQ08sV0FBVztRQUNuQixPQUFPO1lBQ0wsUUFBUSxFQUFFLG1DQUFzQixDQUFDLFdBQVcsRUFBRSxvQkFBb0IsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUM7WUFDNUYsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO1NBQ2pFLENBQUM7S0FDSDtJQUVEOzs7O09BSUc7SUFDSCxJQUFXLElBQUk7UUFDYixJQUFJLElBQUksQ0FBQyxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQzVCLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQztTQUMzRTtRQUNELE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQztLQUNuQjtJQUVPLGdCQUFnQjtRQUN0QixPQUFPO1lBQ0wsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxFQUFFLGFBQWEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7WUFDaEYsR0FBRyx5QkFBaUIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQztZQUM1QyxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDLENBQUMsRUFBRSx1QkFBdUIsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLHVCQUF1QixFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztZQUM5RyxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLEVBQUUsY0FBYyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztZQUN6RixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDcEcsU0FBUyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUztZQUMvQixHQUFHLGtCQUFVLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUM7WUFDOUIsR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQztZQUNqRCxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQjtZQUM3QyxHQUFHLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQztZQUN6RCxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUM7U0FDMUQsQ0FBQztLQUNIO0lBRU8sd0JBQXdCLENBQUMsT0FBMkI7UUFDMUQsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLHFCQUFxQixDQUFDO1FBQzlDLElBQUksQ0FBQyxZQUFLLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxJQUFJLE9BQU8sQ0FBQSxDQUFDLENBQUMsQ0FBQyxPQUFPLEdBQUcsQ0FBQyxJQUFJLE9BQU8sR0FBRyxDQUFDLENBQUMsQ0FBQSxDQUFDLENBQUMsS0FBSyxFQUFFO1lBQ2hGLE1BQU0sSUFBSSxLQUFLLENBQUMsOERBQThELE9BQU8sR0FBRyxDQUFDLENBQUM7U0FDM0Y7UUFDRCxNQUFNLE9BQU8sR0FBRyxPQUFPLENBQUMsa0JBQWtCLEVBQUUsU0FBUyxFQUFFLENBQUM7UUFDeEQsSUFBSSxDQUFDLFlBQUssQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLElBQUksT0FBTyxDQUFBLENBQUMsQ0FBQyxDQUFDLE9BQU8sR0FBRyxDQUFDLElBQUksT0FBTyxHQUFHLElBQUksQ0FBQyxDQUFBLENBQUMsQ0FBQyxLQUFLLEVBQUU7WUFDbkYsTUFBTSxJQUFJLEtBQUssQ0FBQyxzRUFBc0UsT0FBTyxHQUFHLENBQUMsQ0FBQztTQUNuRztRQUNELE9BQU87WUFDTCxpQkFBaUIsRUFBRTtnQkFDakIscUJBQXFCLEVBQUUsT0FBTyxJQUFJLENBQUM7Z0JBQ25DLDJCQUEyQixFQUFFLE9BQU8sSUFBSSxFQUFFO2FBQzNDO1NBQ0YsQ0FBQztLQUNIO0lBRU8sb0JBQW9CLENBQUMsS0FBcUI7UUFDaEQsT0FBTztZQUNMLGNBQWMsRUFBRTtnQkFDZCxHQUFHLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsRUFBRSxlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7Z0JBQzVFLEdBQUcsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxFQUFFLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztnQkFDaEUsVUFBVSxFQUFFO29CQUNWLFlBQVksRUFBRTt3QkFDWixLQUFLLEVBQUUsS0FBSyxDQUFDLG1CQUFtQixDQUFDLFlBQVksQ0FBQyxLQUFLO3dCQUNuRCxVQUFVLEVBQUUsS0FBSyxDQUFDLG1CQUFtQixDQUFDLFlBQVksQ0FBQyxVQUFVO3FCQUM5RDtpQkFDRjtnQkFDRCxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7YUFDM0Q7U0FDRixDQUFDO0tBQ0g7SUFFTyxxQkFBcUIsQ0FBQyxNQUF1QjtRQUNuRCxPQUFPO1lBQ0wsZUFBZSxFQUFFO2dCQUNmLFlBQVksRUFBRSxNQUFNLENBQUMsWUFBWTtnQkFDakMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLEVBQUUsUUFBUSxFQUFFLE1BQU0sQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztnQkFDMUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsTUFBTSxFQUFFLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2dCQUNuRCxHQUFHLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsRUFBRSxZQUFZLEVBQUUsTUFBTSxDQUFDLFlBQVksRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7YUFDdEU7U0FDRixDQUFDO0tBQ0g7SUFFTyx3QkFBd0IsQ0FBQyxTQUE2QjtRQUM1RCxPQUFPO1lBQ0wsa0JBQWtCLEVBQUU7Z0JBQ2xCLGFBQWEsRUFBRSxTQUFTLENBQUMsYUFBYTtnQkFDdEMsWUFBWSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztvQkFDN0UsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sU0FBUyxDQUFDLFlBQVksRUFBRTtnQkFDdEUsR0FBRyxDQUFDLFNBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLENBQUMsRUFBRSxjQUFjLEVBQUUsU0FBUyxDQUFDLG1CQUFtQixDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7YUFDbkc7U0FDRixDQUFDO0tBQ0g7SUFFTyxvQkFBb0I7UUFDMUIsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU3QixtQ0FBbUM7UUFDbkMsSUFBSSxJQUFJLENBQUMsS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUM1QixJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsd0JBQXdCLEVBQUU7Z0JBQ3hELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyx5QkFBeUIsQ0FBQztnQkFDOUQsZUFBZSxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO2FBQzNGLENBQUMsQ0FBQztTQUNKO1FBRUQsMEVBQTBFO1FBQzFFLE1BQU0sZ0JBQWdCLEdBQUc7WUFDdkIsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixPQUFPLEVBQUUsQ0FBQyw4QkFBOEIsRUFBRSxnQ0FBZ0MsRUFBRSw0QkFBNEIsQ0FBQztnQkFDekcsU0FBUyxFQUFFO29CQUNULEtBQUssQ0FBQyxTQUFTLENBQUM7d0JBQ2QsT0FBTyxFQUFFLFdBQVc7d0JBQ3BCLFFBQVEsRUFBRSxlQUFlO3dCQUN6QixZQUFZLEVBQUUsR0FBRztxQkFDbEIsQ0FBQztpQkFDSDthQUNGLENBQUM7WUFDRixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3RCLE9BQU8sRUFBRSxDQUFDLG9CQUFvQixDQUFDO2dCQUMvQixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7YUFDakIsQ0FBQztZQUNGLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDdEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO2dCQUN6QixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQztnQkFDOUIsVUFBVSxFQUFFO29CQUNWLFlBQVksRUFBRSxFQUFFLHFCQUFxQixFQUFFLHlCQUF5QixFQUFFO2lCQUNuRTthQUNGLENBQUM7U0FDSCxDQUFDO1FBRUYsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEtBQUssR0FBRyxDQUFDLGtCQUFrQixDQUFDLE9BQU8sRUFBRTtZQUM5RCxnQkFBZ0IsQ0FBQyxJQUFJLENBQ25CLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDdEIsT0FBTyxFQUFFLENBQUMsbUJBQW1CLEVBQUUsZ0JBQWdCLEVBQUUscUJBQXFCLENBQUM7Z0JBQ3ZFLFNBQVMsRUFBRTtvQkFDVCxLQUFLLENBQUMsU0FBUyxDQUFDO3dCQUNkLE9BQU8sRUFBRSxRQUFRO3dCQUNqQixRQUFRLEVBQUUsTUFBTTt3QkFDaEIsWUFBWSxFQUFFLHFEQUFxRDtxQkFDcEUsQ0FBQztpQkFDSDthQUNGLENBQUMsQ0FDSCxDQUFDO1NBQ0g7UUFFRCxPQUFPLGdCQUFnQixDQUFDO0tBQ3pCOztBQWhNSCxrRUFpTUM7OztBQWhNeUIsMERBQThCLEdBQTZCO0lBQ2pGLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBZ0I7SUFDdkMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLE9BQU87Q0FDL0IsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgeyBTaXplLCBTdGFjaywgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgaW50ZWdyYXRpb25SZXNvdXJjZUFybiwgdmFsaWRhdGVQYXR0ZXJuU3VwcG9ydGVkIH0gZnJvbSAnLi4vcHJpdmF0ZS90YXNrLXV0aWxzJztcbmltcG9ydCB7IEJhdGNoU3RyYXRlZ3ksIE1vZGVsQ2xpZW50T3B0aW9ucywgUzNEYXRhVHlwZSwgVHJhbnNmb3JtSW5wdXQsIFRyYW5zZm9ybU91dHB1dCwgVHJhbnNmb3JtUmVzb3VyY2VzIH0gZnJvbSAnLi9iYXNlLXR5cGVzJztcbmltcG9ydCB7IHJlbmRlckVudmlyb25tZW50LCByZW5kZXJUYWdzIH0gZnJvbSAnLi9wcml2YXRlL3V0aWxzJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBjcmVhdGluZyBhbiBBbWF6b24gU2FnZU1ha2VyIHRyYW5zZm9ybSBqb2IgdGFza1xuICpcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTYWdlTWFrZXJDcmVhdGVUcmFuc2Zvcm1Kb2JQcm9wcyBleHRlbmRzIHNmbi5UYXNrU3RhdGVCYXNlUHJvcHMge1xuICAvKipcbiAgICogVHJhbnNmb3JtIEpvYiBOYW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgdHJhbnNmb3JtSm9iTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSb2xlIGZvciB0aGUgVHJhbnNmb3JtIEpvYi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIHJvbGUgaXMgY3JlYXRlZCB3aXRoIGBBbWF6b25TYWdlTWFrZXJGdWxsQWNjZXNzYCBtYW5hZ2VkIHBvbGljeVxuICAgKi9cbiAgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogTnVtYmVyIG9mIHJlY29yZHMgdG8gaW5jbHVkZSBpbiBhIG1pbmktYmF0Y2ggZm9yIGFuIEhUVFAgaW5mZXJlbmNlIHJlcXVlc3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYmF0Y2ggc3RyYXRlZ3lcbiAgICovXG4gIHJlYWRvbmx5IGJhdGNoU3RyYXRlZ3k/OiBCYXRjaFN0cmF0ZWd5O1xuXG4gIC8qKlxuICAgKiBFbnZpcm9ubWVudCB2YXJpYWJsZXMgdG8gc2V0IGluIHRoZSBEb2NrZXIgY29udGFpbmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGVudmlyb25tZW50IHZhcmlhYmxlc1xuICAgKi9cbiAgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIC8qKlxuICAgKiBNYXhpbXVtIG51bWJlciBvZiBwYXJhbGxlbCByZXF1ZXN0cyB0aGF0IGNhbiBiZSBzZW50IHRvIGVhY2ggaW5zdGFuY2UgaW4gYSB0cmFuc2Zvcm0gam9iLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFtYXpvbiBTYWdlTWFrZXIgY2hlY2tzIHRoZSBvcHRpb25hbCBleGVjdXRpb24tcGFyYW1ldGVycyB0byBkZXRlcm1pbmUgdGhlIHNldHRpbmdzIGZvciB5b3VyIGNob3NlbiBhbGdvcml0aG0uXG4gICAqIElmIHRoZSBleGVjdXRpb24tcGFyYW1ldGVycyBlbmRwb2ludCBpcyBub3QgZW5hYmxlZCwgdGhlIGRlZmF1bHQgdmFsdWUgaXMgMS5cbiAgICovXG4gIHJlYWRvbmx5IG1heENvbmN1cnJlbnRUcmFuc2Zvcm1zPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBNYXhpbXVtIGFsbG93ZWQgc2l6ZSBvZiB0aGUgcGF5bG9hZCwgaW4gTUIuXG4gICAqXG4gICAqIEBkZWZhdWx0IDZcbiAgICovXG4gIHJlYWRvbmx5IG1heFBheWxvYWQ/OiBTaXplO1xuXG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBtb2RlbCB0aGF0IHlvdSB3YW50IHRvIHVzZSBmb3IgdGhlIHRyYW5zZm9ybSBqb2IuXG4gICAqL1xuICByZWFkb25seSBtb2RlbE5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogQ29uZmlndXJlcyB0aGUgdGltZW91dCBhbmQgbWF4aW11bSBudW1iZXIgb2YgcmV0cmllcyBmb3IgcHJvY2Vzc2luZyBhIHRyYW5zZm9ybSBqb2IgaW52b2NhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSAwIHJldHJpZXMgYW5kIDYwIHNlY29uZHMgb2YgdGltZW91dFxuICAgKi9cbiAgcmVhZG9ubHkgbW9kZWxDbGllbnRPcHRpb25zPzogTW9kZWxDbGllbnRPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBUYWdzIHRvIGJlIGFwcGxpZWQgdG8gdGhlIHRyYWluIGpvYi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyB0YWdzXG4gICAqL1xuICByZWFkb25seSB0YWdzPzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAvKipcbiAgICogRGF0YXNldCB0byBiZSB0cmFuc2Zvcm1lZCBhbmQgdGhlIEFtYXpvbiBTMyBsb2NhdGlvbiB3aGVyZSBpdCBpcyBzdG9yZWQuXG4gICAqL1xuICByZWFkb25seSB0cmFuc2Zvcm1JbnB1dDogVHJhbnNmb3JtSW5wdXQ7XG5cbiAgLyoqXG4gICAqIFMzIGxvY2F0aW9uIHdoZXJlIHlvdSB3YW50IEFtYXpvbiBTYWdlTWFrZXIgdG8gc2F2ZSB0aGUgcmVzdWx0cyBmcm9tIHRoZSB0cmFuc2Zvcm0gam9iLlxuICAgKi9cbiAgcmVhZG9ubHkgdHJhbnNmb3JtT3V0cHV0OiBUcmFuc2Zvcm1PdXRwdXQ7XG5cbiAgLyoqXG4gICAqIE1MIGNvbXB1dGUgaW5zdGFuY2VzIGZvciB0aGUgdHJhbnNmb3JtIGpvYi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSAxIGluc3RhbmNlIG9mIHR5cGUgTTQuWExhcmdlXG4gICAqL1xuICByZWFkb25seSB0cmFuc2Zvcm1SZXNvdXJjZXM/OiBUcmFuc2Zvcm1SZXNvdXJjZXM7XG59XG5cbi8qKlxuICogQ2xhc3MgcmVwcmVzZW50aW5nIHRoZSBTYWdlTWFrZXIgQ3JlYXRlIFRyYW5zZm9ybSBKb2IgdGFzay5cbiAqXG4gKi9cbmV4cG9ydCBjbGFzcyBTYWdlTWFrZXJDcmVhdGVUcmFuc2Zvcm1Kb2IgZXh0ZW5kcyBzZm4uVGFza1N0YXRlQmFzZSB7XG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IFNVUFBPUlRFRF9JTlRFR1JBVElPTl9QQVRURVJOUzogc2ZuLkludGVncmF0aW9uUGF0dGVybltdID0gW1xuICAgIHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUkVRVUVTVF9SRVNQT05TRSxcbiAgICBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJVTl9KT0IsXG4gIF07XG5cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IHRhc2tQb2xpY2llcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXTtcbiAgcHJvdGVjdGVkIHJlYWRvbmx5IHRhc2tNZXRyaWNzPzogc2ZuLlRhc2tNZXRyaWNzQ29uZmlnO1xuXG4gIC8qKlxuICAgKiBEYXRhc2V0IHRvIGJlIHRyYW5zZm9ybWVkIGFuZCB0aGUgQW1hem9uIFMzIGxvY2F0aW9uIHdoZXJlIGl0IGlzIHN0b3JlZC5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgdHJhbnNmb3JtSW5wdXQ6IFRyYW5zZm9ybUlucHV0O1xuXG4gIC8qKlxuICAgKiBNTCBjb21wdXRlIGluc3RhbmNlcyBmb3IgdGhlIHRyYW5zZm9ybSBqb2IuXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IHRyYW5zZm9ybVJlc291cmNlczogVHJhbnNmb3JtUmVzb3VyY2VzO1xuICBwcml2YXRlIHJlYWRvbmx5IGludGVncmF0aW9uUGF0dGVybjogc2ZuLkludGVncmF0aW9uUGF0dGVybjtcbiAgcHJpdmF0ZSBfcm9sZT86IGlhbS5JUm9sZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBTYWdlTWFrZXJDcmVhdGVUcmFuc2Zvcm1Kb2JQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgIHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuID0gcHJvcHMuaW50ZWdyYXRpb25QYXR0ZXJuIHx8IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUkVRVUVTVF9SRVNQT05TRTtcbiAgICB2YWxpZGF0ZVBhdHRlcm5TdXBwb3J0ZWQodGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4sIFNhZ2VNYWtlckNyZWF0ZVRyYW5zZm9ybUpvYi5TVVBQT1JURURfSU5URUdSQVRJT05fUEFUVEVSTlMpO1xuXG4gICAgLy8gc2V0IHRoZSBzYWdlbWFrZXIgcm9sZSBvciBjcmVhdGUgbmV3IG9uZVxuICAgIGlmIChwcm9wcy5yb2xlKSB7XG4gICAgICB0aGlzLl9yb2xlID0gcHJvcHMucm9sZTtcbiAgICB9XG5cbiAgICAvLyBzZXQgdGhlIFMzIERhdGEgdHlwZSBvZiB0aGUgaW5wdXQgZGF0YSBjb25maWcgb2JqZWN0cyB0byBiZSAnUzNQcmVmaXgnIGlmIG5vdCBkZWZpbmVkXG4gICAgdGhpcy50cmFuc2Zvcm1JbnB1dCA9IHByb3BzLnRyYW5zZm9ybUlucHV0LnRyYW5zZm9ybURhdGFTb3VyY2UuczNEYXRhU291cmNlLnMzRGF0YVR5cGVcbiAgICAgID8gcHJvcHMudHJhbnNmb3JtSW5wdXRcbiAgICAgIDogT2JqZWN0LmFzc2lnbih7fSwgcHJvcHMudHJhbnNmb3JtSW5wdXQsIHtcbiAgICAgICAgdHJhbnNmb3JtRGF0YVNvdXJjZTogeyBzM0RhdGFTb3VyY2U6IHsgLi4ucHJvcHMudHJhbnNmb3JtSW5wdXQudHJhbnNmb3JtRGF0YVNvdXJjZS5zM0RhdGFTb3VyY2UsIHMzRGF0YVR5cGU6IFMzRGF0YVR5cGUuUzNfUFJFRklYIH0gfSxcbiAgICAgIH0pO1xuXG4gICAgLy8gc2V0IHRoZSBkZWZhdWx0IHZhbHVlIGZvciB0aGUgdHJhbnNmb3JtIHJlc291cmNlc1xuICAgIHRoaXMudHJhbnNmb3JtUmVzb3VyY2VzID0gcHJvcHMudHJhbnNmb3JtUmVzb3VyY2VzIHx8IHtcbiAgICAgIGluc3RhbmNlQ291bnQ6IDEsXG4gICAgICBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGUub2YoZWMyLkluc3RhbmNlQ2xhc3MuTTQsIGVjMi5JbnN0YW5jZVNpemUuWExBUkdFKSxcbiAgICB9O1xuXG4gICAgdGhpcy50YXNrUG9saWNpZXMgPSB0aGlzLm1ha2VQb2xpY3lTdGF0ZW1lbnRzKCk7XG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwcm90ZWN0ZWQgX3JlbmRlclRhc2soKTogYW55IHtcbiAgICByZXR1cm4ge1xuICAgICAgUmVzb3VyY2U6IGludGVncmF0aW9uUmVzb3VyY2VBcm4oJ3NhZ2VtYWtlcicsICdjcmVhdGVUcmFuc2Zvcm1Kb2InLCB0aGlzLmludGVncmF0aW9uUGF0dGVybiksXG4gICAgICBQYXJhbWV0ZXJzOiBzZm4uRmllbGRVdGlscy5yZW5kZXJPYmplY3QodGhpcy5yZW5kZXJQYXJhbWV0ZXJzKCkpLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogVGhlIGV4ZWN1dGlvbiByb2xlIGZvciB0aGUgU2FnZW1ha2VyIHRyYW5zZm9ybSBqb2IuXG4gICAqXG4gICAqIE9ubHkgYXZhaWxhYmxlIGFmdGVyIHRhc2sgaGFzIGJlZW4gYWRkZWQgdG8gYSBzdGF0ZSBtYWNoaW5lLlxuICAgKi9cbiAgcHVibGljIGdldCByb2xlKCk6IGlhbS5JUm9sZSB7XG4gICAgaWYgKHRoaXMuX3JvbGUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdyb2xlIG5vdCBhdmFpbGFibGUgeWV0LS11c2UgdGhlIG9iamVjdCBpbiBhIFRhc2sgZmlyc3QnKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuX3JvbGU7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlclBhcmFtZXRlcnMoKTogeyBba2V5OiBzdHJpbmddOiBhbnkgfSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLih0aGlzLnByb3BzLmJhdGNoU3RyYXRlZ3kgPyB7IEJhdGNoU3RyYXRlZ3k6IHRoaXMucHJvcHMuYmF0Y2hTdHJhdGVneSB9IDoge30pLFxuICAgICAgLi4ucmVuZGVyRW52aXJvbm1lbnQodGhpcy5wcm9wcy5lbnZpcm9ubWVudCksXG4gICAgICAuLi4odGhpcy5wcm9wcy5tYXhDb25jdXJyZW50VHJhbnNmb3JtcyA/IHsgTWF4Q29uY3VycmVudFRyYW5zZm9ybXM6IHRoaXMucHJvcHMubWF4Q29uY3VycmVudFRyYW5zZm9ybXMgfSA6IHt9KSxcbiAgICAgIC4uLih0aGlzLnByb3BzLm1heFBheWxvYWQgPyB7IE1heFBheWxvYWRJbk1COiB0aGlzLnByb3BzLm1heFBheWxvYWQudG9NZWJpYnl0ZXMoKSB9IDoge30pLFxuICAgICAgLi4udGhpcy5wcm9wcy5tb2RlbENsaWVudE9wdGlvbnMgPyB0aGlzLnJlbmRlck1vZGVsQ2xpZW50T3B0aW9ucyh0aGlzLnByb3BzLm1vZGVsQ2xpZW50T3B0aW9ucykgOiB7fSxcbiAgICAgIE1vZGVsTmFtZTogdGhpcy5wcm9wcy5tb2RlbE5hbWUsXG4gICAgICAuLi5yZW5kZXJUYWdzKHRoaXMucHJvcHMudGFncyksXG4gICAgICAuLi50aGlzLnJlbmRlclRyYW5zZm9ybUlucHV0KHRoaXMudHJhbnNmb3JtSW5wdXQpLFxuICAgICAgVHJhbnNmb3JtSm9iTmFtZTogdGhpcy5wcm9wcy50cmFuc2Zvcm1Kb2JOYW1lLFxuICAgICAgLi4udGhpcy5yZW5kZXJUcmFuc2Zvcm1PdXRwdXQodGhpcy5wcm9wcy50cmFuc2Zvcm1PdXRwdXQpLFxuICAgICAgLi4udGhpcy5yZW5kZXJUcmFuc2Zvcm1SZXNvdXJjZXModGhpcy50cmFuc2Zvcm1SZXNvdXJjZXMpLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIHJlbmRlck1vZGVsQ2xpZW50T3B0aW9ucyhvcHRpb25zOiBNb2RlbENsaWVudE9wdGlvbnMpOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IHtcbiAgICBjb25zdCByZXRyaWVzID0gb3B0aW9ucy5pbnZvY2F0aW9uc01heFJldHJpZXM7XG4gICAgaWYgKCFUb2tlbi5pc1VucmVzb2x2ZWQocmV0cmllcykgJiYgcmV0cmllcz8gKHJldHJpZXMgPCAwIHx8IHJldHJpZXMgPiAzKTogZmFsc2UpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgaW52b2NhdGlvbnNNYXhSZXRyaWVzIHNob3VsZCBiZSBiZXR3ZWVuIDAgYW5kIDMuIFJlY2VpdmVkOiAke3JldHJpZXN9LmApO1xuICAgIH1cbiAgICBjb25zdCB0aW1lb3V0ID0gb3B0aW9ucy5pbnZvY2F0aW9uc1RpbWVvdXQ/LnRvU2Vjb25kcygpO1xuICAgIGlmICghVG9rZW4uaXNVbnJlc29sdmVkKHRpbWVvdXQpICYmIHRpbWVvdXQ/ICh0aW1lb3V0IDwgMSB8fCB0aW1lb3V0ID4gMzYwMCk6IGZhbHNlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYGludm9jYXRpb25zVGltZW91dCBzaG91bGQgYmUgYmV0d2VlbiAxIGFuZCAzNjAwIHNlY29uZHMuIFJlY2VpdmVkOiAke3RpbWVvdXR9LmApO1xuICAgIH1cbiAgICByZXR1cm4ge1xuICAgICAgTW9kZWxDbGllbnRDb25maWc6IHtcbiAgICAgICAgSW52b2NhdGlvbnNNYXhSZXRyaWVzOiByZXRyaWVzID8/IDAsXG4gICAgICAgIEludm9jYXRpb25zVGltZW91dEluU2Vjb25kczogdGltZW91dCA/PyA2MCxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyVHJhbnNmb3JtSW5wdXQoaW5wdXQ6IFRyYW5zZm9ybUlucHV0KTogeyBba2V5OiBzdHJpbmddOiBhbnkgfSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIFRyYW5zZm9ybUlucHV0OiB7XG4gICAgICAgIC4uLihpbnB1dC5jb21wcmVzc2lvblR5cGUgPyB7IENvbXByZXNzaW9uVHlwZTogaW5wdXQuY29tcHJlc3Npb25UeXBlIH0gOiB7fSksXG4gICAgICAgIC4uLihpbnB1dC5jb250ZW50VHlwZSA/IHsgQ29udGVudFR5cGU6IGlucHV0LmNvbnRlbnRUeXBlIH0gOiB7fSksXG4gICAgICAgIERhdGFTb3VyY2U6IHtcbiAgICAgICAgICBTM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgIFMzVXJpOiBpbnB1dC50cmFuc2Zvcm1EYXRhU291cmNlLnMzRGF0YVNvdXJjZS5zM1VyaSxcbiAgICAgICAgICAgIFMzRGF0YVR5cGU6IGlucHV0LnRyYW5zZm9ybURhdGFTb3VyY2UuczNEYXRhU291cmNlLnMzRGF0YVR5cGUsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgLi4uKGlucHV0LnNwbGl0VHlwZSA/IHsgU3BsaXRUeXBlOiBpbnB1dC5zcGxpdFR5cGUgfSA6IHt9KSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyVHJhbnNmb3JtT3V0cHV0KG91dHB1dDogVHJhbnNmb3JtT3V0cHV0KTogeyBba2V5OiBzdHJpbmddOiBhbnkgfSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIFRyYW5zZm9ybU91dHB1dDoge1xuICAgICAgICBTM091dHB1dFBhdGg6IG91dHB1dC5zM091dHB1dFBhdGgsXG4gICAgICAgIC4uLihvdXRwdXQuZW5jcnlwdGlvbktleSA/IHsgS21zS2V5SWQ6IG91dHB1dC5lbmNyeXB0aW9uS2V5LmtleUFybiB9IDoge30pLFxuICAgICAgICAuLi4ob3V0cHV0LmFjY2VwdCA/IHsgQWNjZXB0OiBvdXRwdXQuYWNjZXB0IH0gOiB7fSksXG4gICAgICAgIC4uLihvdXRwdXQuYXNzZW1ibGVXaXRoID8geyBBc3NlbWJsZVdpdGg6IG91dHB1dC5hc3NlbWJsZVdpdGggfSA6IHt9KSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyVHJhbnNmb3JtUmVzb3VyY2VzKHJlc291cmNlczogVHJhbnNmb3JtUmVzb3VyY2VzKTogeyBba2V5OiBzdHJpbmddOiBhbnkgfSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIFRyYW5zZm9ybVJlc291cmNlczoge1xuICAgICAgICBJbnN0YW5jZUNvdW50OiByZXNvdXJjZXMuaW5zdGFuY2VDb3VudCxcbiAgICAgICAgSW5zdGFuY2VUeXBlOiBzZm4uSnNvblBhdGguaXNFbmNvZGVkSnNvblBhdGgocmVzb3VyY2VzLmluc3RhbmNlVHlwZS50b1N0cmluZygpKVxuICAgICAgICAgID8gcmVzb3VyY2VzLmluc3RhbmNlVHlwZS50b1N0cmluZygpIDogYG1sLiR7cmVzb3VyY2VzLmluc3RhbmNlVHlwZX1gLFxuICAgICAgICAuLi4ocmVzb3VyY2VzLnZvbHVtZUVuY3J5cHRpb25LZXkgPyB7IFZvbHVtZUttc0tleUlkOiByZXNvdXJjZXMudm9sdW1lRW5jcnlwdGlvbktleS5rZXlBcm4gfSA6IHt9KSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgbWFrZVBvbGljeVN0YXRlbWVudHMoKTogaWFtLlBvbGljeVN0YXRlbWVudFtdIHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHRoaXMpO1xuXG4gICAgLy8gY3JlYXRlIG5ldyByb2xlIGlmIGRvZXNuJ3QgZXhpc3RcbiAgICBpZiAodGhpcy5fcm9sZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aGlzLl9yb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdTYWdlbWFrZXJUcmFuc2Zvcm1Sb2xlJywge1xuICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnc2FnZW1ha2VyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgbWFuYWdlZFBvbGljaWVzOiBbaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBbWF6b25TYWdlTWFrZXJGdWxsQWNjZXNzJyldLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9kZy9zYWdlbWFrZXItaWFtLmh0bWxcbiAgICBjb25zdCBwb2xpY3lTdGF0ZW1lbnRzID0gW1xuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbJ3NhZ2VtYWtlcjpDcmVhdGVUcmFuc2Zvcm1Kb2InLCAnc2FnZW1ha2VyOkRlc2NyaWJlVHJhbnNmb3JtSm9iJywgJ3NhZ2VtYWtlcjpTdG9wVHJhbnNmb3JtSm9iJ10sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIHN0YWNrLmZvcm1hdEFybih7XG4gICAgICAgICAgICBzZXJ2aWNlOiAnc2FnZW1ha2VyJyxcbiAgICAgICAgICAgIHJlc291cmNlOiAndHJhbnNmb3JtLWpvYicsXG4gICAgICAgICAgICByZXNvdXJjZU5hbWU6ICcqJyxcbiAgICAgICAgICB9KSxcbiAgICAgICAgXSxcbiAgICAgIH0pLFxuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbJ3NhZ2VtYWtlcjpMaXN0VGFncyddLFxuICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgfSksXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFsnaWFtOlBhc3NSb2xlJ10sXG4gICAgICAgIHJlc291cmNlczogW3RoaXMucm9sZS5yb2xlQXJuXSxcbiAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgIFN0cmluZ0VxdWFsczogeyAnaWFtOlBhc3NlZFRvU2VydmljZSc6ICdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pLFxuICAgIF07XG5cbiAgICBpZiAodGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4gPT09IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUlVOX0pPQikge1xuICAgICAgcG9saWN5U3RhdGVtZW50cy5wdXNoKFxuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogWydldmVudHM6UHV0VGFyZ2V0cycsICdldmVudHM6UHV0UnVsZScsICdldmVudHM6RGVzY3JpYmVSdWxlJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgICBzdGFjay5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgICBzZXJ2aWNlOiAnZXZlbnRzJyxcbiAgICAgICAgICAgICAgcmVzb3VyY2U6ICdydWxlJyxcbiAgICAgICAgICAgICAgcmVzb3VyY2VOYW1lOiAnU3RlcEZ1bmN0aW9uc0dldEV2ZW50c0ZvclNhZ2VNYWtlclRyYW5zZm9ybUpvYnNSdWxlJyxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pLFxuICAgICAgKTtcbiAgICB9XG5cbiAgICByZXR1cm4gcG9saWN5U3RhdGVtZW50cztcbiAgfVxufVxuIl19