import random
import string

from assimilator.core.database import UnitOfWork, Repository, NotFoundError
from assimilator.core.patterns import LazyCommand
from assimilator.core.database import DataLayerError

from dependencies import create_uow
from examples.alchemy.database.models import User


def create_user(username: str, email: str, balance: float, uow: UnitOfWork) -> int:
    with uow:
        new_user = User(username=username, email=email, balance=balance)
        uow.repository.save(new_user)
        uow.commit()
        return new_user.id      # id generated by default. But, we can change that behaviour if we need


def read_user(id: int, repository: Repository) -> User:
    return repository.get(repository.specs.filter(id=id))


def buy_product(user_id: int, product_price: int, uow: UnitOfWork):
    with uow:
        found_user = read_user(id=user_id, repository=uow.repository)

        found_user.balance -= product_price
        uow.repository.update(found_user)
        uow.commit()


def refresh_user(old_user: User, repository: Repository) -> User:
    try:
        repository.refresh(old_user)
    except DataLayerError:
        pass

    return old_user


def create_many_users(uow):
    with uow:
        for i in range(100):
            new_user = User(
                username="".join(random.sample(string.ascii_letters, 10)),
                email=f"{''.join(random.sample(string.ascii_letters, 10))}@gmail.com",
                balance=random.randint(0, 100),
            )
            uow.repository.save(new_user)

        uow.commit()    # Commit is only called once!


def show_rich_users(balance: int, repository: Repository):
    users: LazyCommand[User] = repository.filter(
        repository.specs.filter(balance__gt=balance),
        repository.specs.paginate(limit=10),
        lazy=True,
    )

    for rich_user in users:
        print("The user", rich_user.username, "is rich!", "Balance:", rich_user.balance)


def delete_user(id: int, uow: UnitOfWork):
    with uow:
        user_to_delete = uow.repository.get(
            uow.repository.specs.filter(id=id)
        )
        uow.repository.delete(user_to_delete)
        uow.commit()


def order_users(repository: Repository):
    for i, ordered_user in enumerate(repository.filter(
        repository.specs.order('id', '-balance'),
        repository.specs.paginate(offset=20, limit=40),
    )):
        print(f"User {i} ordered by id and balance:", ordered_user.username, ordered_user.balance)


if __name__ == '__main__':
    new_user_id = create_user(
        username="Andrey",
        email="python.on.papyrus@gmail.com",
        balance=1000,
        uow=create_uow(),
    )

    print(f"User with '{new_user_id}' was created")

    user = read_user(id=new_user_id, repository=create_uow().repository)
    print("User returned from Redis:", user)

    buy_product(user_id=new_user_id, product_price=100, uow=create_uow())

    updated_user = refresh_user(user, repository=create_uow().repository)
    print("User balance after product purchase:", updated_user.balance)

    create_many_users(create_uow())
    show_rich_users(balance=90, repository=create_uow().repository)

    delete_user(id=new_user_id, uow=create_uow())
    print("User is deleted from the storage!")

    try:
        read_user(id=new_user_id, repository=create_uow().repository)
    except NotFoundError as error:
        print("User was not found due to his deletion! Error:", error)

    order_users(repository=create_uow().repository)
