"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const core_1 = require("@aws-cdk/core");
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * Defines an EventBridge Rule in this stack.
 *
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        this.accountEventBusTargets = {};
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.anyValue({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.anyValue({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    /**
     * Import an existing EventBridge Rule provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventRuleArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new
     * targets.
     *
     * No-op if target is undefined.
     */
    addTarget(target) {
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = targetProps.role ? targetProps.role.roleArn : undefined;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = targetStack.account;
            const targetRegion = targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            if (targetRegion !== sourceRegion) {
                throw new Error('Rule and target must be in the same region');
            }
            if (targetAccount !== sourceAccount) {
                // cross-account event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html
                // for cross-account events, we require concrete accounts
                if (core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account events');
                }
                // and the target region has to be concrete as well
                if (core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account events');
                }
                // the _actual_ target is just the event bus of the target's account
                // make sure we only add it once per account
                const exists = this.accountEventBusTargets[targetAccount];
                if (!exists) {
                    this.accountEventBusTargets[targetAccount] = true;
                    this.targets.push({
                        id,
                        arn: targetStack.formatArn({
                            service: 'events',
                            resource: 'event-bus',
                            resourceName: 'default',
                            region: targetRegion,
                            account: targetAccount,
                        }),
                    });
                }
                // Grant the source account permissions to publish events to the event bus of the target account.
                // Do it in a separate stack instead of the target stack (which seems like the obvious place to put it),
                // because it needs to be deployed before the rule containing the above event-bus target in the source stack
                // (EventBridge verifies whether you have permissions to the targets on rule creation),
                // but it's common for the target stack to depend on the source stack
                // (that's the case with CodePipeline, for example)
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account targets must be part of a CDK app');
                }
                const targetApp = targetProps.targetResource.node.root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
                let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
                if (!eventBusPolicyStack) {
                    eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                        env: {
                            account: targetAccount,
                            region: targetRegion,
                        },
                        stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                    });
                    new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                        action: 'events:PutEvents',
                        statementId: `Allow-account-${sourceAccount}`,
                        principal: sourceAccount,
                    });
                }
                // deploy the event bus permissions before the source stack
                sourceStack.addDependency(eventBusPolicyStack);
                // The actual rule lives in the target stack.
                // Other than the account, it's identical to this one
                // eventPattern is mutable through addEventPattern(), so we need to lazy evaluate it
                // but only Tokens can be lazy in the framework, so make a subclass instead
                const self = this;
                class CopyRule extends Rule {
                    _renderEventPattern() {
                        return self._renderEventPattern();
                    }
                    // we need to override validate(), as it uses the
                    // value of the eventPattern field,
                    // which might be empty in the case of the copied rule
                    // (as the patterns in the original might be added through addEventPattern(),
                    // not passed through the constructor).
                    // Anyway, even if the original rule is invalid,
                    // we would get duplicate errors if we didn't override this,
                    // which is probably a bad idea in and of itself
                    validate() {
                        return [];
                    }
                }
                new CopyRule(targetStack, `${this.node.uniqueId}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                });
                return;
            }
        }
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            sqsParameters: targetProps.sqsParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: inputProps && inputProps.inputTemplate !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule. If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *    {
     *      "resources": [ "r1" ],
     *      "detail": {
     *        "hello": [ 1 ]
     *      }
     *    }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *    {
     *      "resources": [ "r2" ],
     *      "detail": {
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     * The resulting event pattern will be:
     *
     *    {
     *      "resources": [ "r1", "r2" ],
     *      "detail": {
     *        "hello": [ 1 ],
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     */
    addEventPattern(eventPattern) {
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        const eventPattern = this.eventPattern;
        if (Object.keys(eventPattern).length === 0) {
            return undefined;
        }
        // rename 'detailType' to 'detail-type'
        const out = {};
        for (let key of Object.keys(eventPattern)) {
            const value = eventPattern[key];
            if (key === 'detailType') {
                key = 'detail-type';
            }
            out[key] = value;
        }
        return out;
    }
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
}
exports.Rule = Rule;
//# sourceMappingURL=data:application/json;base64,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