## xespresso
Quantum ESPRESSO Calculator for Atomic Simulation Environment (ASE).

For the introduction of ASE , please visit https://wiki.fysik.dtu.dk/ase/index.html


### Functions:
* Support all QE packages, including: pw, band, neb, dos, projwfc, pp ...
* Spin-polarized calculation
* LD(S)A+U
* Automatic submit job
* Automatic set up "nscf" calculation
* Read and plot dos, pdos and layer resolved pdos
* Plot NEB

### Author
* Xing Wang  <xingwang1991@gmail.com>

### Dependencies

* Python
* ASE

### Installation

Clone this repo. Add it to your PYTHONPATH and PATH. On windows, you can edit the system environment variables.

``` sh
export PYTHONPATH="Your-Location":$PYTHONPATH
export ASE_ESPRESSO_COMMAND="/path/to/PACKAGE.x  PARALLEL  -in  PREFIX.PACKAGEi  >  PREFIX.PACKAGEo"
export ESPRESSO_PSEUDO="/path/to/pseudo"
```


### Examples

#### Automatic submit job

A example of setting parameters for the queue. See example/queue.py

``` python
queue = {'nodes': 1, 
         'ntasks-per-node': 4, 
		 'account': 'dcb', 
		 'partition': 'all', 
		 'time': '0:10:00'}
calc = Espresso(queue = queue)
```

#### Add new species
Some atoms are special:
+ atoms with different starting_magnetization
+ atoms with different U values
+ atoms with special basis set

For example, Fe with spin state AFM. See example/spin.py

``` python
atoms.arrays['species'] = atoms.get_chemical_symbols()
atoms.arrays['species'][1] = 'Fe1'
```

#### Setting parameters with "(i), i=1,ntyp"
Hubbard, starting_magnetization, starting_charge and so on. See example/dft+u.py

``` python
input_ntyp = {
'starting_magnetization': {'Fe1': 1.0, 'Fe2': -1.0},
'Hubbard_U': {'Fe1': 3.0, 'Fe2': 3.0},
}
input_data['input_ntyp'] = input_ntyp,
```

#### Control parallelization levels
To control the number of processors in each group: -ni,
-nk, -nb, -nt, -nd) are used.

``` python
calc = Espresso(pseudopotentials = pseudopotentials, 
                 package = 'pw',
                 parallel = '-nk 2 -nt 4 -nd 144',  # parallel parameters
				 }
```

#### nscf calculation

A example of nscf calculation following the above one.

``` python
calc.read_results()
calc.nscf(queue = queue, kpts = (12, 12, 12))
calc.nscf_calculate()
```

#### calculate dos and pdos

A example of calculating and plotting the pdos from the nscf calculation.

``` python
calc.read_results()
calc.post(queue = queue, package = 'dos', Emin = fe - 30, Emax = fe + 30, DeltaE = 0.1)
calc.post(queue = queue, package = 'projwfc', Emin = fe - 30, Emax = fe + 30, DeltaE = 0.1)
```
<!-- <img src="examples/figs/al-pdos.png" width="500"/> -->

#### calculate work function
``` python
calc.post(queue = queue, package = 'pp', plot_num = 11, fileout = 'potential.cube', iflag = 3, output_format=6)
calc.get_work_function()
```

#### restar from previous calculation
``` python
calc.read_results()
atoms = calc.results['atoms']       
calc.run(atoms = atoms, restart = 1)
```

#### NEB calculation
See example/neb.py
``` python
from xespresso.neb import NEBEspresso
calc = NEBEspresso(
                 package = 'neb',
                 images = images,
                 climbing_images = [5],
                 path_data = path_data
				 )
calc.calculate()
calc.read_results()
calc.plot()
```
<img src="xespress/examples/figs/neb.png" width="500"/>

