"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const s3 = require("@aws-cdk/aws-s3");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_patterns_1 = require("@aws-cdk/aws-route53-patterns");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
class SPADeploy extends cdk.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        if (typeof config != 'undefined') {
            this.globalConfig = config;
        }
        else {
            this.globalConfig = {
                encryptBucket: false,
                ipFilter: false
            };
        }
    }
    /**
     * Helper method to provide a configured s3 bucket
     */
    getS3Bucket(config, isForCloudFront) {
        let bucketConfig = {
            websiteIndexDocument: config.indexDoc,
            websiteErrorDocument: config.errorDoc,
            publicReadAccess: true
        };
        if (this.globalConfig.encryptBucket === true) {
            bucketConfig.encryption = s3.BucketEncryption.S3_MANAGED;
        }
        if (this.globalConfig.ipFilter === true || isForCloudFront === true) {
            bucketConfig.publicReadAccess = false;
        }
        let bucket = new s3.Bucket(this, 'WebsiteBucket', bucketConfig);
        if (this.globalConfig.ipFilter === true && isForCloudFront === false) {
            if (typeof this.globalConfig.ipList == 'undefined') {
                this.node.addError('When IP Filter is true then the IP List is required');
            }
            const bucketPolicy = new aws_iam_1.PolicyStatement();
            bucketPolicy.addAnyPrincipal();
            bucketPolicy.addActions('s3:GetObject');
            bucketPolicy.addResources(bucket.bucketArn + '/*');
            bucketPolicy.addCondition('IpAddress', {
                'aws:SourceIp': this.globalConfig.ipList
            });
            bucket.addToResourcePolicy(bucketPolicy);
        }
        return bucket;
    }
    /**
     * Helper method to provide configuration for cloudfront
     */
    getCFConfig(websiteBucket, config, accessIdentity, cert) {
        let cfConfig = {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: websiteBucket,
                        originAccessIdentity: accessIdentity
                    },
                    behaviors: [{ isDefaultBehavior: true }]
                }
            ],
            //We need to redirect all unknown routes back to index.html for angular routing to work
            errorConfigurations: [{
                    errorCode: 403,
                    responsePagePath: (config.errorDoc ? '/' + config.errorDoc : '/' + config.indexDoc),
                    responseCode: 200
                },
                {
                    errorCode: 404,
                    responsePagePath: (config.errorDoc ? '/' + config.errorDoc : '/' + config.indexDoc),
                    responseCode: 200
                }]
        };
        if (typeof config.certificateARN != 'undefined' && typeof config.cfAliases != 'undefined') {
            cfConfig.aliasConfiguration = {
                acmCertRef: config.certificateARN,
                names: config.cfAliases
            };
        }
        if (typeof config.zoneName != 'undefined' && typeof cert != 'undefined') {
            cfConfig.viewerCertificate = aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(cert, {
                aliases: [config.zoneName],
            });
        }
        return cfConfig;
    }
    /**
     * Basic setup needed for a non-ssl, non vanity url, non cached s3 website
     */
    createBasicSite(config) {
        const websiteBucket = this.getS3Bucket(config, false);
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        let cfnOutputConfig = {
            description: 'The url of the website',
            value: websiteBucket.bucketWebsiteUrl
        };
        if (config.exportWebsiteUrlOutput === true) {
            if (typeof config.exportWebsiteUrlName == 'undefined' || config.exportWebsiteUrlName === '') {
                this.node.addError('When Output URL as AWS Export property is true then the output name is required');
            }
            cfnOutputConfig.exportName = config.exportWebsiteUrlName;
        }
        new cdk.CfnOutput(this, 'URL', cfnOutputConfig);
    }
    /**
     * This will create an s3 deployment fronted by a cloudfront distribution
     * It will also setup error forwarding and unauth forwarding back to indexDoc
     */
    createSiteWithCloudfront(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            //Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution: distribution,
            distributionPaths: ['/', '/' + config.indexDoc]
        });
        new cdk.CfnOutput(this, 'cloudfront domain', {
            description: 'The domain of the website',
            value: distribution.domainName
        });
    }
    /**
     * S3 Deployment, cloudfront distribution, ssl cert and error forwarding auto
     * configured by using the details in the hosted zone provided
     */
    createSiteFromHostedZone(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        let zone = aws_route53_1.HostedZone.fromLookup(this, 'HostedZone', { domainName: config.zoneName });
        let cert = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
            hostedZone: zone,
            domainName: config.zoneName,
            region: 'us-east-1',
        });
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity, cert));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            //Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution: distribution,
            distributionPaths: ['/', '/' + config.indexDoc]
        });
        new aws_route53_1.ARecord(this, 'Alias', {
            zone,
            recordName: config.zoneName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(distribution))
        });
        new aws_route53_patterns_1.HttpsRedirect(this, 'Redirect', {
            zone,
            recordNames: ['www.' + config.zoneName],
            targetDomain: config.zoneName,
        });
    }
}
exports.SPADeploy = SPADeploy;
//# sourceMappingURL=data:application/json;base64,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