"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib/");
test('Cloudfront Distribution Included', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "CustomErrorResponses": [
                {
                    "ErrorCode": 403,
                    "ResponseCode": 200,
                    "ResponsePagePath": "/index.html"
                },
                {
                    "ErrorCode": 404,
                    "ResponseCode": 200,
                    "ResponsePagePath": "/index.html"
                }
            ],
            "DefaultCacheBehavior": {
                "ViewerProtocolPolicy": "redirect-to-https"
            },
            "DefaultRootObject": "index.html",
            "HttpVersion": "http2",
            "IPV6Enabled": true,
            "PriceClass": "PriceClass_100",
            "ViewerCertificate": {
                "CloudFrontDefaultCertificate": true
            }
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": [
                        "s3:GetObject*",
                        "s3:GetBucket*",
                        "s3:List*"
                    ],
                    "Effect": "Allow",
                    "Principal": {
                        "CanonicalUser": {
                            "Fn::GetAtt": [
                                "spaDeployOriginAccessIdentityEDA4C19C",
                                "S3CanonicalUserId"
                            ]
                        }
                    },
                    "Resource": [
                        {
                            "Fn::GetAtt": [
                                "spaDeployWebsiteBucket1E4C4442",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "spaDeployWebsiteBucket1E4C4442",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                }
            ]
        }
    }));
});
test('Cloudfront Distribution Included - with non-default error-doc cfg set', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "CustomErrorResponses": [
                {
                    "ErrorCode": 403,
                    "ResponseCode": 200,
                    "ResponsePagePath": "/error.html"
                },
                {
                    "ErrorCode": 404,
                    "ResponseCode": 200,
                    "ResponsePagePath": "/error.html"
                }
            ],
            "DefaultCacheBehavior": {
                "ViewerProtocolPolicy": "redirect-to-https"
            },
            "DefaultRootObject": "index.html",
            "HttpVersion": "http2",
            "IPV6Enabled": true,
            "PriceClass": "PriceClass_100",
            "ViewerCertificate": {
                "CloudFrontDefaultCertificate": true
            }
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": [
                        "s3:GetObject*",
                        "s3:GetBucket*",
                        "s3:List*"
                    ],
                    "Effect": "Allow",
                    "Principal": {
                        "CanonicalUser": {
                            "Fn::GetAtt": [
                                "spaDeployOriginAccessIdentityEDA4C19C",
                                "S3CanonicalUserId"
                            ]
                        }
                    },
                    "Resource": [
                        {
                            "Fn::GetAtt": [
                                "spaDeployWebsiteBucket1E4C4442",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "spaDeployWebsiteBucket1E4C4442",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                }
            ]
        }
    }));
});
test('Cloudfront With Custom Cert and Aliases', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com']
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.test.com"
            ],
            "ViewerCertificate": {
                "AcmCertificateArn": "arn:1234",
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Basic Site Setup', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Basic Site Setup with Error Doc set', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Basic Site Setup, Encrypted Bucket', () => {
    let stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Cloudfront With Encrypted Bucket', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true });
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com']
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.test.com"
            ],
            "ViewerCertificate": {
                "AcmCertificateArn": "arn:1234",
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Basic Site Setup, IP Filter', () => {
    let stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true, ipFilter: true, ipList: ['1.1.1.1'] })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Condition": {
                        "IpAddress": {
                            "aws:SourceIp": [
                                "1.1.1.1"
                            ]
                        }
                    },
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Create From Hosted Zone', () => {
    let app = new core_1.App();
    let stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234'
        }
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.cdkspadeploy.com"
            ],
            "ViewerCertificate": {
                "SslSupportMethod": "sni-only"
            }
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": [
                        "s3:GetObject*",
                        "s3:GetBucket*",
                        "s3:List*"
                    ],
                    "Effect": "Allow",
                    "Principal": {
                        "CanonicalUser": {
                            "Fn::GetAtt": [
                                "spaDeployOriginAccessIdentityEDA4C19C",
                                "S3CanonicalUserId"
                            ]
                        }
                    },
                    "Resource": [
                        {
                            "Fn::GetAtt": [
                                "spaDeployWebsiteBucket1E4C4442",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "spaDeployWebsiteBucket1E4C4442",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                }
            ]
        }
    }));
});
test('Create From Hosted Zone with Error Bucket', () => {
    let app = new core_1.App();
    let stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234'
        }
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.cdkspadeploy.com"
            ],
            "ViewerCertificate": {
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Basic Site Setup, URL Output Enabled With Name', () => {
    let stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true,
        exportWebsiteUrlName: exportName
    });
    // THEN  
    assert_1.expect(stack).to(assert_1.haveOutput({
        exportName: exportName
    }));
});
test('Basic Site Setup, URL Output Enabled With No Name', () => {
    let stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true
    });
    // THEN  
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName: exportName
    }));
});
test('Basic Site Setup, URL Output Not Enabled', () => {
    let stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: false
    });
    // THEN  
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName: exportName
    }));
});
//# sourceMappingURL=data:application/json;base64,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