# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/CLI_DataSource.ipynb (unless otherwise specified).

__all__ = ["logger"]

# Cell

from typing import *

# Internal Cell

import os

import typer
from typer import echo
from tabulate import tabulate
import datetime as dt
import pandas as pd

from ..client import Client
from . import helper
from ..logger import get_logger, set_level

# Internal Cell

app = typer.Typer(
    help="A set of commands for managing the datasources and training ML models on them."
)

# Cell

logger = get_logger(__name__)

# Internal Cell


@app.command()
@helper.requires_auth_token
def dtypes(
    id: str = typer.Argument(
        ...,
        help="Datasource id in the server.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Return the dtypes of the datasource."""

    from ..client import DataSource

    ds = DataSource(id=int(id))
    dtypes = ds.dtypes.T.rename(columns={0: "dtype"})
    typer.echo(dtypes)


# Internal Cell


@app.command()
@helper.requires_auth_token
def head(
    id: str = typer.Argument(
        ...,
        help="Datasource id in the server.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Return the first few rows of the datasource."""

    from ..client import DataSource

    ds = DataSource(id=int(id))
    typer.echo(tabulate(ds.head(), headers="keys", tablefmt="plain", showindex=False))


# Internal Cell


@app.command()
@helper.requires_auth_token
def ls(
    offset: int = typer.Option(
        0,
        "--offset",
        "-o",
        help="The number of datasources to offset at the beginning. If **None**, then the default value **0** will be used.",
    ),
    limit: int = typer.Option(
        100,
        "--limit",
        "-l",
        help="The maximum number of datasources to return from the server. If **None**, then the default value **100** will be used.",
    ),
    disabled: bool = typer.Option(
        False,
        "--disabled",
        help="If set to **True**, then only the deleted datasources will be returned."
        "Else, the default value **False** will be used to return only the list"
        "of active datasources.",
    ),
    completed: bool = typer.Option(
        False,
        "--completed",
        help="If set to **True**, then only the datasources that are successfully downloaded"
        "to the server will be returned. Else, the default value **False** will be used to"
        "return all the datasources.",
    ),
    quiet: bool = typer.Option(
        False,
        "--quiet",
        "-q",
        help="Output only ids of datasource separated by space",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Return the list of datasources."""

    from ..client import DataSource

    dsx = DataSource.ls(
        offset=offset, limit=limit, disabled=disabled, completed=completed
    )
    df = DataSource.as_df(dsx)

    df["pulled_on"] = helper.humanize_date(df["pulled_on"])
    df["no_of_rows"] = helper.humanize_number(df["no_of_rows"])
    df["folder_size"] = helper.humanize_size(df["folder_size"])

    if quiet:
        ids = df["datasource_id"].astype(str).to_list()
        typer.echo("\n".join(ids))
    else:
        typer.echo(tabulate(df, headers="keys", tablefmt="plain", showindex=False))


# Internal Cell


@app.command()
@helper.requires_auth_token
def train(
    id: str = typer.Option(
        ...,
        "--datasource_id",
        "-id",
        help="Datasource id in the server.",
    ),
    client_column: str = typer.Option(
        ...,
        "--client_column",
        help="The column name that uniquely identifies the users/clients.",
    ),
    timestamp_column: Optional[str] = typer.Option(
        None,
        "--timestamp_column",
        help="The timestamp column indicating the time of an event. If not passed, then the default value **None** will be used. ",
    ),
    target_column: str = typer.Option(
        ...,
        "--target_column",
        help="Target column name that indicates the type of the event.",
    ),
    target: str = typer.Option(
        ...,
        "--target",
        help="Target event name to train and make predictions. You can pass the target event as a string or as a "
        "regular expression for predicting more than one event. For example, passing ***checkout** will "
        "train a model to predict any checkout event.",
    ),
    predict_after: str = typer.Option(
        ...,
        "--predict_after",
        help="Time delta in hours of the expected target event.",
    ),
    quiet: bool = typer.Option(
        False,
        "--quiet",
        "-q",
        help="Output model id only.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
):
    """Train a model against the datasource."""

    from ..client import DataSource
    from datetime import timedelta

    ds = DataSource(id=int(id))

    model = ds.train(
        client_column=client_column,
        target_column=target_column,
        target=target,
        predict_after=timedelta(hours=int(predict_after)),
        timestamp_column=timestamp_column,
    )

    if quiet:
        model.wait()
        typer.echo(f"{model.id}")
    else:
        typer.echo(f"Training started for Model ID: {model.id}")
        model.progress_bar()


# Internal Cell


@app.command()
@helper.requires_auth_token
def details(
    id: int = typer.Argument(
        ...,
        help="Datasource id in the server.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Return details of a datasource."""

    from ..client import DataSource

    ds = DataSource(id=int(id))
    df = ds.details()

    df["pulled_on"] = helper.humanize_date(df["pulled_on"])
    df["no_of_rows"] = helper.humanize_number(df["no_of_rows"])
    df["folder_size"] = helper.humanize_size(df["folder_size"])

    typer.echo(tabulate(df, headers="keys", tablefmt="plain", showindex=False))


# Internal Cell


@app.command()
@helper.requires_auth_token
def rm(
    id: int = typer.Argument(
        ...,
        help="Datasource id in the server.",
    ),
    quiet: bool = typer.Option(
        False,
        "--quiet",
        "-q",
        help="Output the deleted data id only.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Delete a datasource from the server."""

    from ..client import DataSource

    ds = DataSource(id=int(id))
    df = ds.delete()

    df["pulled_on"] = helper.humanize_date(df["pulled_on"])
    df["no_of_rows"] = helper.humanize_number(df["no_of_rows"])
    df["folder_size"] = helper.humanize_size(df["folder_size"])

    if quiet:
        typer.echo(df.iloc[0]["datasource_id"])
    else:
        typer.echo(tabulate(df, headers="keys", tablefmt="plain", showindex=False))


# Internal Cell


@app.command()
@helper.requires_auth_token
def tag(
    id: int = typer.Option(
        ...,
        "--datasource_id",
        "-id",
        help="Datasource id in the server.",
    ),
    name: str = typer.Option(
        ...,
        "--name",
        "-n",
        help="A string to tag the datasource.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Tag an existing datasource in server."""

    from ..client import DataSource

    ds = DataSource(id=int(id))
    df = ds.tag(name=name)

    df["pulled_on"] = helper.humanize_date(df["pulled_on"])
    df["no_of_rows"] = helper.humanize_number(df["no_of_rows"])
    df["folder_size"] = helper.humanize_size(df["folder_size"])

    typer.echo(tabulate(df, headers="keys", tablefmt="plain", showindex=False))
