#!/usr/bin/env python3
# vim: set ts=4 sts=4 sw=4 et ci nu ft=python:

# object-oriented wrapper around song objects
import codecs
import datetime
import hashlib
import io
import os
import re

# for type hinting
from typing import IO, Optional, Union

import jinja2
import markdown

# HTML processing, rendering and manipulation
import ukedown.udn
import yaml
from bs4 import BeautifulSoup as bs
from pychord import Chord
from ukedown import patterns

from .filters import custom_filters
from .utils import safe_filename

# installed directory is os.path.dirname(os.path.realpath(__file__))
# a slightly doctored version of the ukedown chord pattern, which separates
# '*' (and any other non-standard chord 'qualities' so we can still transpose
CHORD = r"\(([A-G][adgijmnsu0-9#b\/A-G]*)([*\+])?\)"
CRDPATT = re.compile(CHORD)


class Song(object):
    """
    a Song object represents a song with associated methods
    to generate output, summarise content etc
    songs are instantiated from ukedown files

    This wrapper is intended to make it simpler to construct a DB model
    for this content, plus to take all this code out of any automation
    scripts
    """

    def __init__(self, src: Union[IO, str], **kwargs):
        """
        construct our song object from a ukedown (markdown++) file
        Args:
            src can be one of the following
            src(str):        ukedown content read from a file.
                             This must be unicode (UTF-8)
            src(file handle): an open file handle (or equivalent object)
                             supporting 'read' (stringIO etc).
                             This should produce UTF-8 when read
                             (codecs.open is your friend)
            src(path):       path to a ukedown-formatted file to open and parse

        Kwargs:
            anything can be customised, most attributes/properties are
            auto-generated, but we sometimes need to override them
            Those listed below are commonly-used properties.
            These can also be parsed out of the songsheet itself
            using metadata markup

            title(str):      Song Title
            title_sort(str): Song title in sortable order
            artist(str):     artist name, as printed
            artist_sort:     sortable artist name, usually either
                             "Surname, Firstname" or "Band Name, The"
                             where appropriate.
            tags(list):      tags to apply to this song (tentative, tested etc)
            template(path):  the jinja2 template used to render this song.
                             can be overridden at the songbook level
            id(int):         page number, used as part of the "id" attribute on
                             headers

        """
        self._checksum = None
        self._load_time = datetime.datetime.now()
        self._mod_time = None
        self._index_entry = None
        self._id = 0

        if hasattr(src, "read"):
            # if we're operating on a filehandle
            # or another class that implements 'read'
            self._markup = src.read()
            if hasattr(src, "name"):
                self._filename = src.name
            else:
                self._filename = None
            self._fsize = len(src.read())
        elif os.path.exists(src):
            # did we pass a filename?
            # This is the most common use case
            self.__load(src)
            self._source = src
            self._filename = os.path.basename(src)
            self._fsize = os.path.getsize(src)
            #
        else:
            # presume we've been given content
            self._markup = src
            self._fsize = len(src)

        # does nothing yet
        self._filename = src
        self.__parse(markup=self._markup)
        # arbitrary metadata, some of which will have meaning
        self._meta = {}
        # tags are separate
        self._tags = set([])

        # update with any parameters...
        for key, val in kwargs.items():
            setattr(self, key, val)

        if self._filename is None:
            self._filename = ("{0.title}_-_{0.artist}.udn".format(self)).lower()

        if self._index_entry is None:
            self._index_entry = "{0.title} - {0.artist}".format(self)

        self.__checksum()

    def __unicode__(self):
        return self.songid

    def __str__(self):
        return self.songid

    def __repr__(self):
        return f"<Song: {self.songid}>"

    # other 'private' methods for use in __init__, mostly.

    def __load(self, sourcefile: str):
        """
        utlity function to handle loading from a file-like object

        sets:
            self._markup(str): text content, amy include metadata
            self._mod_time(datetime): last modified time, if any
            self.fsize(int): size of input in bytes.
        """
        try:
            with codecs.open(sourcefile, mode="r", encoding="utf-8") as src:
                self._markup = src.read()
                self._mod_time = datetime.datetime.fromtimestamp(
                    os.path.getmtime(sourcefile)
                )
                self.fsize = os.path.getsize(sourcefile)

        except (IOError, OSError) as E:
            print("Unable to open input file {0.filename} ({0.strerror}".format(E))
            self._markup = None

    def __checksum(self):
        """
        Generate sha256 checksum of loaded content (checking for changes)

        sets:
            self._checksum: sha256 hash of content
        """
        shasum = hashlib.sha256()
        shasum.update(self._markup.encode("utf-8"))
        self._checksum = shasum.hexdigest()

    def __extract_meta(self, markup: Optional[str] = None, leader: str = ";"):
        """
        parse out metadata from file,
        This MUST be done before passing to markdown
        There doesn't have to be any metadata - should work regardless

        Args:
            markup(str): content of file, which we will manipulate in place
            leader(str): leader character - only process lines that begin with this

        sets:
            self._markup(str): cleaned markdown/udn without metadata
            self._meta(dict): metadata (if any) extracted from markup
        """
        if markup is None:
            markup = self._markup
        metap = re.compile(r"^{}\s?(.*)".format(leader), re.I | re.U)
        metadata = []
        content = []

        for line in markup.splitlines():
            res = metap.match(line)
            if res is not None:
                metadata.append(res.group(1))
            else:
                content.append(line)
        self._markup = "\n".join(content)
        self._meta = yaml.safe_load("\n".join(metadata))

    def __parse(self, **kwargs):
        """
        parses ukedown to set attrs and properties
        processes metadata entries in file, converts markup content to HTML

        kwargs:
            properties to set on parsed object, usually passed in from __init__
            These override self._meta - so you can set them externally, add tags
            etc willy-nilly.

        sets:
            self._markup

        """
        # strip out any metadata entries from input
        self.__extract_meta(self._markup)

        # convert remaining markup to HTML
        self.__parse_markup()

        # extract chords and positions in text/markup
        self.__parse_chords()

    def __parse_markup(self):
        """
        Convert markup to HTML, set attributes
        sets:
            self.title:   title (parsed from first line)
            self.artist:  Artist (parsed from first line)
            self.content: HTML content.
        """
        # convert UDN to HTML via markdown + extensions.
        raw_html = markdown.markdown(
            self._markup, extensions=["markdown.extensions.nl2br", "ukedown.udn"]
        )

        # process HTML with BeautifulSoup to parse out headers etx
        soup = bs(raw_html, features="lxml")

        # extract our sole H1 tag, which should be the title - artist string
        hdr = soup.h1.extract()
        try:
            title, artist = [i.strip() for i in hdr.text.split("-", 1)]
        except ValueError:
            title = hdr.text.strip()
            artist = None

        # remove the header from our document
        hdr.decompose()

        # set core attributes
        self._title = title
        self._artist = artist

        # add processed body text (with headers etc converted)
        self.body = "".join([str(x) for x in soup.body.contents]).strip()

    def __parse_chords(self):
        """
        Extract the chords from markup, not HTML. This determines their position
        in the song and allows us to write code to transpose them.

        sets:
            self._chord_locations: nested list of chord, start position, end position
            self._chords: deduplicated chords list, in order of appearence.
        """
        # contains chord objects, plus their start and end positions in the text
        chord_locations = []
        # an ordered, deduped list of chords (to manage which diagrams we need)
        chordlist = []

        # walk over matched chords, convert them and record their locations
        for m in CRDPATT.finditer(self.markup):
            try:
                crd = Chord(m.groups()[0])
                tail = m.groups()[1]
                chord_locations.append([crd, m.end(), tail if tail is not None else ""])
                if crd not in chordlist:
                    chordlist.append(crd)
            except ValueError:
                # raised when this is not a recognised chord
                print(
                    f"Unable to parse chord {m.match} at position {m.start()} in song {self.filename}"
                )
                raise

        # set attributes so we can access these elsewhere
        self._chord_locations = chord_locations
        self._chords = chordlist

    def __get_render_env(self, templatedir: str = "") -> jinja2.Environment:
        """
        Initialises a jinja2 Environment for rendering songsheets

        This will load templates from a provided path (templatedir),
        or if this is not provided (or doesn't exist), from the
        'templates' directory in this package.

        """
        jinja_env = jinja2.Environment(
            loader=jinja2.ChoiceLoader(
                [
                    jinja2.FileSystemLoader(templatedir),
                    jinja2.PackageLoader("udn_songbook"),
                ]
            ),
            trim_blocks=True,
            lstrip_blocks=True,
            keep_trailing_newline=True,
            extensions=["jinja2.ext.do", "jinja2.ext.loopcontrols"],
        )

        # add our custom filters
        jinja_env.filters.update(custom_filters)

        return jinja_env

    def render(
        self,
        environment: Optional[jinja2.Environment] = None,
        template: str = "song.html.j2",
        **context,
    ) -> str:
        """
        Render HTML output using jinja templates.
        This defaults to using the templates packaged with `udn_songbook` but you
        can override this with the `templatedir` and `template` parameters

        KWargs:
            templatedir(str): override location for template files
            template(str): name of song template to look for
            context(dict): key=val pairs to add to template context

        """
        # There are prettier ways to do this but this is simple and readable
        # if we provide a jinja environment (e.g. from a parent songbook), use it
        if environment is None:
            environment = self.__get_render_env()

        tpl = environment.get_template(template)
        return tpl.render(songbook={}, song=self, **context)

    def pdf(self, stylesheet: str = "pdf.css"):
        """
        Generate a PDF songsheet from this song
        This will require weasyprint and a stylesheet

        Stylesheets are loaded from the udn_songbook installation dir
        by default, but you can provide a path to a stylesheet of your
        choice
        """
        pass

    def transpose(self, semitones: int):
        """
        shift all chords in the song by the given number of semitones
        and reparse content

        This will alter the following attributes:

        self._markup
        self._chords
        self._chord_locations


        """
        # take a copy to transpose, as the transposition is an in-place
        # alteration of chord objects
        tmkup = io.StringIO(self._markup)
        transposed = []
        for crd, end, tail in self._chord_locations:
            # change the chord in place
            crd.transpose(semitones)
            # read the section of the markup that contains it
            # and insert the new transposed version
            transposed.append(
                CRDPATT.sub(f"({crd.chord}{tail})", tmkup.read(end - tmkup.tell()))
            )

        # alter the markup in place
        self._markup = "".join(transposed)

        # convert back to HTML
        self.__parse_markup()

        # keep a record of our transposition
        self._meta["transposed"] = semitones

    def save(self, path: str = None):
        """
        Save an edited song back to disk. If path is None, will use the
        original filename (self.sourcefile)
        """

        # did we provide an output file?
        if path is not None:
            outdir, outfile = os.path.split(path)
        # if not, use the current filename, if it exists
        elif self.sourcefile is not None:
            outdir, outfile = os.path.split(self.sourcefile)
        else:
            # create a new filename usingtitle and artist
            outdir = os.curdir
            outfile = f"{self.title} - {self.artist}.udn"

        dest = os.path.join(outdir, safe_filename(outfile))

        try:
            with open(dest, "w") as output:
                output.write(self._markup)
                # stick the metadata at the bottom
                if self._meta is not None:
                    output.write("\n; # metadata\n")
                    for line in yaml.safe_dump(
                        self._meta, default_flow_style=False
                    ).splitlines():
                        output.write(f";{line}")
                self.sourcefile = dest
                print(f"saved song to {dest}")
        except (IOError, OSError) as E:
            # switch to logging at some point
            print(f"unable to save {E.filename} - {E.strerror}")

    # Property-based attribute settings - some are read-only in this interface

    @property
    def markup(self):
        return self._markup

    @markup.setter
    def markup(self, content):
        self._markup = content

    @property
    def filename(self):
        return self._filename

    @filename.setter
    def filename(self, path):
        self._filename = path

    @property
    def artist(self):
        return self._artist

    @artist.setter
    def artist(self, value):
        self._artist = value

    @property
    def title(self):
        return self._title

    @title.setter
    def title(self, value):
        self._title = value

    # no setter for chords, they're parsed from input
    @property
    def chords(self):
        return self._chords

    # tags are read-only too (ish)
    @property
    def tags(self):
        return self._tags

    @tags.setter
    def tags(self, taglist):
        self._tags = set(taglist)

    def tag(self, tag):
        if tag not in self.tags:
            self._tags.add(tag)

    def untag(self, tag):
        if tag in self._tags:
            self._tags.pop(tag)

    def clear_tags(self):
        # remoes ALL tags
        self._tags = set([])

    @property
    def checksum(self):
        return self._checksum

    @property
    def id(self):
        return self._id

    @id.setter
    def id(self, val: int):
        self._id = val

    @property
    def meta(self):
        return self._meta

    @meta.setter
    def meta(self, data, **kwargs):
        """
        Sets metadata by either updating the dict in place, or
        using kwargs to change single keys

        kwargs overrides everything else :)
        """
        # actually updates, not replaces
        try:
            self._meta.update(data)
            if len(kwargs):
                self._meta.update(kwargs)
        except TypeError:
            raise TypeError("data must be a dict")

    @property
    def size(self):
        return self._fsize

    @property
    def loaded(self):
        return "{0._load_time:%Y-%m-%d %H:%M:%S}".format(self)

    @property
    def modified(self):
        return "{0._mod_time:%Y-%m-%d %H:%M:%S}".format(self)

    @property
    def stat(self):
        return "size: {0.fsize}, loaded: {0.loaded}, modified {0.modified}".format(self)

    @property
    def songid(self):
        """
        The string representation in a songbook index
        """
        return self._index_entry

    @songid.setter
    def songid(self, data):
        try:
            self._index_entry = str(data)
        except TypeError:
            raise TypeError("Song IDs must be strings, or be convertible to strings")
