# Copyright 2021 Vincent Texier <vit@free.fr>
#
# This software is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This software is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
import sys
from collections import OrderedDict
from typing import Optional

from PyQt5.QtCore import QTimer
from PyQt5.QtWidgets import QApplication, QDialog, QLineEdit, QWidget
from substrateinterface import Keypair, KeypairType

from tikka.domains.application import Application
from tikka.domains.entities.constants import DATA_PATH, WALLETS_PASSWORD_LENGTH
from tikka.libs.secret import generate_alphabetic
from tikka.slots.pyqt.entities.constants import DEBOUNCE_TIME
from tikka.slots.pyqt.resources.gui.windows.account_import_rc import (
    Ui_AccountImportDialog,
)


class AccountImportWindow(QDialog, Ui_AccountImportDialog):
    """
    AccountImportWindow class
    """

    def __init__(self, application: Application, parent: Optional[QWidget] = None):
        """
        Init import account window

        :param application: Application instance
        :param parent: QWidget instance
        """
        super().__init__(parent=parent)
        self.setupUi(self)

        self.application = application
        self._ = self.application.translator.gettext

        # Mnemonic language selector translated
        mnemonic_language_selector = OrderedDict(
            [
                ("en", self._("English")),
                ("fr", self._("French")),
                ("zh-hans", self._("Chinese simplified")),
                ("zh-hant", self._("Chinese traditional")),
                ("it", self._("Italian")),
                ("ja", self._("Japanese")),
                ("ko", self._("Korean")),
                ("es", self._("Spanish")),
            ]
        )
        for language_code, language_name in mnemonic_language_selector.items():
            self.mnemonicLanguageComboBox.addItem(language_name, userData=language_code)
        self.mnemonicLanguageComboBox.setCurrentIndex(
            self.mnemonicLanguageComboBox.findData(
                self.application.config.get("language")[:2]
            )
        )

        # buttons
        self.buttonBox.button(self.buttonBox.Ok).setEnabled(False)

        # events
        self.mnemonicLineEdit.textChanged.connect(self._on_mnemonic_changed)
        self.showButton.clicked.connect(self.on_show_button_clicked)
        self.mnemonicLanguageComboBox.currentIndexChanged.connect(
            self._on_mnemonic_changed
        )
        self.derivationLineEdit.textChanged.connect(self._on_derivation_changed)
        self.passwordChangeButton.clicked.connect(self._generate_wallet_password)
        self.buttonBox.accepted.connect(self.on_accepted_button)
        self.buttonBox.rejected.connect(self.close)

        # debounce timers
        self.mnemonic_debounce_timer = QTimer()
        self.mnemonic_debounce_timer.timeout.connect(self._mnemonic_debounce_call)
        self.derivation_debounce_timer = QTimer()
        self.derivation_debounce_timer.timeout.connect(self._derivation_debounce_call)

        # fill form
        self.storedPasswordFrame.hide()
        self._generate_wallet_password()

    def _on_mnemonic_changed(self):
        """
        Triggered when mnemonic is changed

        :return:
        """
        if self.mnemonic_debounce_timer.isActive():
            self.mnemonic_debounce_timer.stop()
        self.mnemonic_debounce_timer.start(DEBOUNCE_TIME)

    def _mnemonic_debounce_call(self):
        """
        Debounce function triggered only after idle time when typing mnemonic

        :return:
        """
        # stop mnemonic_debounce_timer to avoid infinite loop
        if self.mnemonic_debounce_timer.isActive():
            self.mnemonic_debounce_timer.stop()

        self.errorLabel.setText("")

        if self.verify_user_entry() is not True:
            return

        try:
            address = self._generate_address()
        except Exception:
            self.errorLabel.setText(self._("Mnemonic or language not valid!"))
            self.buttonBox.button(self.buttonBox.Ok).setEnabled(False)
            return

        if self.address_already_exists(address):
            return

        derivation = self.derivationLineEdit.text().strip()
        if derivation == "":
            self.errorLabel.setText(
                self._("WARNING : it is not secure to store a root account private key")
            )

        self._update_wallet_password_fields()

        self.buttonBox.button(self.buttonBox.Ok).setEnabled(True)

    def on_show_button_clicked(self):
        """
        Triggered when user click on show button

        :return:
        """
        if self.mnemonicLineEdit.echoMode() == QLineEdit.Password:
            self.mnemonicLineEdit.setEchoMode(QLineEdit.Normal)
            self.storedpasswordLineEdit.setEchoMode(QLineEdit.Normal)
            self.showButton.setText(self._("Hide"))
        else:
            self.mnemonicLineEdit.setEchoMode(QLineEdit.Password)
            self.storedpasswordLineEdit.setEchoMode(QLineEdit.Password)
            self.showButton.setText(self._("Show"))

    def _on_derivation_changed(self):
        """
        Triggered when user enter text in derivationLineEdit

        :return:
        """
        if self.derivation_debounce_timer.isActive():
            self.derivation_debounce_timer.stop()
        self.derivation_debounce_timer.start(DEBOUNCE_TIME)

    def _derivation_debounce_call(self):
        """
        Debounce function triggered only after idle time when typing derivation

        :return:
        """
        # stop derivation_debounce_timer to avoid infinite loop
        if self.derivation_debounce_timer.isActive():
            self.derivation_debounce_timer.stop()

        self.errorLabel.setText("")
        derivation = self.derivationLineEdit.text().strip()

        try:
            address = self._generate_address()
        except Exception:
            self.errorLabel.setText(self._("Derivation is not valid!"))
            self.buttonBox.button(self.buttonBox.Ok).setEnabled(False)
            return

        if derivation == "":
            self.errorLabel.setText(
                self._("WARNING : it is not secure to store a root account private key")
            )

        if self.address_already_exists(address):
            return

        self._update_wallet_password_fields()

        self.buttonBox.button(self.buttonBox.Ok).setEnabled(True)

    def _update_wallet_password_fields(self):
        """
        Show stored password or new password depending on root account exists or not

        :return:
        """
        # if root account exists, hide the password field
        mnemonic = self.mnemonicLineEdit.text().strip()
        language_code = self.mnemonicLanguageComboBox.currentData()
        root_keypair = Keypair.create_from_mnemonic(
            mnemonic=mnemonic,
            language_code=language_code,
            crypto_type=KeypairType.SR25519,
            ss58_format=self.application.currencies.get_current().ss58_format,
        )
        if self.application.passwords.exists(root_keypair.ss58_address):
            stored_password = self.application.passwords.get_clear_password(
                root_keypair
            )
            self.storedpasswordLineEdit.setText(stored_password)
            self.storedPasswordFrame.show()
            self.passwordFrame.hide()
        else:
            self.storedPasswordFrame.hide()
            self.passwordFrame.show()

    def _generate_address(self) -> str:
        """
        Generate, display and return address

        :return:
        """
        mnemonic = self.mnemonicLineEdit.text().strip()
        language_code = self.mnemonicLanguageComboBox.currentData()
        derivation = self.derivationLineEdit.text().strip()
        address = Keypair.create_from_uri(
            suri=mnemonic + derivation,
            ss58_format=self.application.currencies.get_current().ss58_format,
            crypto_type=KeypairType.SR25519,
            language_code=language_code,
        ).ss58_address

        self.addressValueLabel.setText(address)
        return address

    def address_already_exists(self, address: str) -> bool:
        """
        Modify form if address already exists

        :param address: Address to check
        :return:
        """
        if self.application.accounts.get_by_address(address) is not None:
            self.errorLabel.setText(self._("Account already exists!"))
            self.buttonBox.button(self.buttonBox.Ok).setEnabled(False)
            return True

        return False

    def verify_user_entry(self) -> bool:
        """
        Verify user entry, return True if OK, False otherwise

        :return:
        """
        # verify mnemonic
        language_code = self.mnemonicLanguageComboBox.currentData()
        if not Keypair.validate_mnemonic(
            self.mnemonicLineEdit.text().strip(), language_code
        ):
            self.errorLabel.setText(self._("Mnemonic or language not valid!"))
            self.buttonBox.button(self.buttonBox.Ok).setEnabled(False)
            return False

        return True

    def _generate_wallet_password(self):
        """
        Generate new password for wallet encryption in UI

        :return:
        """
        self.passwordLineEdit.setText(generate_alphabetic(WALLETS_PASSWORD_LENGTH))

    def on_accepted_button(self):
        """
        Triggered when user click on ok button

        :return:
        """
        # user inputs
        mnemonic = self.mnemonicLineEdit.text().strip()
        name = self.nameLineEdit.text().strip()
        language_code = self.mnemonicLanguageComboBox.currentData()

        # generated inputs
        password = self.passwordLineEdit.text()
        derivation_ = self.derivationLineEdit.text().strip()

        self.application.accounts.create_new_account(
            mnemonic, language_code, derivation_, name, password
        )


if __name__ == "__main__":
    qapp = QApplication(sys.argv)
    application_ = Application(DATA_PATH)
    AccountImportWindow(application_).exec_()
