"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const rule_1 = require("../lib/rule");
class SomeTarget {
    constructor(input) {
        this.input = input;
    }
    bind() {
        return { id: 'T1', arn: 'ARN1', input: this.input };
    }
}
module.exports = {
    'json template': {
        'can just be a JSON object'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromObject({ SomeObject: 'withAValue' })));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: "{\"SomeObject\":\"withAValue\"}"
                    }
                ]
            }));
            test.done();
        },
        'can use joined JSON containing refs in JSON object'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromObject({
                data: lib_1.EventField.fromPath('$'),
                stackName: cdk.Aws.STACK_NAME,
            })));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        InputTransformer: {
                            InputPathsMap: {
                                f1: '$'
                            },
                            InputTemplate: {
                                'Fn::Join': [
                                    '',
                                    [
                                        '{"data":<f1>,"stackName":"',
                                        { Ref: 'AWS::StackName' },
                                        '"}'
                                    ]
                                ]
                            },
                        }
                    }
                ]
            }));
            test.done();
        },
        'can use token'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            const user = new aws_iam_1.User(stack, 'User');
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromObject({ userArn: user.userArn })));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: {
                            'Fn::Join': [
                                '',
                                [
                                    '{\"userArn\":\"',
                                    {
                                        'Fn::GetAtt': [
                                            'User00B015A1',
                                            'Arn'
                                        ]
                                    },
                                    '\"}'
                                ]
                            ]
                        }
                    }
                ]
            }));
            test.done();
        },
    },
    'text templates': {
        'strings with newlines are serialized to a newline-delimited list of JSON strings'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromMultilineText('I have\nmultiple lines')));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: "\"I have\"\n\"multiple lines\""
                    }
                ]
            }));
            test.done();
        },
        'escaped newlines are not interpreted as newlines'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromMultilineText('this is not\\na real newline'))),
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                    Targets: [
                        {
                            Input: "\"this is not\\\\na real newline\""
                        }
                    ]
                }));
            test.done();
        },
        'can use Tokens in text templates'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            const world = cdk.Lazy.stringValue({ produce: () => 'world' });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromText(`hello ${world}`)));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: "\"hello world\""
                    }
                ]
            }));
            test.done();
        }
    },
};
//# sourceMappingURL=data:application/json;base64,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