import os
import glob
import obspy
from obspy import UTCDateTime,read
from obspy.io.sac.sactrace import SACTrace
import shutil
import re
import matplotlib.pyplot as plt
from cuhk_seis.basic_utils import read_sac_ref_time,spherical_dist,get_st
import numpy as np
from tqdm import tqdm

def cut_eve_wf(phs_file,sta_file,src_root="day_data",tar_root="eve_wf",tb = -20,te = 60):
    """
    cut event waveform from continous waveform based on phase file generated by 
    HYPOINVERSE

    Parameters:
    phs_file: phase file, which is the output from HYPOINVERSE
    sta_file: station file in format: net sta lon lat ele
    src_root: source path for continous waveform
    tar_root: target path of eve based waveform
    tb: waveform begin time in relative to the event occurence time
    te: waveform end time in relative to the event ouccurence time
    """
    if not os.path.exists(tar_root):
        os.mkdir(tar_root)
    base_path = os.path.abspath("./")
    phs_cont = []
    with open(phs_file,"r") as f1:
        for line in f1:
            phs_cont.append(line.rstrip())
    f1.close()
    sta_dict = {}
    with open(sta_file,'r') as f2:
        for line in f2:
            line = line.rstrip()
            net,sta,_lon,_lat,_ele = re.split(" +",line)[:5]
            sta_dict[sta] = [float(_lon),float(_lat),int(_ele)]
    f2.close()
    event_count = 0
    for line in tqdm(phs_cont):
        f_para = line[0:2]
        if re.match("\d+",f_para): 
            event_count += 1
            _yr=line[0:4];_mo=line[4:6];_day=line[6:8]
            _hr=line[8:10];_min=line[10:12];_seconds=line[12:14]+"."+line[14:16]+"0"
            evla=float(line[16:18])+(float(line[19:21])+float(line[21:23])*0.01)/60
            evlo=float(line[23:26])+(float(line[27:29])+float(line[29:31])*0.01)/60
            evdp=float(line[32:36])/100; evid=str(int(line[136:146])).zfill(10)
           # _,no,year,month,day,o_time,ab_sec,res,evla,evlo,evdp,mag,mag_res,np,ns,nt,sta_gap=re.split(" +",line)
            #e_hr,e_min,e_seconds=re.split(":",o_time)
            e_sec = float(_seconds)
            e_time = UTCDateTime(_yr+"-"+_mo+"-"+_day+"T"+_hr+":"+_min+":"+"00")+e_sec
            year = e_time.year; month = e_time.month; day = e_time.day
            e_hr = e_time.hour;e_min = e_time.minute;e_sec = e_time.second; e_msec = int(e_time.microsecond/10000)
            jday = UTCDateTime(year,month,day).julday
            str_day = str(year)+str(month).zfill(2)+str(day).zfill(2)
            str_time = str(e_hr).zfill(2)+str(e_min).zfill(2)+str(e_sec).zfill(2)+str(e_msec).zfill(2)
            eve = str_day + str_time
            #create day folder
            if os.path.exists(os.path.join(tar_root,str_day)):
                pass
            else:
                os.makedirs(os.path.join(tar_root,str_day))
            if os.path.exists(os.path.join(tar_root,str_day,eve)):
                shutil.rmtree(os.path.join(tar_root,str_day,eve))
            os.makedirs(os.path.join(tar_root,str_day,eve))
            processed_sta_record = []

        if re.match("[A-Z]+",f_para) and f_para != "  ":
            net = line[5:7]
            sta = re.split(" +",line[0:5])[0]
            if line[14]==" ":
                #if sec or msec is 0, it will be "  " in out.arc file
                str_sec = line[42:44]; str_sec_m = line[44:46]
                if str_sec == "  ":
                    str_sec = "00"
                if str_sec_m == "  ":
                    str_sec_m = "00"
                p_type="S"
                phs_time = UTCDateTime(line[17:21]+"-"+line[21:23]+"-"+line[23:25]+"T"+\
                                   line[25:27]+":"+line[27:29]+":"+"00")+(int(str_sec)+int(str_sec_m)*0.01)
            else:
                p_type="P"
                #if sec or msec is 0, it will be "  " in out.arc file
                str_sec = line[30:32]; str_sec_m = line[32:34]
                if str_sec == "  ":
                    str_sec = "00"
                if str_sec_m == "  ":
                    str_sec_m = "00"
                phs_time = UTCDateTime(line[17:21]+"-"+line[21:23]+"-"+line[23:25]+"T"+\
                                   line[25:27]+":"+line[27:29]+":"+"00")+(int(str_sec)+int(str_sec_m)*0.01)
            ref_sec = phs_time - e_time
            #_,net,sta,p_type,ab_sec,ref_sec,amp,res,weight,azimuth=re.split(" +",line)
            starttime = e_time + tb
            endtime = e_time + te
            src_folder=os.path.join(src_root,sta)
            if sta not in processed_sta_record:
                processed_sta_record.append(sta)
                st = get_st(net,sta,starttime,endtime,src_folder)
                if len(st)==0:
                    continue
                for trace in st:
                    chn = trace.stats.channel
                    out_file = os.path.join(tar_root,str_day,eve,f"{sta}.{chn}")
                    trace.write(out_file,format="SAC")
                    sac = SACTrace.read(out_file,headonly=True)
                    sac.lovrok = 1
                    sac.evlo = evlo
                    sac.evla = evla
                    sac.stlo = sta_dict[sta][0]
                    sac.stla = sta_dict[sta][1]
                    sac.stel = sta_dict[sta][2]
                    sac.nzyear = year
                    sac.nzjday = jday
                    sac.nzhour = e_hr
                    sac.nzmin = e_min; sac.nzsec = int(_seconds[:2])
                    sac.nzmsec = int(_seconds[3:6])
                    sac.b = sac.b+tb
                    sac.o = 0
                    sac.write(out_file,headonly=True)
            tar_path = os.path.join(tar_root,str_day,eve)
            sac_files = glob.glob(os.path.join(tar_path,sta+"*"))
            if p_type == "P":
                for sac_file in sac_files:
                    sac = SACTrace.read(sac_file,headonly=True)
                    sac.a = ref_sec
                    sac.write(sac_file,headonly=True)
            else:
                for sac_file in sac_files:
                    sac = SACTrace.read(sac_file,headonly=True)
                    sac.t0 = ref_sec
                    sac.write(sac_file,headonly=True)  

def wf_dist_plot(st,length=10,color=None,label_sta=True,out_format="PNG",scaling_factor=2):
    '''    
    Description
    ------------
    Plot event waveform by distance. The start time is event origin time.
    Data should be in sac format. The output is a saved file with the title
    of reference time.

    Parameters
    -----------
                st: obspy Stream object
            length: The time window is defined by length in seconds.
             color: The usage of color is the same as matplotlib.pyplot. 
                    Using default color if not defined.
         label_sta: Whether label station name on the plot.
        out_format: "png","jpg","pdf"..., The same as matplotlib.pyplot.savefig
    scaling_factor: The waveform are normalized, increase scaling_facotr to
                    make the waveform plot more obvious

    Below data information needed:
    |   P arrival: tr.stats.sac.a
    |   S arrival: tr.stats.sac.t0
    |        evla: tr.stats.sac.evla
    |        evlo: tr.stats.sac.evlo
    |        stla: tr.stats.sac.stla
    |        stlo: tr.stats.sac.stlo
    '''
    st.detrend("linear")
    st.detrend("constant")
    try:
        e_mag = st[0].stats.sac.mag
    except:
        e_mag = -9
    starttime=st[0].stats.starttime
    endtime =st[0].stats.endtime
    #Reference time shoule be the same for all traces.
    sac_ref_time = read_sac_ref_time(st[0]) #In UTCDateTime format
    o_value = st[0].stats.sac.o 
    event_time = sac_ref_time + o_value #event origin time
    st.trim(starttime = event_time, endtime = event_time + length)
    if len(st) == 0:
        print("Error: Nothing to plot!")
    #Inititae parameters
    for tr in st:
        evla = tr.stats.sac.evla
        evlo = tr.stats.sac.evlo
        stla = tr.stats.sac.stla
        stlo = tr.stats.sac.stlo
        #It is recommend to set tr.stats.distance in meters by osbpy guideline
        tr.stats.distance = spherical_dist(evlo,evla,stlo,stla)*111*1000
    #Get mininumtime, maximum time, and max distance
    min_time = st[0].stats.starttime
    max_time = st[0].stats.endtime
    max_dist = st[0].stats.distance/1000 #in km
    for tr in st[1:]:
        if tr.stats.starttime < min_time:
            min_time = tr.stats.starttime
        if tr.stats.endtime > max_time:
            max_time = tr.stats.endtime
        if tr.stats.distance/1000 > max_dist:
            max_dist = tr.stats.distance/1000
    sampling_rate = st[0].stats.sampling_rate

    #Initiate plot parameters
    plt.figure(figsize = (8,10))
    plt.xlim(0,max_time-min_time)
    plt.ylim(0,max_dist+3)
    plt.xticks(size=12)
    plt.yticks(size=12)
    plt.xlabel("Time (s)",fontdict={'size':16})
    plt.ylabel("Distance (km)",fontdict={'size':16})
    #Plot trace by trace
    for tr in st:
        sta = tr.stats.station
        tr_ref_time = read_sac_ref_time(tr) 
        tr_o_value = tr.stats.sac.o
        event_time = tr_ref_time + tr_o_value
        x_start = event_time - min_time
        dist = tr.stats.distance/1000
        #Normalize the event
        disp_data = tr.data/(max(tr.data) - min(tr.data))
        disp_data = disp_data*scaling_factor
        if color == None:
            plt.plot(np.arange(0,len(tr.data))/sampling_rate+x_start,
                    disp_data+dist,
                    linewidth = 0.5)
        else:
            plt.plot(np.arange(0,len(tr.data))/sampling_rate+x_start,
                    disp_data+dist,
                    color = color,
                    linewidth = 0.5)
        if label_sta:
            plt.text(0.1,dist+0.2,sta,fontsize=12)
        #Plot P arrival if available
        try:
            a = tr.stats.sac.a
            rela_a = tr_ref_time + a - min_time
            gap = 0.5*max_dist/25
            plt.plot([rela_a,rela_a],[dist-gap,dist+gap],color='b',linewidth=2)
        except:
            pass
        #Plot S arrival if available
        try:
            t0 = tr.stats.sac.t0
            rela_t0 = tr_ref_time + t0 - min_time
            gap = 0.5*max_dist/25
            plt.plot([rela_t0,rela_t0],[dist-gap,dist+gap],color='r',linewidth=2)
        except:
            pass
        if e_mag != -9:
            plt.title(str(tr_ref_time)+f"_M{e_mag}",fontdict={'size':18})
        else:
            plt.title(str(tr_ref_time),fontdict={'size':18})
    plt.savefig(f"{sac_ref_time.strftime('%Y%m%d%H%M%S')}.png",format=out_format)
    plt.close()            
