import * as lib from 'aws-cdk-lib';
import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as ecs from 'aws-cdk-lib/aws-ecs';
import { ILogGroup } from 'aws-cdk-lib/aws-logs';
import { Construct } from 'constructs';
export interface LowCostECSProps extends lib.StackProps {
    /**
     * Domain name of the hosted zone.
     */
    readonly hostedZoneDomain: string;
    /**
     * Email for expiration emails to register to your let's encrypt account.
     *
     * @link https://letsencrypt.org/docs/expiration-emails/
     *
     * Also registered as a subscriber of the sns topic, notified on certbot task failure.
     * Subscription confirmation email would be sent on stack creation.
     *
     * @link https://docs.aws.amazon.com/sns/latest/dg/sns-email-notifications.html
     */
    readonly email: string;
    /**
     * Domain names for A records to elastic ip of ECS host instance.
     *
     * @default - [ props.hostedZone.zoneName ]
     */
    readonly recordDomainNames?: string[];
    /**
     * Vpc of the ECS host instance and cluster.
     *
     * @default - Creates vpc with only public subnets and no NAT gateways.
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Security group of the ECS host instance
     *
     * @default - Creates security group with allowAllOutbound and ingress rule (ipv4, ipv6) => (tcp 80, 443).
     */
    readonly securityGroup?: ec2.SecurityGroup;
    /**
     * Instance type of the ECS host instance.
     *
     * @default - t2.micro
     */
    readonly hostInstanceType?: string;
    /**
     * The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request.
     * Host instance asg would use spot instances if hostInstanceSpotPrice is set.
     *
     * @link https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ecs.AddCapacityOptions.html#spotprice
     * @default - undefined
     */
    readonly hostInstanceSpotPrice?: string;
    /**
     * Log group of the certbot task and the aws-cli task.
     *
     * @default - Creates default cdk log group
     */
    readonly logGroup?: ILogGroup;
    /**
     * Docker image tag of certbot/dns-route53 to create certificates.
     *
     * @link https://hub.docker.com/r/certbot/dns-route53/tags
     * @default - v1.29.0
     */
    readonly certbotDockerTag?: string;
    /**
     * Certbot task schedule interval in days to renew the certificate.
     *
     * @default - 60
     */
    readonly certbotScheduleInterval?: number;
    /**
     * Docker image tag of amazon/aws-cli.
     * This image is used to associate elastic ip on host instance startup, and run certbot cfn on ecs container startup.
     *
     * @default - latest
     */
    readonly awsCliDockerTag?: string;
    /**
     * Enable container insights or not
     *
     * @default - undefined (container insights disabled)
     */
    readonly containerInsights?: boolean;
    /**
     * Removal policy for the file system and log group (if using default).
     *
     * @default - RemovalPolicy.DESTROY
     */
    readonly removalPolicy?: lib.RemovalPolicy;
    /**
     * Task definition for the server ecs task.
     *
     * @default - Nginx server task definition defined in sampleServerTask()
     */
    readonly serverTaskDefinition?: ecs.Ec2TaskDefinition;
}
export declare class LowCostECS extends lib.Stack {
    constructor(scope: Construct, id: string, props: LowCostECSProps);
    private sampleSeverTask;
}
