"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepFunctionInvokeAction = exports.StateMachineInput = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * Represents the input for the StateMachine.
 */
class StateMachineInput {
    constructor(input, inputArtifact, inputType) {
        this.input = input;
        this.inputArtifact = inputArtifact;
        this.inputType = inputType;
    }
    /**
     * When the input type is FilePath, input artifact and
     * filepath must be specified.
     */
    static filePath(inputFile) {
        return new StateMachineInput(inputFile.location, inputFile.artifact, 'FilePath');
    }
    /**
     * When the input type is Literal, input value is passed
     * directly to the state machine input.
     */
    static literal(object) {
        return new StateMachineInput(JSON.stringify(object), undefined, 'Literal');
    }
}
exports.StateMachineInput = StateMachineInput;
/**
 * StepFunctionInvokeAction that is provided by an AWS CodePipeline.
 */
class StepFunctionInvokeAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.stateMachine,
            category: codepipeline.ActionCategory.INVOKE,
            provider: 'StepFunctions',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 1,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs: (props.stateMachineInput && props.stateMachineInput.inputArtifact) ? [props.stateMachineInput.inputArtifact] : [],
            outputs: (props.output) ? [props.output] : [],
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        var _a, _b, _c, _d, _e;
        // allow pipeline to invoke this step function
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:StartExecution', 'states:DescribeStateMachine'],
            resources: [this.props.stateMachine.stateMachineArn],
        }));
        // allow state machine executions to be inspected
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:DescribeExecution'],
            resources: [cdk.Stack.of(this.props.stateMachine).formatArn({
                    service: 'states',
                    resource: 'execution',
                    resourceName: `${this.props.stateMachine.stateMachineArn}:${(_a = this.props.executionNamePrefix) !== null && _a !== void 0 ? _a : ''}*`,
                    sep: ':',
                })],
        }));
        // allow the Role access to the Bucket, if there are any inputs/outputs
        if (((_b = this.actionProperties.inputs) !== null && _b !== void 0 ? _b : []).length > 0) {
            options.bucket.grantRead(options.role);
        }
        if (((_c = this.actionProperties.outputs) !== null && _c !== void 0 ? _c : []).length > 0) {
            options.bucket.grantWrite(options.role);
        }
        return {
            configuration: {
                StateMachineArn: this.props.stateMachine.stateMachineArn,
                Input: (_d = this.props.stateMachineInput) === null || _d === void 0 ? void 0 : _d.input,
                InputType: (_e = this.props.stateMachineInput) === null || _e === void 0 ? void 0 : _e.inputType,
                ExecutionNamePrefix: this.props.executionNamePrefix,
            },
        };
    }
}
exports.StepFunctionInvokeAction = StepFunctionInvokeAction;
//# sourceMappingURL=data:application/json;base64,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