"use strict";
const assert_1 = require("@aws-cdk/assert");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
function stackIncludingLambdaInvokeCodePipeline(props) {
    const stack = new core_1.Stack();
    new codepipeline.Pipeline(stack, 'Pipeline', {
        stages: [
            {
                stageName: 'Source',
                actions: [
                    new cpactions.GitHubSourceAction({
                        actionName: 'GitHub',
                        output: props.lambdaInput || new codepipeline.Artifact(),
                        oauthToken: core_1.SecretValue.plainText('secret'),
                        owner: 'awslabs',
                        repo: 'aws-cdk',
                    }),
                ],
            },
            {
                stageName: 'Invoke',
                actions: [
                    new cpactions.LambdaInvokeAction({
                        actionName: 'Lambda',
                        lambda: new lambda.Function(stack, 'Lambda', {
                            code: lambda.Code.fromCfnParameters(),
                            handler: 'index.handler',
                            runtime: lambda.Runtime.NODEJS_10_X,
                        }),
                        userParameters: props.userParams,
                        inputs: props.lambdaInput ? [props.lambdaInput] : undefined,
                        outputs: props.lambdaOutput ? [props.lambdaOutput] : undefined,
                    }),
                ],
            },
        ],
    });
    return stack;
}
module.exports = {
    'Lambda invoke Action': {
        'properly serializes the object passed in userParameters'(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                userParams: {
                    key: 1234,
                },
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'UserParameters': '{"key":1234}',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'properly resolves any Tokens passed in userParameters'(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                userParams: {
                    key: core_1.Lazy.stringValue({ produce: () => core_1.Aws.REGION }),
                },
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'UserParameters': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                '{"key":"',
                                                {
                                                    'Ref': 'AWS::Region',
                                                },
                                                '"}',
                                            ],
                                        ],
                                    },
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'properly resolves any stringified Tokens passed in userParameters'(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                userParams: {
                    key: core_1.Token.asString(null),
                },
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'UserParameters': '{"key":null}',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        "assigns the Action's Role with read permissions to the Bucket if it has only inputs"(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                lambdaInput: new codepipeline.Artifact(),
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': 'lambda:ListFunctions',
                            'Resource': '*',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': 'lambda:InvokeFunction',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                'kms:Decrypt',
                                'kms:DescribeKey',
                            ],
                            'Effect': 'Allow',
                        },
                    ],
                },
            }));
            test.done();
        },
        "assigns the Action's Role with write permissions to the Bucket if it has only outputs"(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                lambdaOutput: new codepipeline.Artifact(),
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': 'lambda:ListFunctions',
                            'Resource': '*',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': 'lambda:InvokeFunction',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                's3:DeleteObject*',
                                's3:PutObject*',
                                's3:Abort*',
                            ],
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                'kms:Encrypt',
                                'kms:ReEncrypt*',
                                'kms:GenerateDataKey*',
                            ],
                            'Effect': 'Allow',
                        },
                    ],
                },
            }));
            test.done();
        },
        "assigns the Action's Role with read-write permissions to the Bucket if it has both inputs and outputs"(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                lambdaInput: new codepipeline.Artifact(),
                lambdaOutput: new codepipeline.Artifact(),
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': 'lambda:ListFunctions',
                            'Resource': '*',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': 'lambda:InvokeFunction',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                'kms:Decrypt',
                                'kms:DescribeKey',
                            ],
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                's3:DeleteObject*',
                                's3:PutObject*',
                                's3:Abort*',
                            ],
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                'kms:Encrypt',
                                'kms:ReEncrypt*',
                                'kms:GenerateDataKey*',
                            ],
                            'Effect': 'Allow',
                        },
                    ],
                },
            }));
            test.done();
        },
        'exposes variables for other actions to consume'(test) {
            const stack = new core_1.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const lambdaInvokeAction = new cpactions.LambdaInvokeAction({
                actionName: 'LambdaInvoke',
                lambda: lambda.Function.fromFunctionArn(stack, 'Func', 'arn:aws:lambda:us-east-1:123456789012:function:some-func'),
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3_Source',
                                bucket: s3.Bucket.fromBucketName(stack, 'Bucket', 'bucket'),
                                bucketKey: 'key',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Invoke',
                        actions: [
                            lambdaInvokeAction,
                            new cpactions.ManualApprovalAction({
                                actionName: 'Approve',
                                additionalInformation: lambdaInvokeAction.variable('SomeVar'),
                                notificationTopic: sns.Topic.fromTopicArn(stack, 'Topic', 'arn:aws:sns:us-east-1:123456789012:mytopic'),
                                runOrder: 2,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Invoke',
                        'Actions': [
                            {
                                'Name': 'LambdaInvoke',
                                'Namespace': 'Invoke_LambdaInvoke_NS',
                            },
                            {
                                'Name': 'Approve',
                                'Configuration': {
                                    'CustomData': '#{Invoke_LambdaInvoke_NS.SomeVar}',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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